/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PutFileCommand.java,v 1.17 2008/03/20 18:49:50 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.execution.core.internal.resources.CoreResourceBundle;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;
import org.eclipse.osgi.util.NLS;

/**
 * The put file command is a file server command that is used to transfer files
 * from the client to the server. The main components of state that make up the
 * put file command are local identifiers of the files to transfer, remote
 * identifiers of the files to ensure existence on the server-side, options to
 * tweak the put file behavior and progress monitor to monitor the progress as
 * well as cancel the transfer. An advantage to the put file command is that it
 * supports the transfer of multiple files in one pass, using the same socket
 * and connection for all the files transferred. A single file transfer is also
 * possible, basically 1 to many file identifiers can be passed in to identity
 * the local and remote files to act upon.
 * 
 * @author Scott E. Schneider
 */
class PutFileCommand extends AbstractFileTransferCommand implements IPutFileCommand {

	/**
	 * The client side personality of the put file command, locates the physical
	 * files, opens them and transfers them to server
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractFileTransferCommand.Client {

		/**
		 * Creates the client side personality for the put file command
		 * 
		 * @param channel
		 *            the channel to communicate with the server
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate client
			super.execute();

			// Retrieve local and remote identifiers
			FileIdentifierList localIdentifiers = PutFileCommand.this.localIdentifiers;
			FileIdentifierList remoteIdentifiers = PutFileCommand.this.remoteIdentifiers;

			// Send number of files that will be transferred
			this.send(localIdentifiers.size());

			// Send all remote identifier names
			this.send(remoteIdentifiers.getArray());

			// Send all the files to the server one at a time but continuous
			for (Iterator identifiers = localIdentifiers.iterator(); identifiers.hasNext();) {
				String fileName = (String) identifiers.next();
				File localFile = new File(fileName);
				if (localFile.isFile()) {
					this.send(localFile);
				}
			}

		}

	}

	/**
	 * The server side personality of the put file command
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractFileTransferCommand.Server {

		/**
		 * Constructs the server side state personality for the put file command
		 * 
		 * @param channel
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate server
			super.execute();

			// Receive number of files that will be transferred
			this.receiveInt();

			// Receive the remote identifiers from the client
			FileIdentifierList remoteIdentifiers = FileIdentifierList.create(this.receiveStrings());
			PutFileCommand.this.remoteIdentifiers = remoteIdentifiers;

			// Receive all the files, one at a time but continuous
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers.hasNext();) {
				String fileName = (String) identifiers.next();
				File remoteFile = new File(fileName);
				boolean readyToTransfer = true;
				if (remoteFile.isAbsolute()) {
					File remoteDirectory = new File(remoteFile.getParent());
					if (!remoteDirectory.exists()) {
						readyToTransfer = remoteDirectory.mkdirs();
					}
					if (readyToTransfer) {
						remoteFile.createNewFile();
						this.receive(remoteFile);
					}
					FileSystemServices.println(NLS.bind(CoreResourceBundle.PutFileCommand_IDENTIFIED_RECEIVED_, remoteFile.getAbsolutePath()), this);
				}
			}

		}

	}

	/**
	 * Constructs a put file command from a socket channel, this is the
	 * constructor that the file server uses
	 * 
	 * @param channel
	 *            the channel to communicate with the client
	 */
	public PutFileCommand(ISocketChannel channel) {
		super(PutFileCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Constructs a put file command from a socket channel to the server, and
	 * the command state that defines the exact purpose of this command
	 * 
	 * @param channel
	 *            the channel to coomunicate with the server
	 * @param cookie
	 *            the cookie that identifies this particular command instance
	 * @param localIdentifiers
	 *            the local identifiers identifying the local files to transfer
	 * @param remoteIdentifiers
	 *            the remote identifiers identifying the remote files to copy
	 *            into
	 * @param options
	 *            the options that tweak this command's behavior
	 * @param monitor
	 *            the progress monitor to use
	 */
	PutFileCommand(ISocketChannel channel, Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options, IProgressMonitor monitor) {
		super(PutFileCommand.class, cookie, localIdentifiers, remoteIdentifiers, options, monitor);
		this.setState(new Client(channel));
	}

}