/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ChannelCommunicator.java,v 1.7 2008/12/12 22:21:52 jcayne Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.communicator;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;

import com.ibm.icu.util.Calendar;

/**
 * The channel communicator implementation implements communication send and
 * receive methods that abstract away the details of the lower-level underlying
 * channel and buffer handling.
 * 
 * @author Scott E. Schneider
 */
class ChannelCommunicator implements IChannelCommunicator {

	/**
	 * The channel receiving communication traffic
	 */
	private ReadableByteChannel readableChannel;

	/**
	 * The channel sending communication traffic
	 */
	private WritableByteChannel writableChannel;

	/**
	 * Constructs a channel communicator given the specified readable and
	 * writable byte channels, constructor visible to this package and used by
	 * the factory to create channel communicators instances.
	 * 
	 * @param readableChannel
	 *            the channel to listen for and receive communications on
	 * @param writableChannel
	 *            the channel to send communications on
	 * 
	 * @see ChannelCommunicatorFactory#create(ReadableByteChannel,
	 *      WritableByteChannel)
	 */
	ChannelCommunicator(ReadableByteChannel readableChannel,
			WritableByteChannel writableChannel) {
		this.readableChannel = readableChannel;
		this.writableChannel = writableChannel;
	}

	/**
	 * Receive file from sender
	 * 
	 * @param file
	 *            file identifying the file to store the incoming file in
	 */
	public void receive(File file) throws IOException {
		long length = 0;
		
		// The length of time to wait for the server, before we assume that it's a file not found error 
		final int TIMEOUT_ON_FILE_TRANSFER = 6000;
		
		try {
			length = this.receiveLongForFile(TIMEOUT_ON_FILE_TRANSFER);
		} catch (IOException e) {
			/*
			 * The remote server did not send us the length of the file, so we
			 * infer the the file does not exist.
			 */
			throw new FileNotFoundException();
		}
		try {
			FileOutputStream outputStream = new FileOutputStream(file);
			FileChannel outputChannel = outputStream.getChannel();
			outputChannel.transferFrom(this.readableChannel, 0, length);
			outputChannel.close();
			outputStream.close();
		} finally {
			long receivedLength = -1;
			if (file.exists()) {
				receivedLength = file.length();
			}
			this.send(receivedLength);
			if (receivedLength != length) {
				throw new IOException();
				// TODO: externalize a string to explain the failure.
			}
		}
	}
	
	/**
	 * Receive a fixed length long from the sender. 
	 * @param timeoutInMsecs The number of milliseconds this method should wait for a long before throwing an IOException.
	 * @return the long received
	 * @throws IOException 
	*/
	private long receiveLongForFile(int timeoutInMsecs) throws IOException {
		SocketChannel sc = 	null;
		boolean channelSupportsUnblocking = false;		
		
		// This method was created for bug 220484 - jwest
		try {

			// Allocate a buffer to receive long
			ByteBuffer buffer = ByteBuffer.allocate(8);
			
			long startTimeInMsec = 0;
			
			// If the channel supports unblocking the input...
			if(this.readableChannel instanceof org.eclipse.hyades.internal.execution.core.file.socket.SocketChannel) {

				org.eclipse.hyades.internal.execution.core.file.socket.SocketChannel sctptp = 
					(org.eclipse.hyades.internal.execution.core.file.socket.SocketChannel)this.readableChannel;
				
				if(sctptp.getReadable() instanceof SocketChannel) {
				
					sc = (SocketChannel)sctptp.getReadable();
					
					channelSupportsUnblocking = true;
					
					sc.configureBlocking(false);
				
					startTimeInMsec = Calendar.getInstance().getTimeInMillis();
				}
			}

			
						
			// Read contents into buffer
			while (buffer.hasRemaining()) {
				this.readableChannel.read(buffer);
				
				if(channelSupportsUnblocking) {
					Thread.sleep(10);
					
					// If we have timed out, throw an exception
					if(Calendar.getInstance().getTimeInMillis() - startTimeInMsec > timeoutInMsecs) {
						
							sc.configureBlocking(true);
						
						throw new IOException();
					}
				}
			}
			
			if(channelSupportsUnblocking) {
				sc.configureBlocking(true);
			}

			// Flip buffer to process
			buffer.flip();

			// Return long received
			return buffer.getLong();

		} catch (BufferUnderflowException e) {
			if(channelSupportsUnblocking && sc != null && !sc.isBlocking()) {
				sc.configureBlocking(true);
			}
			throw new IOException();
		} catch (InterruptedException ie) {
			if(channelSupportsUnblocking && sc != null && !sc.isBlocking()) {
				sc.configureBlocking(true);
			}
			return -1;
		}
	}
	

	/**
	 * Receive a fixed number of bytes from the server. It is expected that the
	 * caller has determined the number of bytes to be read, either through
	 * advance knowledge or through their own protocol.
	 * 
	 * @param length
	 *            the number of bytes sent, this must match the number of bytes
	 *            sent by the client
	 * @return the byte array received
	 * @throws IOException
	 */
	public byte[] receiveBytes(int length) throws IOException {

		try {

			// Allocate a buffer to receive the string
			ByteBuffer buffer = ByteBuffer.allocate(length);

			// Read contents into buffer
			while (buffer.hasRemaining()) {
				this.readableChannel.read(buffer);
			}

			// Flip buffer prepare for reading
			buffer.flip();

			// Return the array of received bytes
			byte[] receivedBytes = new byte[length];
			buffer.get(receivedBytes);
			return receivedBytes;

		} catch (BufferUnderflowException e) {
			throw new IOException();
		}

	}

	/**
	 * Receives a fixed length int from the sender
	 * 
	 * @return the int received
	 * @throws IOException
	 */
	public int receiveInt() throws IOException {
		try {

			// Allocate a buffer to receive integer
			ByteBuffer buffer = ByteBuffer.allocate(4);

			// Read contents into buffer
			while (buffer.hasRemaining()) {
				this.readableChannel.read(buffer);
			}

			// Flip buffer prepare
			buffer.flip();
			return buffer.getInt();

		} catch (BufferUnderflowException e) {
			throw new IOException();
		}
	}

	/**
	 * Receive a fixed length long from the sender
	 * 
	 * @return the long received
	 * @throws IOException
	 */
	public long receiveLong() throws IOException {
		try {

			// Allocate a buffer to receive long
			ByteBuffer buffer = ByteBuffer.allocate(8);

			// Read contents into buffer
			while (buffer.hasRemaining()) {
				this.readableChannel.read(buffer);
			}

			// Flip buffer to process
			buffer.flip();

			// Return long received
			return buffer.getLong();

		} catch (BufferUnderflowException e) {
			throw new IOException();
		}
	}

	/**
	 * Receive variable length data from the server, using a simple protocol
	 * where the length in bytes is first sent, followed by the data itself,
	 * using the UTF-8 encoding.
	 * 
	 * If the length is negative one, then no data will follow up the length and
	 * it indicates empty string or null.
	 * 
	 * @return the string received, null if no data was sent
	 * @throws IOException
	 */
	public String receiveString() throws IOException {
		int length = this.receiveInt();
		if (length >= 0) {
			byte[] bytes = this.receiveBytes(length);
			return new String(bytes, "UTF-8"); //$NON-NLS-1$
		} else {
			return null;
		}
	}

	/**
	 * Receive strings being sent from the client
	 * 
	 * @return the received string array
	 * @throws IOException
	 */
	public String[] receiveStrings() throws IOException {
		int length = this.receiveInt();
		String[] strings = new String[length];
		for (int i = 0; i < length; i++) {
			strings[i] = this.receiveString();
		}
		return strings;
	}

	/**
	 * Send a fixed length message of type byte[]. This is the quickest way to
	 * send and have a message received because it does not involve more than
	 * one write to the server. However, it requires advance knowledge about the
	 * number of bytes being sent and received (on both the client and server
	 * sides.)
	 * 
	 * Note that the client must know the number of bytes to consume on the
	 * receiving end of this call
	 * 
	 * @param bytes
	 *            fixed length message to send
	 * @throws IOException
	 */
	public void send(byte[] bytes) throws IOException {
		int length = bytes.length;
		ByteBuffer buffer = ByteBuffer.allocate(length);
		buffer.put(bytes);
		buffer.flip();
		this.writableChannel.write(buffer);
	}

	/**
	 * Send file to receiver
	 * 
	 * @param file
	 *            file identifying the file to send
	 * @param length
	 *            the file length (or the amount to send)
	 */
	public void send(File file) throws IOException, FileNotFoundException {
		FileInputStream inputStream = new FileInputStream(file);
		FileChannel inputChannel = inputStream.getChannel();
		long length = file.length();
		this.send(length);
		inputChannel.transferTo(0, length, this.writableChannel);
		inputChannel.close();
		inputStream.close();
		long receivedLength = this.receiveLong();
		if (receivedLength != length) {
			throw new IOException();
			// TODO: externalize string to describe the failure
		}
	}

	/**
	 * Send a fixed length int to the channel
	 * 
	 * @param data
	 *            the data to send
	 * @throws IOException
	 */
	public void send(int data) throws IOException {
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.putInt(data);
		buffer.flip();
		this.writableChannel.write(buffer);
	}

	/**
	 * Send a fixed length long to the channel
	 * 
	 * @param data
	 *            the data to send
	 * @throws IOException
	 */
	public void send(long data) throws IOException {
		ByteBuffer buffer = ByteBuffer.allocate(8);
		buffer.putLong(data);
		buffer.flip();
		this.writableChannel.write(buffer);
	}

	/**
	 * Send variable length data to the server, this takes a few more server
	 * calls but is more flexible and potential takes less space since only the
	 * data needed is sent (just enough and not more) -- follows a simple
	 * protocol where the length of the data is first sent to the server is a
	 * fixed length message and then the data follows. The implementation is
	 * just a call to send the length as a long and then a call to send the
	 * string as a fixed length.
	 * 
	 * @param data
	 *            the variable length message to send
	 * @param tagLength
	 *            specifies if the length sent you be a negative integer with
	 *            the fact that the length is negative conveying additional
	 *            information that is interpreted correctly by the receiving
	 *            side (for example, if tagLength is set to true and the data is
	 *            length 10, length will be sent as -10 instead of 10, the
	 *            receiving end can take the absolute value but also have an
	 *            extra bit of information to act upon)
	 * @throws IOException
	 */
	public void send(String data) throws IOException {
		this.send(data, false);
	}

	/**
	 * Send variable length data to the server, this takes a few more server
	 * calls but is more flexible and potential takes less space since only the
	 * data needed is sent (just enough and not more) -- follows a simple
	 * protocol where the length of the data is first sent to the server is a
	 * fixed length message and then the data follows. The implementation is
	 * just a call to send the length as a long and then a call to send the
	 * string as a fixed length.
	 * 
	 * @param data
	 *            the variable length message to send, null data indicates no
	 *            data to be sent except -1 which indicates no data
	 * @param tagLength
	 *            specifies if the length sent you be a negative integer with
	 *            the fact that the length is negative conveying additional
	 *            information that is interpreted correctly by the receiving
	 *            side (for example, if tagLength is set to true and the data is
	 *            length 10, length will be sent as -10 instead of 10, the
	 *            receiving end can take the absolute value but also have an
	 *            extra bit of information to act upon), if using tagLength the
	 *            data must be greather than one in length or it will be
	 *            misinterpreted as no data (-1 as defined in the data param
	 *            docs)
	 * @throws IOException
	 */
	public void send(String data, boolean tagLength) throws IOException {
		if (data != null) {
			byte[] bytes = data.getBytes("UTF-8"); //$NON-NLS-1$
			int length = bytes.length;
			if (tagLength) {
				length *= -1;
			}
			this.send(length);
			this.send(bytes);
		} else {

			// Negative one length indicates no data
			this.send(-1);

		}
	}

	/**
	 * Sends an array of variable string with any optimizations on the transfer
	 * put in this reusable method
	 * 
	 * @param data
	 *            the strings to send
	 * @throws IOException
	 */
	public void send(String[] data) throws IOException {
		this.send(data.length);
		for (int i = 0; i < data.length; i++) {
			this.send(data[i]);
		}
	}
}