/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IChannelCommunicator.java,v 1.1 2005/09/21 20:51:32 sschneid Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.communicator;

import java.io.File;
import java.io.IOException;

/**
 * The channel communicator interface contains communication send and receive
 * methods that abstract away the details of the lower-level underlying channel
 * and buffer handling.
 * 
 * @author Scott E. Schneider
 */
public interface IChannelCommunicator {

	/**
	 * Receive a file from the readable channel
	 * 
	 * @param file
	 *            the file to hold the contents of the received file
	 * @throws IOException
	 */
	public void receive(File file) throws IOException;

	/**
	 * Receive the specified length of bytes of the readable channel
	 * 
	 * @param length
	 *            the number of bytes to be read from the readable channel
	 *            abstracted by the channel communicator
	 * @return the byte array received
	 * @throws IOException
	 */
	public byte[] receiveBytes(int length) throws IOException;

	/**
	 * Receive the next integer from the readable channel
	 * 
	 * @return the integer received
	 * @throws IOException
	 */
	public int receiveInt() throws IOException;

	/**
	 * Receive the next long from the readable channel
	 * 
	 * @return the long received
	 * @throws IOException
	 */
	public long receiveLong() throws IOException;

	/**
	 * Receive the next string from the readable channel
	 * 
	 * @return a string from the readable channel
	 * @throws IOException
	 */
	public String receiveString() throws IOException;

	/**
	 * Receive a set of strings that have been sent over the readable channel
	 * 
	 * @return a set of strings, one to many, as sent by the sender on the
	 *         readable channel
	 * @throws IOException
	 */
	public String[] receiveStrings() throws IOException;

	/**
	 * Send an array of bytes over the writable channel
	 * 
	 * @param bytes
	 *            the bytes to send over the writable channel
	 * @throws IOException
	 */
	public void send(byte[] bytes) throws IOException;

	/**
	 * Sends a file over the writable channel
	 * 
	 * @param file
	 *            a file to send over the writable channel
	 * @throws IOException
	 */
	public void send(File file) throws IOException;

	/**
	 * Send an integer over the writable channel
	 * 
	 * @param data
	 *            the data to send over the writable channel
	 * @throws IOException
	 */
	public void send(int data) throws IOException;

	/**
	 * Send a long over the writable channel
	 * 
	 * @param data
	 *            the long to send over the communication channel
	 * @throws IOException
	 */
	public void send(long data) throws IOException;

	/**
	 * Send a string over the writable channel to be received by a readable
	 * channel
	 * 
	 * @param data
	 *            the string to send over the writable channel
	 * @throws IOException
	 */
	public void send(String data) throws IOException;

	/**
	 * Send a string over the writable channel, tagging the length of the string
	 * sent to indicate something "special" that is interepted by the receiver,
	 * the length of the string is sent but it is sent with a negative length,
	 * this negative length can be used to trigger some additional information
	 * on the receiver end.
	 * 
	 * @param data
	 *            the string to send along the writable channel
	 * @param tagLength
	 *            mark or tag the length with a negative indicator
	 * @throws IOException
	 */
	public void send(String data, boolean tagLength) throws IOException;

	/**
	 * Send an array of strings along the writable channel to a readable channel
	 * on the other end
	 * 
	 * @param data
	 *            the strings to send
	 * @throws IOException
	 */
	public void send(String[] data) throws IOException;

}