/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractFileServerCommand.java,v 1.7 2008/04/09 16:05:00 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.IOException;
import java.util.Observable;
import java.util.Observer;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.execution.core.loader.ScopedChannelClassLoader;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.communicator.ChannelCommunicatorFactory;
import org.eclipse.hyades.internal.execution.core.file.communicator.IChannelCommunicator;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * The abstract base class for all file server commands, there are other more
 * specific abstract classes that further extend this one, a concrete file
 * server command will likely extend those downstream subclasses and not this
 * base class directly.
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractFileServerCommand implements IFileServerCommand {

	/**
	 * The pre-defined base class states are defined here, one for the client
	 * and one for the server. File server commands are expected to subclass one
	 * of these states when defining the state command that gets passed into the
	 * super constructor calls. It is not necessary to extend these subclasses,
	 * but without these, explicit connection and handling must be put all in
	 * the state command passed in, no connection code is inherited fo free. The
	 * client takes care of the system code necessary to establish the command
	 * on the server to communicate with, commands should definitely call the
	 * superclass constructor so they don't have to write this code themselves,
	 * also if the protocol or headers changes, calling the super will help
	 * existing commands be maintained going further
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Client extends State {

		/**
		 * Creates a client state personality
		 * 
		 * @param channel
		 *            the server channel to communicate with
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			/*
			 * Send the command identity to the server-side to load command,
			 * last argument set to true to indicate alternate classloading is
			 * required by the server
			 */
			this.communicator.send(AbstractFileServerCommand.this.identity.getName(), true);

			/*
			 * Assume the role of scoped channel classloader provider, delegates
			 * classloading consumption requests to scoped channel classloader --
			 * the provider provides the requested class resources from the
			 * machine and local classpath this code is running in. The maximum
			 * class length is passed in to be 64k, if classes are found beyond
			 * this range in practice, this number should be bumped up.
			 */
			ScopedChannelClassLoader.Provider provider = new ScopedChannelClassLoader.Provider(
					AbstractFileServerCommand.this.context, Thread.currentThread().getContextClassLoader(), this.channel,
					this.channel, 64);

			// Add listeners for informative and debug purposes
			Observer observer = new Observer() {
				public void update(Observable observable, Object argument) {
					FileSystemServices.println("Class consumed by server (from providing client) " + argument, this);  // $NON-NLS-1$
				}
			};
			provider.addObserver(observer);

			// Blocks until all resources consumed as required
			provider.provide();

			// Clean up to avoid memory leaks
			provider.deleteObserver(observer);

		}

	}

	/**
	 * The server state base class is fairly straightforward and reserved for
	 * future use, subclasses should be sure to call super.execute() at the top
	 * of their execute methods in order to get future protocol and header
	 * changes without having to change command code
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Server extends State {

		/**
		 * The server side personality base class
		 * 
		 * @param client
		 *            the client channel to communicate with
		 */
		Server(ISocketChannel client) {
			super(client);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {
		}

	}

	/**
	 * The base class for the client and server state personalities
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class State implements IFileServerCommand {

		/**
		 * The channel to communicate with the client or server depending on how
		 * the specializing state objects interpret it
		 */
		final ISocketChannel channel;

		/**
		 * The communicator used to convey the file system services protocol
		 * over the wire (abstract low-level communication interactions)
		 */
		final IChannelCommunicator communicator;

		/**
		 * Constructs a state object, not intended to be directly subclassed by
		 * concrete commands, although it is not prevented
		 * 
		 * @param channel
		 *            the channel
		 */
		State(ISocketChannel channel) {
			this.channel = channel;
			this.communicator = ChannelCommunicatorFactory.getInstance().create(this.channel, this.channel);
		}

		/**
		 * Dispose any resources held by the state, see outer class
		 * documentation for more information
		 * 
		 * @see AbstractFileServerCommand#dispose()
		 */
		public void dispose() {

			// If the channel exists and is not open, close it
			try {
				if (this.channel != null) {
					if (this.channel.isOpen()) {
						this.channel.close();
					}
				}
			} catch (IOException e) {
				// No need to handle this exception
			}

		}

		/**
		 * Determines if the channel is open, if the channel is null or the
		 * channel is not open, false is returned
		 * 
		 * @return indicates if the connection is connected or not
		 */
		boolean isOpen() {
			return (this.channel != null ? this.channel.isOpen() : false);
		}

	}

	/**
	 * The context this file server command is running under, basically the
	 * unique identity of the file server command factory
	 */
	private String context;

	/**
	 * Stores the cookie for this particular command instance pairing (between
	 * client and server)
	 */
	private Cookie cookie;

	/**
	 * Reference cookie until cookies are implemented, placeholder
	 */
	{
		if (this.cookie == null);
	}

	/**
	 * The identity of this particular command, used to match up the client and
	 * server implementations at the code-level, whereas the cookie matches up
	 * identities at the instance-level across VMs between the client and the
	 * server
	 */
	private Class identity;

	/**
	 * The progress monitor to use for client-side status inquiry and canceling
	 * of task at hand, this is not supported for all commands and is optionally
	 * supported as commands need it
	 */
	private IProgressMonitor monitor;

	/**
	 * Store the options for this particular command, it is assumed that any
	 * options will be transmitted and initialized into the server matching
	 * instance as needed
	 */
	private Option[] options;

	/**
	 * Using the state pattern, basically a command once in a different state
	 * takes a new personality and behaves differently like it changes its class
	 * at run-time, there are two conceptual states, client and server. If the
	 * command is on the client it behaves one way (such as opening a connection
	 * to the server and sending commands) whereas if the command is in the
	 * server state it accepts an already connected socket to the client where a
	 * response will then be calculated and sent. In one state the command is
	 * the client and in the other state the command is the server. This allows
	 * the code to be encapsulated per given command yet allows the code to be
	 * written specifically for client or server based on the state that is
	 * passed in to the constructor.
	 */
	private IFileServerCommand state;

	/**
	 * A convenience constructor for creating a server purposed command
	 * 
	 * @param context
	 *            the context this command is executing under
	 * @param identity
	 *            the command identity
	 */
	AbstractFileServerCommand(String context, Class identity) {
		this(context, identity, Cookie.NONE, Option.NONE, new NullProgressMonitor());
	}

	/**
	 * Creates an abstract file server command given the identity, cookie,
	 * options and progress monitor
	 * 
	 * @param context
	 *            The context identifier that identifies the context this
	 *            command was created under, a command is created under a shared
	 *            file server command factory
	 * @param identity
	 *            the identity of the command being constructed
	 * @param cookie
	 *            the cookie identifying this instance of the command (same
	 *            cookie across VMs between client and server)
	 * @param options
	 *            the options to tweak the behavior of this command with
	 * @param monitor
	 *            the progress monitor to use
	 */
	AbstractFileServerCommand(String context, Class identity, Cookie cookie, Option[] options, IProgressMonitor monitor) {
		this.context = context;
		this.identity = identity;
		this.cookie = cookie;
		this.options = options;
		this.monitor = monitor;
		FileSystemServices.println("An abstract file server command is created with identity " + this.identity
				+ "\r\n\t\tand context " + this.context + "\r\n\t\tand cookie " + cookie + "\r\n\t\tand options "
				+ options + "\r\n\t\tand progress monitor " + this.monitor, this); // $NON-NLS-1$
	}

	/**
	 * Dispose the command, will release any resources such as network
	 * connections, if not called, resources and connections will build up and
	 * likely cause un- desirable results
	 */
	public void dispose() {
		FileSystemServices.println("File server command with identity " + identity + " is now disposed", this); // $NON-NLS-1$
		this.state.dispose();
	}

	/**
	 * Execute the command, instance acts polymorphically based on the way the
	 * command was constructed (either client-side or server-side)
	 */
	public final void execute() throws IOException {
		if (this.state != null) {
			FileSystemServices.println("Executing file server command with identity " + identity, this); // $NON-NLS-1$
			this.state.execute();
		} else {
			System.out.println("There is no behavior defined for the current state of this command: " + this); // $NON-NLS-1$
		}
	}

	/**
	 * The options set for this particular command
	 * 
	 * @return the options, could be null or Option.NONE if not set yet
	 */
	Option[] getOptions() {
		return this.options;
	}

	/**
	 * Sets the state of this particular command, there are two conceptual
	 * states, server and client -- a command acts with a client personality or
	 * server personality based on the state set here, there might be other uses
	 * for this with other personality states, but right now client and server
	 * make sense (for example, maybe there is a debug server state and a debug
	 * client state in the future)
	 * 
	 * @param state
	 *            the state to set this command as, before this state is set the
	 *            command does nothing upon execute
	 */
	void setState(IFileServerCommand state) {
		this.state = state;
	}

}