/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html $Id:
 * AbstractFileTransferCommand.java,v 1.1 2005/05/03 14:32:54 sschneid Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * Super class to file transfer commands, handles common state and behavior for
 * all file transfer commands such as the get file and put file commands -- the
 * local identifiers and remote identifiers are stored for use during the
 * transfer. Upon use, if the identifiers are detected to be relative instead of
 * absolute paths the relative paths will be replaced with the absolute paths.
 * 
 * A client can transfer a file with the remote identifier not known for
 * example, and upon transfer (after the server receives the file and finds a
 * suitable resting place for it) the remote identifiers will be replaced with
 * the appropriate absolute path. Clients will then have access to the absolute
 * path used in the transfer.
 * 
 * @author Scott E. Schneider
 */
abstract class AbstractFileTransferCommand extends AbstractFileServerCommand implements IFileTransferCommand {

	/**
	 * The client side personality common to all file transfer commands
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Client extends AbstractFileServerCommand.Client {

		/**
		 * Constructs a client given the socket channel to use
		 * 
		 * @param channel
		 *            the channel to communicate with the server
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

	}

	/**
	 * The server side personality common to all file transfer commands
	 * 
	 * @author Scott E. Schneider
	 */
	abstract class Server extends AbstractFileServerCommand.Server {

		/**
		 * Constructs a server given the socket channel to use
		 * 
		 * @param channel
		 *            the channel to communicate with the client
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

	}

	/**
	 * The local and remote identifiers for this file transfer command,
	 * asserting that the length of both lists will be equal supporting one to
	 * many files for bulk transfers.
	 * 
	 * The file identifiers can be relative or absolute paths that are specified
	 * to either the local or remote systems (for example, a reomte identifier
	 * with an absolute path is valid on the remote system not the local system)
	 */
	FileIdentifierList localIdentifiers;

	/**
	 * The remote identifier list, the size of this list must match the local
	 * identifier size
	 */
	FileIdentifierList remoteIdentifiers;

	/**
	 * Constructs an abstract file transfer command given the the identity of
	 * the concrete leaf-level command.
	 * 
	 * @param context
	 *            the context this command is executing within 
	 * @param identity
	 *            the actual concrete command being used
	 */
	AbstractFileTransferCommand(String context, Class identity) {
		super(context, identity);
	}

	/**
	 * Constructs a file transfer command given the command identity, the
	 * cookie, the local identifiers, the remote identifiers, options and an a
	 * progress monitor.
	 * 
	 * @param context
	 *            the context this command is executing withinV
	 * @param identity
	 *            the identity of the concrete command being used
	 * @param cookie
	 *            the cookie identifying the exact instance of the command
	 * @param localIdentifiers
	 *            the identifiers of the local files to operate on, the list is
	 *            filtered based on local identifier validation
	 * @param remoteIdentifiers
	 *            the identifiers of the remote files to operate on
	 * @param options
	 *            the options for this command
	 * @param monitor
	 *            the progress monitor for this command's execution
	 */
	AbstractFileTransferCommand(String context, Class identity, Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options, IProgressMonitor monitor) {
		super(context, identity, cookie, options, monitor);

		// Store local and remote identifiers as instance variables
		this.localIdentifiers = localIdentifiers;
		this.remoteIdentifiers = remoteIdentifiers;

	}

}