/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html $Id:
 * DeleteFileCommand.java,v 1.1 2005/05/03 14:32:54 sschneid Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * The delete directory command, an abstract file manipulation command subclass
 * used for delete directories on the server -- the delete directories command
 * supports bulk operation removing a group of files in one operation
 * 
 * @author Scott E. Schneider
 * @author jcanches
 */
class DeleteDirectoryCommand extends AbstractFileManipulationCommand implements
		IDeleteDirectoryCommand {

	/**
	 * The client side personality of the delete command
	 * 
	 * @author Scott E. Schneider
	 * @author jcanches
	 */
	private class Client extends AbstractFileManipulationCommand.Client {

		/**
		 * Creates the client side personality for the delete command
		 * 
		 * @param channel
		 *            the channel to communicate with the server
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate client
			super.execute();

			// Send the number of files to be removed
			FileIdentifierList remoteIdentifiers = DeleteDirectoryCommand.this.remoteIdentifiers;
			this.communicator.send(remoteIdentifiers.size());

			// Send all remote identifier names
			this.communicator.send(remoteIdentifiers.getArray());

		}
	}

	/**
	 * The server side personality of the delete command
	 * 
	 * @author Scott E. Schneider
	 * @author jcanches
	 */
	private class Server extends AbstractFileManipulationCommand.Server {

		/**
		 * Constructs the server side state personality for the delete command
		 * 
		 * @param channel
		 *            the channel to communicate with the client
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Delete a directory as indicated via the specified directory argument,
		 * recursively (if existent) deletes directories under the specified
		 * directory including files contained within
		 * 
		 * @param directory
		 *            the directory to delete
		 */
		private void deleteDirectory(File directory) {
			File[] files = directory.listFiles();
			for (int i = 0; i < files.length; i++) {
				if (files[i].isFile()) {
					files[i].delete();
				} else {
					deleteDirectory(files[i]);
				}
			}
			directory.delete();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate server
			super.execute();

			// Receive number of files that will be deleted
			this.communicator.receiveInt();

			// Receive the remote identifiers from the client
			FileIdentifierList remoteIdentifiers = FileIdentifierList
					.create(this.communicator.receiveStrings());
			DeleteDirectoryCommand.this.remoteIdentifiers = remoteIdentifiers;

			// Remove all the files on the server side
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers
					.hasNext();) {
				String fileName = (String) identifiers.next();
				File remoteDir = new File(fileName);
				if (remoteDir.exists() && !remoteDir.isFile()) {
					deleteDirectory(remoteDir);
					FileSystemServices.println("Absolute file identifier "
							+ remoteDir.getAbsolutePath()
							+ " identified and directory removed on server",
							this); // $NON-NLS-1$
				} else {
					FileSystemServices
							.println(
									"Absolute file identifier "
											+ remoteDir.getAbsolutePath()
											+ " not identified and therefore ignoring delete request",
									this); // $NON-NLS-1$
				}
			}

		}
	}

	/**
	 * Constructs a delete directory command given the specified cookie and
	 * monitor instances
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param cookie
	 *            the unique identity identifying the cached server side
	 *            operands of this operation
	 * @param monitor
	 *            the progress monitor used for progress and cancel support
	 */
	DeleteDirectoryCommand(String context, Cookie cookie,
			IProgressMonitor monitor) {
		super(context, DeleteDirectoryCommand.class, cookie, monitor);
	}

	/**
	 * Constructs a delete file command given the specified channel
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the client on
	 */
	public DeleteDirectoryCommand(String context, ISocketChannel channel) {
		super(context, DeleteDirectoryCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Constructs a delete file command given the specified channel, remote
	 * identifier operands and progress monitor
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the server on
	 * @param remoteIdentifiers
	 *            the remote identifier operands
	 * @param monitor
	 *            the progress monitor used for progress nad cancel control
	 */
	DeleteDirectoryCommand(String context, ISocketChannel channel,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) {
		super(context, DeleteDirectoryCommand.class, remoteIdentifiers, monitor);
		this.setState(new Client(channel));

	}

}
