/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html $Id:
 * DeleteFileCommand.java,v 1.1 2005/05/03 14:32:54 sschneid Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * The delete file command, an abstract file manipulation command subclass used
 * for delete files on the server -- the delete file command supports bulk
 * operation removing a group of files in one operation
 * 
 * @author Scott E. Schneider
 */
class DeleteFileCommand extends AbstractFileManipulationCommand implements IDeleteFileCommand {

	/**
	 * The client side personality of the delete command
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractFileManipulationCommand.Client {

		/**
		 * Creates the client side personality for the delete command
		 * 
		 * @param channel
		 *            the channel to communicate with the server
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate client
			super.execute();

			// Send the number of files to be removed
			FileIdentifierList remoteIdentifiers = DeleteFileCommand.this.remoteIdentifiers;
			this.communicator.send(remoteIdentifiers.size());

			// Send all remote identifier names
			this.communicator.send(remoteIdentifiers.getArray());

		}
	}

	/**
	 * The server side personality of the delete command
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractFileManipulationCommand.Server {

		/**
		 * Constructs the server side state personality for the delete command
		 * 
		 * @param channel
		 *            the channel to communicate with the client
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate server
			super.execute();

			// Receive number of files that will be deleted
			this.communicator.receiveInt();

			// Receive the remote identifiers from the client
			FileIdentifierList remoteIdentifiers = FileIdentifierList.create(this.communicator.receiveStrings());
			DeleteFileCommand.this.remoteIdentifiers = remoteIdentifiers;

			// Remove all the files on the server side
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers.hasNext();) {
				String fileName = (String) identifiers.next();
				File remoteFile = new File(fileName);
				if (remoteFile.exists()) {
					remoteFile.delete();
					FileSystemServices.println("Absolute file identifier " + remoteFile.getAbsolutePath()
							+ " identified and file removed on server", this); // $NON-NLS-1$
				} else {
					FileSystemServices.println("Absolute file identifier " + remoteFile.getAbsolutePath()
							+ " not identified and therefore ignoring delete request", this); // $NON-NLS-1$
				}
			}

		}
	}

	/**
	 * Constructs a delete file command given the specified cookie and monitor
	 * instances
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param cookie
	 *            the unique identity identifying the cached server side
	 *            operands of this operation
	 * @param monitor
	 *            the progress monitor used for progress and cancel support
	 */
	DeleteFileCommand(String context, Cookie cookie, IProgressMonitor monitor) {
		super(context, DeleteFileCommand.class, cookie, monitor);
	}

	/**
	 * Constructs a delete file command given the specified channel
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the client on
	 */
	public DeleteFileCommand(String context, ISocketChannel channel) {
		super(context, DeleteFileCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Constructs a delete file command given the specified channel, remote
	 * identifier operands and progress monitor
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the server on
	 * @param remoteIdentifiers
	 *            the remote identifier operands
	 * @param monitor
	 *            the progress monitor used for progress nad cancel control
	 */
	DeleteFileCommand(String context, ISocketChannel channel, FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) {
		super(context, DeleteFileCommand.class, remoteIdentifiers, monitor);
		this.setState(new Client(channel));

	}

}