/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: DetermineServerReachCommand.java,v 1.5 2008/05/20 16:55:30 jkubasta Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;

import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * A useful command that determines if the file server can connect to a remote
 * host on the given IP address and port number. Currently used to determine if
 * a firewall is blocking a particular port in going going to the workbench
 * machine from the agent controller machine.
 * 
 * @author Scott E. Schneider
 */
class DetermineServerReachCommand extends AbstractServerInterrogationCommand
		implements IDetermineServerReachCommand {

	/**
	 * The client state personality of this command, refer to the server state
	 * class personality for the ying and the yang constrast between the server
	 * and client states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractServerInterrogationCommand.Client {

		/**
		 * The client state making this command behave as the client
		 * 
		 * @param channel
		 *            the server channel (the channel to communicate with the
		 *            server)
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Execute the query server status command from within the client state,
		 * sends and then receives for comparison the bounced back message to
		 * verify it worked.
		 */
		public void execute() throws IOException {

			// Invoke super class executes to send command
			super.execute();

			// Send message to server using simple communication abstractions
			this.communicator.send(DetermineServerReachCommand.this.host);
			this.communicator.send(DetermineServerReachCommand.this.port);
			DetermineServerReachCommand.this.isHostReachable = this
					.receiveBoolean();

		}

		/**
		 * Receive a boolean through the channel communicator, this should be
		 * pushed in to the channel communicator in the future but as not to
		 * break compatibility it is left here for now
		 * 
		 * @return true or false based on the value received
		 * @throws IOException
		 */
		private boolean receiveBoolean() throws IOException {
			return this.communicator.receiveBytes(1)[0] == DetermineServerReachCommand.TRUE[0];
		}
	}

	/**
	 * The server state personality of this command, refer to the client state
	 * class personality for the ying and the yang contrast between the client
	 * and server states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractServerInterrogationCommand.Server {

		/**
		 * The server state making this command behave as the server (the
		 * channel to communicate with the client)
		 * 
		 * @param channel
		 *            the client channel
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Attempt successful connection to specified host and port combination,
		 * updates the internal instance state to reflect the attempt result
		 */
		private void determineServerReach() {

			// If host is non-null and port is greater than 0
			if (DetermineServerReachCommand.this.host != null
					&& DetermineServerReachCommand.this.port >= 0) {

				try {
					
					// This was changed from NIO to IO, because NIO doesn't support IPv6 at the moment.
					Socket s = new Socket(DetermineServerReachCommand.this.host,
							DetermineServerReachCommand.this.port);
					
					DetermineServerReachCommand.this.isHostReachable = true;
					s.close();  /* bugzilla_152351: Socket should be closed on a successful connection */
				} catch (Throwable t) {

					/*
					 * If a throwable occurs then one way or another the
					 * connection cannot be made, either with a connection
					 * refused with invalid port or firewall blocking or invalid
					 * name lookup resolution for the address -- will keep
					 * default is host reachable value, this exception is
					 * expected for the not reachable cases, return gracefully
					 */

				}

			}

		}

		/**
		 * Execute the query server command from within the server state,
		 * receives and echo back fixed length string request as response.
		 */
		public void execute() throws IOException {

			// Invoke super class executes to initialize appropriately
			super.execute();

			// Receive data from client using simple communication abstractions
			DetermineServerReachCommand.this.host = this.communicator
					.receiveString();
			DetermineServerReachCommand.this.port = this.communicator
					.receiveInt();

			// Try to make a connection to the server socket
			this.determineServerReach();

			// Send back results of reachability connection test
			this.send(DetermineServerReachCommand.this.isHostReachable);

		}

		/**
		 * Sends a boolean through the channel communicator, this should be
		 * pushed in to the channel communicator in the future but as not to
		 * break compatibility it is left here for now
		 * 
		 * @param data
		 *            the data to send through the channel communicator
		 * @throws IOException
		 */
		private void send(boolean data) throws IOException {
			this.communicator.send(data ? DetermineServerReachCommand.TRUE
					: DetermineServerReachCommand.FALSE);
		}

	}

	/**
	 * Constant used to indicate false by using a byte
	 */
	private static final byte[] FALSE = new byte[] { 0 };

	/**
	 * Constant used to indicate true by using a byte
	 */
	private static final byte[] TRUE = new byte[] { 1 };

	/**
	 * The host to attempt connection to
	 */
	private String host;

	/**
	 * Indicates the last known state of server availability
	 */
	private boolean isHostReachable;

	/**
	 * The port to attempt connection using
	 */
	private int port;

	/**
	 * Initializes the server availability state and is refreshed upon request
	 */
	{
		this.isHostReachable = false;
	}

	/**
	 * Used for the server-side, where a channel is given to communicate with
	 * the client
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the client channel
	 */
	public DetermineServerReachCommand(String context, ISocketChannel channel) {
		super(context, DetermineServerReachCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Used for the client-side, where a cookie is given to identify this
	 * instance of request
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to the server
	 */
	public DetermineServerReachCommand(String context, ISocketChannel channel,
			String host, int port) {
		super(context, DetermineServerReachCommand.class);
		this.setState(new Client(channel));
		this.host = host;
		this.port = port;
	}

	/**
	 * Attempts connection to the specified host on the given port to determine
	 * if it is possible
	 */
	public boolean isHostReachable() {
		return this.isHostReachable;
	}

}
