/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileServerCommandFactory.java,v 1.12 2008/04/09 16:05:00 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketException;
import java.util.HashMap;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileLocation;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.execution.core.util.Guid;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannelFactory;
import org.eclipse.hyades.internal.execution.core.file.socket.SocketChannelFactory;

/**
 * Used by the client and server side of the file manager implementation to
 * create commands that encapsulate the state and behavior associated with a
 * particular action that should be conducted.
 * <p>
 * There is a maintained singleton on the server and a qualified set of
 * instances on the client side per address and port combination. The current
 * supports commands are gathered from the public interface to the file server
 * command factory.
 * <p>
 * This class encapsulates the client side connection to the server, figuring
 * out the proper port to use to initiate file server commands and whether to
 * create new or reuse existing sockets. In server operation, this factory will
 * operate on the socket channels that are passed-in per call and not keep the
 * server running -- this is done in the file server extended class.
 * <p>
 * There is also a polymorphic create method that will create a command based on
 * the fully-qualified Java class name; this is used on the server-side to
 * create commands based on the comman identity that is sent over the wire.
 * 
 * @see org.eclipse.hyades.internal.execution.file.FileServerExtended
 * @see org.eclipse.hyades.execution.local.file.FileManagerExtendedImpl
 * @see #createDeleteFileCommand(Cookie, IProgressMonitor)
 * @see #createDeleteFileCommand(FileIdentifierList, IProgressMonitor)
 * @see #createGetFileCommand(Cookie, FileIdentifierList, FileIdentifierList,
 *      Option[], IProgressMonitor)
 * @see #createListContentCommand(FileIdentifierList, IProgressMonitor)
 * @see #createModifyPermissionCommand(FileIdentifierList, String,
 *      IProgressMonitor)
 * @see #createPutFileCommand(Cookie, FileIdentifierList, FileIdentifierList,
 *      Option[], IProgressMonitor)
 * @see #createQueryServerStatusCommand()
 * @author Scott E. Schneider
 */
public class FileServerCommandFactory implements IFileServerCommandFactory {

	/**
	 * Factor to multiply the previous timeout by each time a new attempt is
	 * required
	 */
	private static final float DEFAULT_CONNECT_RETRY_CALCULATION_FACTOR = 0.90f;

	/**
	 * When the retry connect timeout drops below this cutoff, the connection
	 * algorithm gives up and throws a server not available exception
	 */
	private static final int DEFAULT_CONNECT_RETRY_CUTOFF = 2000;

	/**
	 * Used to recalculate the timeout value in the recalculate method in this
	 * class
	 */
	private static final int DEFAULT_CONNECT_RETRY_INITIAL_TIMEOUT = 3000;

	/**
	 * The offset value is added to all recalculated values before use,
	 * including the initial value
	 */
	private static final int DEFAULT_CONNECT_RETRY_OFFSET = 200;

	/**
	 * Stores the factories, used on the client-side
	 */
	private static final HashMap factories;

	/**
	 * Stores the one factory, used on the server-side
	 */
	private static IFileServerCommandFactory factory;

	/**
	 * Create the pool of factories to use
	 */
	static {
		factories = new HashMap();
	}

	/**
	 * Calculates a lookup key to use for this address and factory combination,
	 * used to cache the combinations so the factory maintains a pool of valid
	 * combinations on demand
	 * 
	 * @param fileServerAddress
	 *            the address of the file system services server
	 * @param socketFactory
	 *            the socket channel factory to use for sockets
	 * @return the lookup key to use
	 */
	private static String deriveLookupKey(InetSocketAddress fileServerAddress,
			ISocketChannelFactory socketFactory) {
		return fileServerAddress + "~" + socketFactory;
	}

	/**
	 * Get file server command factory, for the server side, uses the default
	 * socket channel factory
	 * 
	 * @return the file server command factory for server side use
	 */
	public synchronized static IFileServerCommandFactory getInstance() {
		return FileServerCommandFactory.getInstance(SocketChannelFactory
				.getInstance());
	}

	/**
	 * A pool of factories is kept, qualified by the address of the server, this
	 * is used by client that wish to issue commands to the server
	 * 
	 * @param fileServerLocation
	 *            the location and other important information qualifying the
	 *            exact file server host to use
	 * @return a factory ready to use for the specified address
	 */
	public synchronized static IFileServerCommandFactory getInstance(
			IFileLocation fileServerLocation,
			ISocketChannelFactory socketFactory) {

		// Retrieve internet address and port from the location instance
		InetAddress address = fileServerLocation.getInetAddress();
		int port = fileServerLocation.getPort();
		InetSocketAddress fileServerAddress = new InetSocketAddress(address,
				port);

		// Return file server command factory based on qualifiers
		return FileServerCommandFactory.getInstance(fileServerAddress,
				socketFactory);

	}

	/**
	 * A pool of factories is kept, qualified by the address of the connection,
	 * this is used by clients that wish to issue commands
	 * 
	 * @param fileServerAddress
	 *            the file server address combination of IP address and port
	 *            number
	 * @return returns the file server command factory to use, either cached or
	 *         created new in cases of a unique address
	 */
	public synchronized static IFileServerCommandFactory getInstance(
			InetSocketAddress fileServerAddress,
			ISocketChannelFactory socketFactory) {

		// Determine if a factory already exists for this address
		IFileServerCommandFactory factory = (IFileServerCommandFactory) FileServerCommandFactory.factories
				.get(FileServerCommandFactory.deriveLookupKey(
						fileServerAddress, socketFactory));

		// If a factory doesn't already exist, create one and store it
		if (factory == null) {
			factory = new FileServerCommandFactory(fileServerAddress,
					socketFactory);
			FileServerCommandFactory.factories
					.put(FileServerCommandFactory.deriveLookupKey(
							fileServerAddress, socketFactory), factory);
		}

		// Return a fully-initialized factory
		return factory;

	}

	/**
	 * Singleton instance used by file server, no address is needed since its
	 * the server, this method is used by the server side of the file system
	 * services -- this assumes one file system services server per virtual
	 * machine considering the instance is branded with a socket factory that
	 * cannot change once set
	 * 
	 * @return the factory to interpet commands and create concrete commands to
	 *         execute
	 */
	public synchronized static IFileServerCommandFactory getInstance(
			ISocketChannelFactory socketFactory) {
		if (FileServerCommandFactory.factory == null) {
			FileServerCommandFactory.factory = new FileServerCommandFactory(
					socketFactory);
		}
		return FileServerCommandFactory.factory;
	}

	/**
	 * Address to use for this file server command factory when creating
	 * commands
	 */
	private InetSocketAddress address;

	/**
	 * Unique identity of this file server command factory instance
	 */
	private String identity;

	/**
	 * Factory used to create and wrap sockets with a higher-level local socket
	 * channel abstraction (local meaning *not* the standard socket channel from
	 * the libraries)
	 */
	private ISocketChannelFactory socketFactory;

	/**
	 * A singleton, not instantiated outside of this class
	 * 
	 * @param address
	 *            the address to use
	 */
	private FileServerCommandFactory(InetSocketAddress address,
			ISocketChannelFactory socketFactory) {
		this(socketFactory);
		this.address = address;
	}

	/**
	 * A singleton, not instantiated outside of this class
	 */
	private FileServerCommandFactory(ISocketChannelFactory socketFactory) {

		// Store socket channel factory for use by this factory
		this.socketFactory = socketFactory;

		// Generate GUID for file server command factory instance
		this.identity = this.generateIdentity();

	}

	/**
	 * Connect the socket channel using the initial plus offset timeout value
	 * for exception conditions.
	 * 
	 * @return the socket channel to the server ready to use
	 */
	private synchronized ISocketChannel connectSocketChannel()
			throws ServerNotAvailableException {
		return this
				.connectSocketChannel(FileServerCommandFactory.DEFAULT_CONNECT_RETRY_INITIAL_TIMEOUT
						+ FileServerCommandFactory.DEFAULT_CONNECT_RETRY_OFFSET);
	}

	/**
	 * Create a socket channel to the server given the specified timeout value,
	 * this timeout value will be used in exceptional conditions, and then
	 * recalculated using the appropriately named method in this class, this
	 * method attempts to create new connection or use one that is unused but
	 * already exists per the specified socket channel's behavior
	 * 
	 * @param timeout
	 *            the timeout value to use
	 * @return the socket channel ready to use, should never be null
	 * @throws ServerNotAvailableException
	 *             server not available exception if a socket cannot be
	 *             connected
	 * 
	 * @see #recalculateTimeout(int)
	 * @see ServerNotAvailableException
	 */
	private synchronized ISocketChannel connectSocketChannel(int timeout)
			throws ServerNotAvailableException {

		try {

			/*
			 * Attempt to open connection to file server using channels, to
			 * avoid the cost of connection negotiation, use bulk operations
			 * that embody multiple operands in one operation
			 */
			return this.socketFactory.create(this.address);

		} catch (SocketException e1) {

			// If any bind exception or all connection exceptions except refuse
			if (e1 instanceof BindException || e1 instanceof ConnectException) {

				e1.printStackTrace();

				// Return immediately on a connection refused exception
				if (e1.getMessage().indexOf("refuse") != -1) {//$NON-NLS-1$
					throw new ServerNotAvailableException(e1);
				}

				// Connection failed due to bind exception, try again after a
				// calculated wait
				try {

					if (timeout < FileServerCommandFactory.DEFAULT_CONNECT_RETRY_CUTOFF) {
						throw new ServerNotAvailableException(e1);
					} else {

						System.err.println("About to wait for " + timeout
								+ " seconds!"); // $NON-NLS-1$

						// Wait the variable timeout amount before re-attempt
						this.wait(timeout);

						/*
						 * Recurse back into this method with a shorter timeout
						 * value, the recursion exits when timeout drops below
						 * default timeout cutoff value and then a server not
						 * available exception is thrown
						 */
						return this.connectSocketChannel(this
								.recalculateTimeout(timeout));

					}

				} catch (InterruptedException e2) {
					throw new ServerNotAvailableException(e2);
				}

			}

			throw new ServerNotAvailableException(e1);

		} catch (Throwable t) {
			throw new ServerNotAvailableException(t);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createDeleteDirectoryCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IDeleteDirectoryCommand createDeleteDirectoryCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException {
		return new DeleteDirectoryCommand(this.identity, this
				.connectSocketChannel(), remoteIdentifiers, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createDeleteFileCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IDeleteFileCommand createDeleteFileCommand(Cookie cookie,
			IProgressMonitor monitor) {
		return new DeleteFileCommand(this.identity, cookie, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createDeleteFileCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IDeleteFileCommand createDeleteFileCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException {
		return new DeleteFileCommand(this.identity,
				this.connectSocketChannel(), remoteIdentifiers, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.dynamic.IFileServerCommandFactory
	 *      #createDetermineServerReachCommand(java.lang.String, int)
	 */
	public IDetermineServerReachCommand createDetermineServerReachCommand(
			String host, int port) throws ServerNotAvailableException {
		return new DetermineServerReachCommand(this.identity, this
				.connectSocketChannel(), host, port);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createFileServerCommand(java.lang.String,
	 *      org.eclipse.hyades.internal.execution.core.file.ISocketChannel)
	 */
	public IFileServerCommand createFileServerCommand(String identity,
			ISocketChannel clientChannel)
			throws InvalidFileServerCommandException {

		try {

			// Output to debug console
			FileSystemServices.println(
					"Attempting dynamic class loading of command class "
							+ identity, this); // $NON-NLS-1$

			// Dynamically load class with given identity, specific classloader
			Class classObject = Thread.currentThread().getContextClassLoader()
					.loadClass(identity);

			// Invoke a non-default constructor passing in channel
			Constructor constructor = classObject.getConstructor(new Class[] {
					String.class, ISocketChannel.class });
			IFileServerCommand command = (IFileServerCommand) constructor
					.newInstance(new Object[] { this.identity, clientChannel });

			// Output to debug console
			FileSystemServices.println(
					"New instance of command class constructed from the command class "
							+ identity, this); // $NON-NLS-1$

			// Return the command that is ready for requests
			return command;

		} catch (ClassNotFoundException e) {

			// Command could not be instantiated with the given identity
			e.printStackTrace();

			throw new InvalidFileServerCommandException();

		} catch (InstantiationException e) {

			// Command could not be instantiated with the given identity
			e.printStackTrace();

			throw new InvalidFileServerCommandException();

		} catch (IllegalAccessException e) {

			// Command could not be instantiated with the given identity
			e.printStackTrace();

			throw new InvalidFileServerCommandException();

		} catch (InvocationTargetException e) {

			// Command could not be instantiated with the given identity
			e.printStackTrace();

			throw new InvalidFileServerCommandException();

		} catch (NoSuchMethodException e) {

			// Command could not be instantiated with the given identity
			e.printStackTrace();

			throw new InvalidFileServerCommandException();

		}

	}

	/*
	 * (non-Javadoc)O
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.dynamic.IFileServerCommandFactory#createGetFileCommand(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IGetFileCommand createGetFileCommand(
			FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws ServerNotAvailableException {
		return new GetFileCommand(this.identity, this.connectSocketChannel(),
				localIdentifiers, remoteIdentifiers, options, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createListContentCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IListContentCommand createListContentCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) {
		return new ListContentCommand(this.identity, remoteIdentifiers, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createModifyPermissionCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IModifyPermissionCommand createModifyPermissionCommand(
			FileIdentifierList remoteIdentifiers, String permissionDirective,
			IProgressMonitor monitor) {
		return new ModifyPermissionCommand(this.identity, remoteIdentifiers,
				permissionDirective, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createPutFileCommand(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IPutFileCommand createPutFileCommand(Cookie cookie,
			FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws ServerNotAvailableException {
		return new PutFileCommand(this.identity, this.connectSocketChannel(),
				cookie, localIdentifiers, remoteIdentifiers, options, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory#createQueryServerStatusCommand()
	 */
	public IQueryServerStatusCommand createQueryServerStatusCommand()
			throws ServerNotAvailableException {
		return new QueryServerStatusCommand(this.identity, Cookie.NONE, this
				.connectSocketChannel());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.dynamic.IFileServerCommandFactory
	 *      #createValidateDirectoryExistence(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IValidateDirectoryExistenceCommand createValidateDirectoryExistenceCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException {
		return new ValidateDirectoryExistenceCommand(this.identity, this
				.connectSocketChannel(), remoteIdentifiers, monitor);
	}

	/**
	 * Generate an identity for this factory instance
	 * 
	 * @return a globally unique identifier for this factory
	 */
	private String generateIdentity() {
		return new Guid().toString();
	}

	/**
	 * Recalculate a new timeout value to try for the next connect attempt, used
	 * typically when a bind exception occurs because an address is already in
	 * use, this is due to local addresses being exhausted without old ones
	 * being reclaimed yet, this is a temporary situation which we'll wait out
	 * with this timeout value.
	 * 
	 * @param timeout
	 *            the timeout value to seed the new timeout value on
	 * @return the new timeout value that has been recalculated
	 */
	private int recalculateTimeout(int timeout) {
		float recalculatedTimeout = timeout
				* FileServerCommandFactory.DEFAULT_CONNECT_RETRY_CALCULATION_FACTOR
				+ FileServerCommandFactory.DEFAULT_CONNECT_RETRY_OFFSET;
		return Math.round(recalculatedTimeout);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.dynamic.IFileServerCommandFactory#reset()
	 */
	public void reset() {
		this.identity = this.generateIdentity();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return this.getClass().getName()
				+ " [identity="
				+ this.identity
				+ (this.address != null
						&& this.address.toString().trim().length() > 0 ? ", address="
						+ this.address.toString()
						: "") + "]";//$NON-NLS-1$
	}

}