/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html $Id:
 * IFileServerCommandFactory.java,v 1.1 2005/05/03 14:32:54 sschneid Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * Interface for file server command factories -- used by both the client and
 * server sides of the file service implementation. The server side uses the
 * create file server command method while the client side of this mechanism
 * uses the rest. File server commands have client and server side personalities
 * that in effect, change the class of the command at run-time (the state
 * pattern is implemented in all file server commands, one state represents the
 * server-side personality and another represents the client-side).
 * 
 * @author Scott E. Schneider
 */
public interface IFileServerCommandFactory {

	/**
	 * Create delete directory command for use by file transfer service clients
	 * 
	 * @param remoteIdentifiers
	 *            the identifiers on the remote server to be deleted
	 * @param monitor
	 *            progress monitor for the delete directory command
	 * 
	 * @return the delete directory command that is ready to command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IDeleteDirectoryCommand createDeleteDirectoryCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException;

	/**
	 * Create delete file command for use by file transfer service clients
	 * 
	 * @param cookie
	 *            the cookie to use for identifying the associated file
	 *            operation that created files, a cookie used for a put file
	 *            command for example, can then be used to pass in the delete
	 *            file command -- this will then result in the delete command
	 *            deleting only the files that were created during the put file
	 *            command operation, the cookie identifies which files to remove
	 *            via the remote server process
	 * @param monitor
	 *            progress monitor for the delete file command
	 * 
	 * @return the delete file command that is ready to command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IDeleteFileCommand createDeleteFileCommand(Cookie cookie,
			IProgressMonitor monitor) throws ServerNotAvailableException;

	/**
	 * Create delete file command for use by file transfer service clients
	 * 
	 * @param remoteIdentifiers
	 *            the identifiers on the remote server to be deleted
	 * @param monitor
	 *            progress monitor for the delete file command
	 * 
	 * @return the delete file command that is ready to command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IDeleteFileCommand createDeleteFileCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException;

	/**
	 * Create determine server reach command for use by file system service
	 * clients
	 * 
	 * @param host
	 *            the host for the server to attempt reach to (connection to)
	 * @param port
	 *            the port to attempt reach (connection) on
	 * @throws ServerNotAvailableException
	 */
	IDetermineServerReachCommand createDetermineServerReachCommand(String host,
			int port) throws ServerNotAvailableException;

	/**
	 * Create a file server command to be used by the file transfer service
	 * server, all server-side hosted file server commands are created through
	 * this method -- the appropriate command to create is identified by the
	 * identity argument passed in. The client channel is passed in, which is
	 * the channel to use for communicating with the client-side of the file
	 * transfer mechanism.
	 * 
	 * Uses the default classloader to load the file server command.
	 * 
	 * @param identity
	 *            the identity of the command to create on the server
	 * @param clientChannel
	 *            the channel to use to communicate with the client from the
	 *            server
	 * 
	 * @return a file server command for server-side handling
	 * 
	 * @throws InvalidFileServerCommandException
	 * 
	 * @see #createFileServerCommand(String, ISocketChannel, ClassLoader)
	 */
	IFileServerCommand createFileServerCommand(String identity,
			ISocketChannel clientChannel)
			throws InvalidFileServerCommandException;

	/**
	 * Create get file command for use by file transfer service clients
	 * 
	 * @param localIdentifiers
	 *            the identifiers of the files to be created locally, the
	 *            destination file identifiers for the transfer
	 * @param remoteIdentifiers
	 *            the identifiers of the files to be transferred from the remote
	 *            server to the local client, the source file identifiers for
	 *            the transfer
	 * @param options
	 *            options to configure the get file command behavior
	 * @param monitor
	 *            progress monitor for the get file command operation
	 * 
	 * @return the get file command that is ready to command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IGetFileCommand createGetFileCommand(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws ServerNotAvailableException;

	/**
	 * Create validate directory existence exists command for use by file system
	 * service clients
	 * 
	 * @param remoteIdentifiers
	 *            the identifiers to verify existence of, all or nothing, all
	 *            directories in the list must exist for the return value of
	 *            this service to return true
	 * @param monitor
	 *            progress monitor for the verify directory exists operation
	 * @return the verify directory exists commmand this is ready to command
	 * @throws ServerNotAvailableException
	 */
	IValidateDirectoryExistenceCommand createValidateDirectoryExistenceCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException;

	/**
	 * Create list content command for use by file transfer service clients,
	 * used to list the content of a directory or folder on the remote server
	 * 
	 * @param remoteIdentifiers
	 *            identifies the directories to list the contents of
	 * @param monitor
	 *            progress monitor for the list content command
	 * 
	 * @return the list content command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IListContentCommand createListContentCommand(
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws ServerNotAvailableException;

	/**
	 * Create modify permission command for use by file transfer service
	 * clients, used to modify the permissions of files on the remote file
	 * server
	 * 
	 * @param remoteIdentifiers
	 *            the remote identifiers to affect with the modify permission
	 *            command's operation
	 * @param permissionDirective
	 *            the permission directives to affect the files
	 * @param monitor
	 *            progress monitor for the modify permission command
	 * 
	 * @return the modify permission command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IModifyPermissionCommand createModifyPermissionCommand(
			FileIdentifierList remoteIdentifiers, String permissionDirective,
			IProgressMonitor monitor) throws ServerNotAvailableException;

	/**
	 * Create put file command for use by file transfer service clients
	 * 
	 * @param cookie
	 *            the cookie to use for identifying the instance of the command
	 *            between client and server, for example, the cookie represents
	 *            an instance of the command on the client and can then be
	 *            associated with the command on the server that is associated
	 *            with the same operation
	 * @param localIdentifiers
	 *            the identifiers of the files to be sent to the remote server,
	 *            the source file identifiers in this operation
	 * @param remoteIdentifiers
	 *            the identifiers of the files to be created on the remote
	 *            server process, the destination file identifiers in the
	 *            operation
	 * @param options
	 *            options to configure the put file command behavior
	 * @param monitor
	 *            progress monitor for the put file command
	 * 
	 * @return the put file command that is ready to command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IPutFileCommand createPutFileCommand(Cookie cookie,
			FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws ServerNotAvailableException;

	/**
	 * Create query server status command for use by file transfer service
	 * clients, used to query the status of the file server, a simple echo test
	 * or equivalent is used as a simple sanity test
	 * 
	 * @return the query server status command
	 * 
	 * @throws ServerNotAvailableException
	 */
	IQueryServerStatusCommand createQueryServerStatusCommand()
			throws ServerNotAvailableException;

	/**
	 * Resets the factory, clears any appropriate state for this particular
	 * factory instance
	 */
	void reset();

}