/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PutFileCommand.java,v 1.8 2008/04/09 16:05:00 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * The put file command is a file server command that is used to transfer files
 * from the client to the server. The main components of state that make up the
 * put file command are local identifiers of the files to transfer, remote
 * identifiers of the files to ensure existence on the server-side, options to
 * tweak the put file behavior and progress monitor to monitor the progress as
 * well as cancel the transfer. An advantage to the put file command is that it
 * supports the transfer of multiple files in one pass, using the same socket
 * and connection for all the files transferred. A single file transfer is also
 * possible, basically one to many file identifiers can be passed in to identity
 * the local and remote files to act upon.
 * 
 * @author Scott E. Schneider
 */
class PutFileCommand extends AbstractFileTransferCommand implements IPutFileCommand {

	/**
	 * The client side personality of the put file command, locates the physical
	 * files, opens them and transfers them to server
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractFileTransferCommand.Client {

		/**
		 * Creates the client side personality for the put file command
		 * 
		 * @param channel
		 *            the channel to communicate with the server
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate client
			super.execute();

			// Retrieve local and remote identifiers
			FileIdentifierList localIdentifiers = PutFileCommand.this.localIdentifiers;
			FileIdentifierList remoteIdentifiers = PutFileCommand.this.remoteIdentifiers;

			/*
			 * Filter out invalid local identifiers, removing the appropriate
			 * matching end of the pair in the remote identifier list as well.
			 * These identifiers are stored as a reference, so the provider of
			 * the original identifier lists will observe changes made as
			 * filtered.
			 */
			FileIdentifierList.filter(localIdentifiers, remoteIdentifiers, new FileIdentifierList.Filter() {
				public boolean filter(String localIdentifier, String remoteIdentifier) {
					return !(new File(localIdentifier).isFile());
				}
			});

			// Send number of files that will be transferred
			this.communicator.send(localIdentifiers.size());

			// Send all remote identifier names
			this.communicator.send(remoteIdentifiers.getArray());

			// Send all the files to the server one at a time but continuous
			for (Iterator identifiers = localIdentifiers.iterator(); identifiers.hasNext();) {
				String fileName = (String) identifiers.next();
				File localFile = new File(fileName);
				this.communicator.send(localFile);
			}

			/*
			 * Receive remote identifier resolutions back from server, used to
			 * indicate remote identifiers that were changed from relative to
			 * absolution upon use.
			 */
			String[] remoteIdentifierResolutions = this.communicator.receiveStrings();

			// For each resolution that is non-null -- update state
			for (int i = 0; i < remoteIdentifierResolutions.length; i++) {
				String remoteIdentifierResolution = remoteIdentifierResolutions[i];
				if (remoteIdentifierResolution != null) {
					remoteIdentifiers.remove(i);
					remoteIdentifiers.add(i, remoteIdentifierResolution);
				}
			}

		}
	}

	/**
	 * The server side personality of the put file command
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractFileTransferCommand.Server {

		/**
		 * Constructs the server side state personality for the put file command
		 * 
		 * @param channel
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.internal.execution.core.file.ICommand#execute()
		 */
		public void execute() throws IOException {

			// Initiate server
			super.execute();

			// Receive number of files that will be transferred
			this.communicator.receiveInt();

			// Receive the remote identifiers from the client
			FileIdentifierList remoteIdentifiers = FileIdentifierList.create(this.communicator.receiveStrings());
			PutFileCommand.this.remoteIdentifiers = remoteIdentifiers;

			/**
			 * Remote identifiers that are relative and need to be resolved into
			 * absolute paths will be tracked here and returned back to the put
			 * file command client. They are bundled up and then the remote
			 * identifier list is updated all at one time.
			 */
			final String[] remoteIdentifierResolutions = new String[remoteIdentifiers.size()];

			// Receive all the files, one at a time but continuous
			int index = 0;
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers.hasNext(); index++) {

				// Retrieve next remote file identifier
				String remoteIdentifier = (String) identifiers.next();
				File remoteFile = new File(remoteIdentifier);

				if (remoteFile.isAbsolute()) {

					/*
					 * If remote file is an absolute path then transfer it
					 * directly to the specified absolute location.
					 */
					boolean readyToTransfer = true;

					// Create directories if they don't exist
					File remoteDirectory = new File(remoteFile.getParent());
					if (!remoteDirectory.exists()) {
						readyToTransfer = remoteDirectory.mkdirs();
					}
					if (readyToTransfer) {
						remoteFile.createNewFile();
						this.communicator.receive(remoteFile);
					}

					FileSystemServices.println("Absolute file identifier " + remoteFile.getAbsolutePath()
							+ " identified and file received by server", this); // $NON-NLS-1$

				} else {

					/*
					 * Remote file identifier is relative and not absolute,
					 * force it to be absolute by using a suitable resting place
					 * on the server and then update the remote identifier to
					 * contain the absolute path used.
					 * 
					 * Look for the file extension and detach it from the
					 * primary part of the file name.
					 */
					String[] segments = REGULAR_EXPRESSION.split(remoteIdentifier);

					// If segments exist, figure prefix and suffix
					if (segments != null) {

						// Initialize suffix and prefix and segment length
						String suffix = "";
						String prefix = "";
						int length = segments.length;

						// Build prefix using last period to trigger suffix
						for (int i = 0; i < length - 1; i++) {
							prefix += segments[i] + ".";
						}

						/*
						 * If no period is present, make all prefix -- otherwise
						 * make suffix just the last segment after period.
						 */
						if (prefix.length() == 0) {
							prefix = remoteIdentifier;
						} else {
							suffix = segments[length - 1];
						}

						// Retrieve temporary directory (platform specific)
						String temporaryDirectory = System.getProperty("java.io.tmpdir");//$NON-NLS-1$

						// Handle potential for trailing path separators
						String qualifiedFilePrefix = temporaryDirectory
								+ (temporaryDirectory.lastIndexOf(File.separatorChar) == temporaryDirectory.length() - 1 ? ""
										: "/") + prefix;
						
						// Determine qualified directory
						{

							// Allocate a qualified directory variable
							File qualifiedDirectory;

							// Find separators, supports cross-platform
							int firstSeparator = qualifiedFilePrefix.indexOf(File.separator);

							/*
							 * Need to handle case where relative part has
							 * forward slash but platform default is backslash,
							 * Windows case basically
							 */
							int lastSeparator = Math.max(qualifiedFilePrefix.lastIndexOf(File.separator),
									qualifiedFilePrefix.lastIndexOf('/'));

							// Determine directory using separator positions
							if (firstSeparator >= 0 && firstSeparator != lastSeparator) {
								String qualifiedFilePrefixParent = qualifiedFilePrefix.substring(0, lastSeparator);
								qualifiedDirectory = new File(qualifiedFilePrefixParent);
								qualifiedDirectory.mkdirs();
								prefix = qualifiedFilePrefix.substring(lastSeparator + 1);
							} else {
								qualifiedDirectory = new File(qualifiedFilePrefix);
							}

							// Create the remote file to store transfer
							remoteFile = new File(qualifiedDirectory, prefix.concat(suffix));

						}

						// Populate sparse array with remote identifier path
						remoteIdentifierResolutions[index] = remoteFile.getAbsolutePath();

					}

					// Receive file from the client
					this.communicator.receive(remoteFile);

					FileSystemServices.println("Relative file identifier resolved to " + remoteFile.getAbsolutePath()
							+ " and file received by server", this); // $NON-NLS-1$

				}

			}

			// Send modified remote identifiers in sparse array
			this.communicator.send(remoteIdentifierResolutions);

		}
	}

	/**
	 * Regular expression splitting file prefix and suffix for identifiers,
	 * compiled once to optimize performance
	 */
	private static final Pattern REGULAR_EXPRESSION = Pattern.compile("\\.");

	/**
	 * Constructs a put file command from a socket channel, this is the
	 * constructor that the file server uses
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the client
	 */
	public PutFileCommand(String context, ISocketChannel channel) {
		super(context, PutFileCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Constructs a put file command from a socket channel to the server, and
	 * the command state that defines the exact purpose of this command
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to coomunicate with the server
	 * @param cookie
	 *            the cookie that identifies this particular command instance
	 * @param localIdentifiers
	 *            the local identifiers identifying the local files to transfer
	 * @param remoteIdentifiers
	 *            the remote identifiers identifying the remote files to copy
	 *            into
	 * @param options
	 *            the options that tweak this command's behavior
	 * @param monitor
	 *            the progress monitor to use
	 */
	PutFileCommand(String context, ISocketChannel channel, Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options, IProgressMonitor monitor) {
		super(context, PutFileCommand.class, cookie, localIdentifiers, remoteIdentifiers, options, monitor);
		this.setState(new Client(channel));
	}

}