/********************************************************************** 
 * Copyright (c) 2005 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: QueryServerStatusCommand.java,v 1.2 2008/03/20 18:49:52 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.IOException;

import org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * A simple command that attempts to ping the server and make sure it is
 * available and ready for other commands to be sent, it sends a message to the
 * server of an expected length and the server echos back the message with the
 * same length. The command then compares the request with the response message
 * and if they equal it sets the server availability to true; if any exceptions
 * occur of the message does not come back as expected, the server availability
 * is set to false;
 * 
 * @author Scott E. Schneider
 */
class QueryServerStatusCommand extends AbstractServerInterrogationCommand implements IQueryServerStatusCommand {

	/**
	 * The client state personality of this command, refer to the server state
	 * class personality for the ying and the yang constrast between the server
	 * and client states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractServerInterrogationCommand.Client {

		/**
		 * The client state making this command behave as the client
		 * 
		 * @param channel
		 *            the server channel (the channel to communicate with the
		 *            server)
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Execute the query server status command from within the client state,
		 * sends and then receives for comparison the bounced back message to
		 * verify it worked.
		 */
		public void execute() throws IOException {

			// Invoke super class executes to send command
			super.execute();

			// Send message to server using simple communication abstractions
			this.communicator.send(QueryServerStatusCommand.MESSAGE);
			String response = this.communicator.receiveString();

			// Set the is server available state based the response matching
			QueryServerStatusCommand.this.isServerAvailable = response.equals(QueryServerStatusCommand.MESSAGE);

		}
	}

	/**
	 * The server state personality of this command, refer to the client state
	 * class personality for the ying and the yang contrast between the client
	 * and server states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractServerInterrogationCommand.Server {

		/**
		 * The server state making this command behave as the server (the
		 * channel to communicate with the client)
		 * 
		 * @param channel
		 *            the client channel
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Execute the query server command from within the server state,
		 * receives and echo back fixed length string request as response.
		 */
		public void execute() throws IOException {

			// Invoke super class executes to initialize appropriately
			super.execute();

			// Send message to client using simple communication abstractions
			String request = this.communicator.receiveString();

			// Echo back message to calling client and client will compare
			this.communicator.send(request);

		}

	}

	/**
	 * Arbitrary message to use for testing the server
	 */
	private static final String MESSAGE = "Hello";//$NON-NLS-1$

	/**
	 * Initializes the server availability state and is refreshed upon request
	 */
	{
		this.isServerAvailable = false;
	}

	/**
	 * Indicates the last known state of server availability
	 */
	private boolean isServerAvailable;

	/**
	 * Used for the client-side, where a cookie is given to identify this
	 * instance of request
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param cookie
	 *            a cookie to identify this command instance, currently the
	 *            cookie in this command is not used
	 * @param channel
	 *            the channel to the server
	 */
	public QueryServerStatusCommand(String context, Cookie cookie, ISocketChannel channel) {
		super(context, QueryServerStatusCommand.class);
		this.setState(new Client(channel));
	}

	/**
	 * Used for the server-side, where a channel is given to communicate with
	 * the client
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the client channel
	 */
	public QueryServerStatusCommand(String context, ISocketChannel channel) {
		super(context, QueryServerStatusCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Queries the server by doing a quick sanity echo test to make sure
	 * everything is working as required, this is exposed to clients of this
	 * command through this particular command's public interface
	 */
	public boolean isServerAvailable() {
		return this.isServerAvailable;
	}

}