/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ValidateDirectoryExistenceCommand.java,v 1.3 2008/04/09 16:05:00 jcayne Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.core.file.dynamic;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.communicator.IChannelCommunicator;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;

/**
 * A file inspection command that determines if a directory is existent on the
 * server, if the item to inspect is not a directory even though it exists,
 * existence of a valid directory is false. It will be deemed existent if the
 * item of interest is both a directory and is existent. If multiple items are
 * specified, an parallel array is returned with the existence findings for each
 * specified item in order.
 * 
 * @author Scott E. Schneider
 */
class ValidateDirectoryExistenceCommand extends AbstractFileInspectionCommand
		implements IValidateDirectoryExistenceCommand {

	/**
	 * The client state personality of this command, refer to the server state
	 * class personality for the ying and the yang constrast between the server
	 * and client states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Client extends AbstractFileInspectionCommand.Client {

		/**
		 * The client state making this command behave as the client
		 * 
		 * @param channel
		 *            the server channel (the channel to communicate with the
		 *            server)
		 */
		Client(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Execute the client side part of the verify directory exists command,
		 * sends the remote file identifier's string representations and
		 * receives an array of booleans indicating if each remote identifier in
		 * order exists
		 */
		public void execute() throws IOException {

			// Invoke super class executes to send command
			super.execute();

			// Retrieve remote identifiers
			FileIdentifierList remoteIdentifiers = ValidateDirectoryExistenceCommand.this.remoteIdentifiers;

			// Send number of directories that will be inspected
			this.communicator.send(remoteIdentifiers.size());

			// Send all remote identifier names
			this.communicator.send(remoteIdentifiers.getArray());

			// Sets the directory existence state based on server return state
			ValidateDirectoryExistenceCommand.this.isDirectoryExistent = ValidateDirectoryExistenceCommand.this
					.receiveBooleans(this.communicator);

		}
	}

	/**
	 * The server state personality of this command, refer to the client state
	 * class personality for the ying and the yang contrast between the client
	 * and server states
	 * 
	 * @author Scott E. Schneider
	 */
	private class Server extends AbstractFileInspectionCommand.Server {

		/**
		 * The server state making this command behave as the server (the
		 * channel to communicate with the client)
		 * 
		 * @param channel
		 *            the client channel
		 */
		Server(ISocketChannel channel) {
			super(channel);
		}

		/**
		 * Execute the verify directory exists command from within the server
		 * state, receives the amount and name of each identifier and then sends
		 * back a boolean array of results
		 */
		public void execute() throws IOException {

			// Invoke super class executes to initialize appropriately
			super.execute();

			// Initialize array to hold findings
			int count = this.communicator.receiveInt();
			ValidateDirectoryExistenceCommand.this.isDirectoryExistent = new boolean[count];

			// Receive the remote identifiers from the client
			FileIdentifierList remoteIdentifiers = FileIdentifierList
					.create(this.communicator.receiveStrings());
			ValidateDirectoryExistenceCommand.this.remoteIdentifiers = remoteIdentifiers;

			// Determine if items are valid existent directories
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers
					.hasNext(); count--) {
				String fileName = (String) identifiers.next();
				File remoteDir = new File(fileName);
				if (remoteDir.exists() && !remoteDir.isFile()) {
					ValidateDirectoryExistenceCommand.this.isDirectoryExistent[count - 1] = true;
					FileSystemServices.println("Absolute file identifier "
							+ remoteDir.getAbsolutePath()
							+ " verified to be valid and existent on server",
							this); // $NON-NLS-1$
				} else {
					FileSystemServices
							.println(
									"Absolute file identifier "
											+ remoteDir.getAbsolutePath()
											+ " not identified and therefore ignoring inspection request",
									this); // $NON-NLS-1$
				}
			}

			// Return back the directory existence findings
			ValidateDirectoryExistenceCommand.this.send(this.communicator,
					ValidateDirectoryExistenceCommand.this.isDirectoryExistent);

		}
	}

	/**
	 * Indicates the last known state of the directory existence
	 */
	private boolean[] isDirectoryExistent;

	/**
	 * Constructs a validate directory existence command from a socket channel
	 * to the server, and the command state that defines the exact purpose of
	 * this command, this is the constructor the client side uses
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to coomunicate with the server
	 * @param remoteIdentifiers
	 *            the remote identifiers identifying the directories to validate
	 *            into
	 * @param monitor
	 *            the progress monitor to use
	 */
	ValidateDirectoryExistenceCommand(String context, ISocketChannel channel,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) {
		super(context, ValidateDirectoryExistenceCommand.class,
				remoteIdentifiers, monitor);
		this.setState(new Client(channel));
	}

	/**
	 * Constructs a validate directory existence command using the specified
	 * channel, this is the constructor that the file server uses
	 * 
	 * @param context
	 *            the context this command is executing within
	 * @param channel
	 *            the channel to communicate with the client
	 */
	public ValidateDirectoryExistenceCommand(String context,
			ISocketChannel channel) {
		super(context, ValidateDirectoryExistenceCommand.class);
		this.setState(new Server(channel));
	}

	/**
	 * Determines if each remote identifier specified is found to be a valid
	 * existent directory on the server
	 */
	public boolean[] isDirectoryExistent() {
		return this.isDirectoryExistent;
	}

	/**
	 * Receive a boolean array using the specified communicator
	 * 
	 * @return the array of booleans received from the communicators other end
	 * @throws IOException
	 */
	public boolean[] receiveBooleans(IChannelCommunicator communicator)
			throws IOException {
		int length = communicator.receiveInt();
		boolean[] booleans = new boolean[length];
		if (length >= 0) {
			byte[] bytes = communicator.receiveBytes(length);
			for (int i = 0; i < booleans.length; i++) {
				// MIN_VALUE = FALSE and MAX_VALUE = TRUE
				booleans[i] = bytes[i] == Byte.MAX_VALUE;
			}
		} else {
			throw new IOException();
		}
		return booleans;
	}

	/**
	 * Send the specified boolean array using the specified communicator
	 * 
	 * @param data
	 *            the boolean array to send
	 * @throws IOException
	 */
	public void send(IChannelCommunicator communicator, boolean[] data)
			throws IOException {
		communicator.send(data.length);
		byte[] bytes = new byte[data.length];
		for (int i = 0; i < data.length; i++) {
			// MIN_VALUE = FALSE and MAX_VALUE = TRUE
			bytes[i] = data[i] ? Byte.MAX_VALUE : Byte.MIN_VALUE;
		}
		communicator.send(bytes);
	}

}