/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SocketChannelFactory.java,v 1.7 2008/05/20 16:55:30 jkubasta Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.internal.execution.core.file.socket;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;

/**
 * Factory to create the local socket channel abstraction, this factory offers a
 * default instance to use for normal unsecure socket establishment but other
 * factories can be implemented that implement the socket channel factory
 * interface and used where socket channel factories are applicable and required
 * 
 * @author Scott E. Schneider
 */
public class SocketChannelFactory implements ISocketChannelFactory {

	/**
	 * The default socket channel factory instance
	 */
	private static final ISocketChannelFactory factory;

	/**
	 * Create and store the singleton instance, not using on demand creation,
	 * always create on class load
	 */
	static {
		factory = new SocketChannelFactory();
	}

	/**
	 * Default instance accessor method
	 * 
	 * @return the default socket channel factory, another implementation is
	 *         possible by implementing the socket channel factory interface
	 */
	public static ISocketChannelFactory getInstance() {
		return SocketChannelFactory.factory;
	}

	/**
	 * Limit instantiation to factory extenders
	 */
	protected SocketChannelFactory() {
	}


	/* (non-Javadoc)
	 * @see org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannelFactory#create(java.net.InetSocketAddress)
	 */
	public ISocketChannel create(InetSocketAddress address) throws IOException {
		return this.create(address, 60000);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannelFactory#create(java.net.InetSocketAddress, int)
	 */
	public ISocketChannel create(InetSocketAddress address, int timeout)
			throws IOException {
		
		if(address.getAddress().getAddress().length == 16) {
			Socket s = new Socket(address.getAddress(), address.getPort());
			
			s.setSoTimeout(timeout);
			s.setTcpNoDelay(true);
			s.setReuseAddress(true);
			
			return new SocketChannel(s);
			
		}
		
		java.nio.channels.SocketChannel socketChannel = java.nio.channels.SocketChannel
				.open();
		socketChannel.configureBlocking(true);
		Socket socket = socketChannel.socket();
		socket.setSoTimeout(timeout);
		socket.setTcpNoDelay(true);
		socket.setReuseAddress(true);
		socketChannel.connect(address);
		return new SocketChannel(socketChannel);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.ISocketChannelFactory#create(java.nio.channels.SocketChannel)
	 */
	public ISocketChannel create(java.nio.channels.SocketChannel realChannel) {
		return new SocketChannel(realChannel);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.core.file.ISocketChannelFactory#create(java.net.Socket)
	 */
	public ISocketChannel create(Socket socket) throws IOException {
		return new SocketChannel(socket);
	}

}