/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SocketChannel.java,v 1.1 2007/04/09 14:04:26 samwai Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.core.file.socket;

import java.io.IOException;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

/**
 * An abstraction above the socket channel to allow a socket channel to be emulated by taking a readable byte channel and a writeable byte channel and combining them into a new interface. Socket channel is a class and not an interface in Java, so an interface and an adapter class implementation is introduced to satisfy our purposes.
 * 
 * A socket channel instance can be created from a standard socket or from a standard channel.
 * 
 * @see org.eclipse.hyades.execution.core.file.socket.ISocketChannel
 * 
 * @author Scott E. Schneider
 */
class SocketChannel implements ISocketChannel {

	/**
	 * An adapter that adapts a socket to look like a channel. A simple interface is exposed to clients with the is open and close methods.
	 * 
	 * @author Scott E. Schneider
	 */
	private class ChannelInterfaceAdapter implements Channel {

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.nio.channels.Channel#close()
		 */
		public void close() throws IOException {
			SocketChannel.this.socket.close();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.nio.channels.Channel#isOpen()
		 */
		public boolean isOpen() {
			return !SocketChannel.this.socket.isClosed();
		}

	}

	/**
	 * The channel interface where all channel interface calls are delegated to
	 */
	private final Channel channel;

	/**
	 * The readable byte channel, all readable byte channel interface invocations are delegated to this instance
	 */
	private final ReadableByteChannel readable;

	/**
	 * The underlying socket for this socket channel
	 */
	private final Socket socket;

	/**
	 * The writable byte channel, all writable byte channel interface invocations are delegate to this instance
	 */
	private final WritableByteChannel writable;

	/**
	 * For a standard "real" channel from java.nio, simply store the instance in three interface-held variables, used by the corresponding methods in this class to adapt the calls (just re-forward them into the appropriate delegates in most cases)
	 * 
	 * @param realChannel
	 *            the real channel obtained from the socket channel related classes in java.nio
	 */
	SocketChannel(java.nio.channels.SocketChannel realChannel) {
		this.channel = realChannel;
		this.readable = realChannel;
		this.writable = realChannel;
		this.socket = realChannel.socket();
	}

	/**
	 * A socket channel emulation is created given a standard socket, this constructor will take the socket, obtain input and output streams and then using the java.nio utility class it will create channels from these. For the main channel interface it will adapt these appropriately to the underlying socket here
	 * 
	 * @param socket
	 *            the socket that is abstracted by the socket channel instance
	 */
	SocketChannel(Socket socket) throws IOException {
		this.channel = new ChannelInterfaceAdapter();
		this.readable = Channels.newChannel(socket.getInputStream());
		this.writable = Channels.newChannel(socket.getOutputStream());
		this.socket = socket;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.nio.channels.Channel#close()
	 */
	public void close() throws IOException {
		if (this.socket != null) {
			socket.close();
		}
		if (this.channel != null) {
			this.channel.close();
		}
		if (this.readable != null) {
			this.readable.close();
		}
		if (this.writable != null) {
			this.writable.close();
		}
	}

	/**
	 * Returns the underlying socket for this socket channel
	 */
	public Socket getSocket() {
		return this.socket;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.nio.channels.Channel#isOpen()
	 */
	public boolean isOpen() {
		return this.channel.isOpen();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.nio.channels.ReadableByteChannel#read(java.nio.ByteBuffer)
	 */
	public int read(ByteBuffer buffer) throws IOException {
		return this.readable.read(buffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.nio.channels.WritableByteChannel#write(java.nio.ByteBuffer)
	 */
	public int write(ByteBuffer buffer) throws IOException {
		return this.writable.write(buffer);
	}

}
