/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EclipseExecutionComponentFactoryImpl.java,v 1.5 2008/03/20 18:49:50 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IEclipseExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * This specialization of ExecutionComponentFactoryImpl uses Eclipse
 * IConfigurationElements instead of String classnames when registering
 * execution components for local use.  This allows the use of Eclipse
 * classloaders, via createExecutableExtension(), so that execution components
 * need not live in predetermined Hyades plugins.  
 *  
 * @author jtoomey
 */
public class EclipseExecutionComponentFactoryImpl
	extends ExecutionComponentFactoryImpl
	implements IEclipseExecutionComponentFactory {

	/**
     * Get the component factory in this VM. Each VM has it's own factory.
     * Each factory shares the same, well known id and provides the bootstrap
     * for remote object instantiation.
     * 
     * @return
     */
    public static IExecutionComponentFactory getInstance(final ISession session) {
    	IExecutionComponentFactory factory;
    	synchronized(instances) {
    		factory=(IExecutionComponentFactory)instances.get(session);
    		if(factory==null || ! (factory instanceof EclipseExecutionComponentFactoryImpl)) {
    			factory=new EclipseExecutionComponentFactoryImpl(session);
    			instances.put(session, factory);
    			session.addExecutionComponentStateChangeListener(
        		        new IExecutionComponentStateChangeListener() {
        		            public void stateChanged(ExecutionComponentStateChangeEvent newState) {
        		                if (newState.getState() == IExecutionComponent.DEAD) {
        		                    ExecutionComponentFactoryImpl.removeInstance(session);
        		                }
                            }
        		        }
        			);
    		}
    		return factory;
    	}
    }

	/**
	 * @param session
	 */
	public EclipseExecutionComponentFactoryImpl(ISession session) {
		super(session);
	}

	/**
	 * This method adds registers the execution components specified by configElement.
	 * configElement is expected to have three attributes, named implClass, stubClass 
	 * and skeletonClass.  The implClass and StubClass will both be instantiated using
	 * createExecutableExtension(), and will thus use the Eclipse classloaders.  Since 
	 * the skeleton is only instantiated on the remote side, it will still use 
	 * Class.forName.NewInstance().
	 *  
	 * @author jtoomey
	 * @see org.eclipse.hyades.execution.core.IEclipseExecutionComponentFactory#addExecutionComponent(org.eclipse.core.runtime.IConfigurationElement)
	 */
	public synchronized void addExecutionComponent( IConfigurationElement configElement )
		throws ClassNotFoundException 
	{
		String type = configElement.getAttribute("name");//$NON-NLS-1$

		try {
			// First process the impl class
			
			/* Locate the impl class locally */
			configElement.createExecutableExtension("implClass");//$NON-NLS-1$
			
			/* If we have the impl class, lets try and find it remotely */
			ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{String.class, String.class},
				new Object[]{type, configElement.getAttribute("implClass")},
				"addExecutionComponent");//$NON-NLS-1$
	
			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable) rtnValue);
		
			// Next verify that we can locate the stub class
			configElement.createExecutableExtension("stubClass");//$NON-NLS-1$
			
			// Next locate the skeleton class on the remote machine.
			addSkeleton(type, configElement.getAttribute("skeletonClass"));//$NON-NLS-1$
			
			// Add the IConfigurationElement to the hashmap
			components.put(type, configElement);
		}
		catch ( CoreException exc )
		{
			throw new ClassNotFoundException(exc.getMessage());
		}

	}

	/** 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#createExecutionComponentByType(java.lang.String)
	 */
	public synchronized IExecutionComponent createExecutionComponentByType(String type) {
		ExecutionComponentStub stubInstance=null;
		IExecutionComponent implInstance = null;
		
		Object execComp = components.get(type);
		
		// If we have class objects for the execution components, create the
		// local stub with newInstance
		if ( execComp instanceof ClassRelationship)
		{
			ClassRelationship rel=(ClassRelationship)execComp;
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				implInstance=(IExecutionComponent)impl.newInstance();
				stubInstance=(ExecutionComponentStub)stub.newInstance();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		// If we have an Eclipse IConfigurationElement, create the local stub 
		// using Eclipse classloaders 
		else if ( execComp instanceof IConfigurationElement )
		{
			IConfigurationElement configElem = (IConfigurationElement) execComp;

			try {
				implInstance = (IExecutionComponent) configElem.createExecutableExtension("implClass");//$NON-NLS-1$
				stubInstance = (ExecutionComponentStub) configElem.createExecutableExtension("stubClass");//$NON-NLS-1$
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else 
		{
			throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_CANNOT_CONFIGURE_TYPE_);
		}
		
		/* if the local instance creation was succesful, complete the initialization
		 * and then create the remote instance */
		if(stubInstance!=null) {
			
			stubInstance.setDelegate(implInstance);
			stubInstance.setSessionContext(sessionContext);
			Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
			stubInstance.init();
			
			ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createExecutionComponentByType");//$NON-NLS-1$

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IExecutionComponent)) {
				throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_INCORRECT_TYPE_);
			}		
		}
		return stubInstance;
	}
	
	/**
	 * Create a remote object and return its stub.
	 * 
	 * @param type
	 * @return
	 */
	public synchronized IRemoteObject createRemoteObjectByType(String type){
		IRemoteObject stubInstance=null;
		Object implInstance = null;
		
		Object execComp = components.get(type);
		
		// If we have class objects for the execution components, create the
		// local stub with newInstance
		if ( execComp instanceof ClassRelationship)
		{
			ClassRelationship rel=(ClassRelationship)execComp;
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				implInstance= impl.newInstance();
				stubInstance=(IRemoteObject)stub.newInstance();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		// If we have an Eclipse IConfigurationElement, create the local stub 
		// using Eclipse classloaders 
		else if ( execComp instanceof IConfigurationElement )
		{
			IConfigurationElement configElem = (IConfigurationElement) execComp;

			try {
				implInstance = configElem.createExecutableExtension("implClass");//$NON-NLS-1$
				stubInstance = (IRemoteObject) configElem.createExecutableExtension("stubClass");//$NON-NLS-1$
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException(NLS.bind(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_FACTORY_CANNOT_CONFIGURE_TYPE_, type));
		}
		
		/* if the local instance creation was succesfull then create the remote instance */
		if(stubInstance!=null) {
			
			stubInstance.setDelegate(implInstance);
			stubInstance.setSessionContext(sessionContext);
			Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
			stubInstance.init();

			ReturnData invokeRtn = delegateRemoteCall(
				new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createRemoteObjectByType");//$NON-NLS-1$

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IRemoteObject)) {
				throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_NOT_INSTANCE_VALUE_);
			}		
		}
		return stubInstance;
	}


}

class ConfigurationElementRelationship {
	public IConfigurationElement impl;
	public IConfigurationElement stubClass;	
}
