/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionComponentFactoryImpl.java,v 1.8 2008/03/20 18:49:50 dmorris Exp $
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import java.io.IOException;
import java.util.HashMap;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.CallData;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.osgi.util.NLS;

/**
 * The execution component factory.
 * 
 * Each agent VM will have an instance of the factory. The factory acts as
 * it's own stub and is capable of remotely invoking the appropriate methods.
 * 
 * The factory's state is initialized when the class is loaded.
 * Something on remote agents needs to initiate the loading of this class.
 * A simple call to <code>RemoteFactory.getInstance()</code> will do.
 */
public class ExecutionComponentFactoryImpl implements IExecutionComponentFactory, IRemoteObject  {
	
	protected static HashMap instances=new HashMap();
	
	protected HashMap components=new HashMap();
	
	protected ISession sessionContext;
	
	protected ExecutionComponentFactoryImpl(ISession session){
		sessionContext=session;
		Marshaller.addInstanceToMap(getUniqueId(), this);
	}
	
	/**
     * Get the component factory in this VM. Each VM has it's own factory.
     * Each factory shares the same, well known id and provides the bootstrap
     * for remote object instantiation.
     * 
     * @return
     */
    public static IExecutionComponentFactory getInstance(final ISession session) {
    	IExecutionComponentFactory factory;
    	synchronized(instances) {
    		factory=(IExecutionComponentFactory)instances.get(session);
    		if(factory==null) {
    			factory=new ExecutionComponentFactoryImpl(session);
    			instances.put(session, factory);
    			session.addExecutionComponentStateChangeListener(
    		        new IExecutionComponentStateChangeListener() {
    		            public void stateChanged(ExecutionComponentStateChangeEvent newState) {
    		                if (newState.getState() == IExecutionComponent.DEAD) {
    		                    ExecutionComponentFactoryImpl.removeInstance(session);
    		                }
                        }
    		        }
    			);
    		}
    		return factory;
    	}
    }
	
	public ISession getSessionContext() {
		return sessionContext;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setSessionContext(org.eclipse.hyades.execution.core.ISession)
	 */
	public void setSessionContext(ISession session) {
		throw new UnsupportedOperationException(LocalResourceBundle.ExecutionComponentFactoryImpl_SESSION_CONTEXT_SHOULD_NOT_SET_);
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#getUniqueId()
	 */
	public Integer getUniqueId() {
		return new Integer(-1);
	}
	
	/**
	 * Remove the session execution component factory instance for the specified session
	 * 
	 * @param session the factory to remove from the instance map
	 */
	public static void removeInstance(ISession session) {
		synchronized(instances) {
			if (instances.containsKey(session)) {
			    ExecutionComponentFactoryImpl factory = (ExecutionComponentFactoryImpl) instances.get(session);
			    factory.sessionContext = null;
				instances.remove(session);
			}
		}
	}

	/** 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#createExecutionComponentByType(java.lang.String)
	 */
	public synchronized IExecutionComponent createExecutionComponentByType(String type) {
		ExecutionComponentStub stubInstance=null;
		
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				IExecutionComponent implInstance=(IExecutionComponent)impl.newInstance();
				stubInstance=(ExecutionComponentStub)stub.newInstance();
				stubInstance.setDelegate(implInstance);
				stubInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
				stubInstance.init();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_CANNOT_CONFIGURE_TYPE_);
		}
		
		/* if the local instance creation was succesfull then create the remote instance */
		if(stubInstance!=null) {
			
			ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createExecutionComponentByType");//$NON-NLS-1$

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IExecutionComponent)) {
				throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_INCORRECT_TYPE_);
			}		
		}
		return stubInstance;
	}
	
	/**
	 * Create a remote object and return its stub.
	 * 
	 * @param type
	 * @return
	 */
	public synchronized IRemoteObject createRemoteObjectByType(String type){
		IRemoteObject stubInstance=null;
		
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				Object implInstance=impl.newInstance();
				stubInstance=(IRemoteObject)stub.newInstance();
				stubInstance.setDelegate(implInstance);
				stubInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
				stubInstance.init();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException(NLS.bind(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_FACTORY_CANNOT_CONFIGURE_TYPE_, type));
		}
		
		/* if the local instance creation was succesfull then create the remote instance */
		if(stubInstance!=null) {
			
			ReturnData invokeRtn = delegateRemoteCall(
				new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createRemoteObjectByType");//$NON-NLS-1$

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IRemoteObject)) {
				throw new RemoteInvocationException(LocalResourceBundle.EclipseExecutionComponentFactoryImpl_NOT_INSTANCE_VALUE_);
			}		
		}
		return stubInstance;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addExecutionComponent(java.lang.String, java.lang.String)
	 */
	public void addExecutionComponent(String type, String implClass) throws ClassNotFoundException {
		
	
		/* Locate the impl class locally */
		Class clazz=Class.forName(
			implClass, true, this.getClass().getClassLoader()/*new UpdateablePathClassLoader(getClass().getClassLoader())*/);
		
		/* If we have the impl class, lets try and find it remotely */
		ReturnData invokeRtn = delegateRemoteCall(
		new Class[]{String.class, String.class},
			new Object[]{type, implClass},
			"addExecutionComponent");//$NON-NLS-1$

		invokeRtn.getReturnValue();
		if(invokeRtn.isError())
			throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
		
		/* add this class to our hashmap */
		ClassRelationship rel=new ClassRelationship();
		rel.impl=clazz;
		components.put(type, rel);
		
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addStub(java.lang.Class, java.lang.String)
	 */
	public void addStub(String type, String stubClass) throws ClassNotFoundException {
		/* look for the stub class locally */
		Class clazz=Class.forName(
			stubClass, true, this.getClass().getClassLoader()/*new UpdateablePathClassLoader(getClass().getClassLoader())*/);
		
		/* Update the components hashmap.  */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel==null) {
			throw new ClassNotFoundException(LocalResourceBundle.ExecutionComponentFactoryImpl_CLASS_NOT_SPECIFIED_);
		} 
		
		rel.stubClass=clazz;
	
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addSkeleton(java.lang.Class, java.lang.String)
	 */
	public void addSkeleton(String type, String skeletonClass) throws ClassNotFoundException {
	
		/* Check the remote side for the class */
		ReturnData invokeRtn = delegateRemoteCall(
		new Class[]{String.class, String.class},
			new Object[]{type, skeletonClass},
			"addSkeleton");//$NON-NLS-1$

		invokeRtn.getReturnValue();
		if(invokeRtn.isError())
			throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
	
	}


	/**
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#delegateRemoteCall()
	 */
	public ReturnData delegateRemoteCall(Class[] argTypes, Object[] callArgs, String call)
		throws RemoteInvocationException {

		// Gather the necessary data, marshal it and submit it for remote
		// invocation.
		CallData callData = new CallData(getUniqueId(), argTypes, callArgs, call);
	
		/* Create our custom command */
		BinaryCustomCommand command = new BinaryCustomCommand();
		try {
			command.setData(Marshaller.marshalMethodCall(callData));
		}
		catch ( IOException e ) {
			throw new RemoteInvocationException(e);
		}
		
		try {
			((SessionStub)sessionContext).invokeRemote(command);
			
		}
		catch(ClassCastException e) {
			
		}
	
		// Wait for notification that the remote call has returned, unmarshal
		// the return data, return the result of the remote call.
		ReturnData returnData = null;
		
		try {
			// bugzilla 162605, added new method for proper 
			// synchronization and delivery of ReturnData
			returnData = Marshaller.waitForReturnDataWithTimeout(callData);

		} catch (InterruptedException e) {
			// Timed out waiting for response from remote call.
			throw new RemoteInvocationException(e);
		}

		return returnData;
	}
	
	/**
	 * This implementation returns the factory. The factory is its own delegate.
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#getDelegate()
	 */
	public Object getDelegate() {
		return this;
	}

	/**
	 * This implementation does nothing. The factory is its own delegate.
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setDelegate(java.lang.Object)
	 */
	public void setDelegate(Object delegate) {}

	/**
	 * This implementation does nothing.
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {}
	
	protected void finalize() throws Throwable {
	    this.sessionContext = null;
	    super.finalize();
	}
	
}

class ClassRelationship {
	public Class impl;	
	public Class stubClass;
}
