/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NodeImpl.java,v 1.27 2008/03/20 18:49:50 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.Principal;
import java.util.Properties;

import org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter;
import org.eclipse.hyades.execution.core.DaemonConnectException;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.INodeExtended;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.util.ConnectionSpecifier;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.NoSuchApplicationException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessActiveException;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.security.DuplicateUserException;
import org.eclipse.hyades.internal.execution.security.LoginFailedException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;
import org.eclipse.ui.PlatformUI;

/**
 * Node implementation for the execution framework, now supports the ability to
 * connect to a connect given a connection specifier.
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Scott E. Schneider
 * @version February 28, 2008
 * @since   January 31, 2008
 */
public class NodeImpl implements INodeExtended {

	/* whether we are building in debug mode */
	private static final boolean DEBUG = false;

	private boolean connected = false;

	/* The FileManager used by this Node */
	IFileManagerExtended fileManager;

	/* The name of the node */
	private final String name;

	/**
	 * The underlying node that retains the connection
	 */
	public Node node;

	/* The objects for synchronizing our session handshake */
	private Object sessionLock = new Object();

	/**
	 * Constructs a node implementation given the specified name
	 */
	public NodeImpl(String name) {
		this.name = name;
	}

	private ISession _connect(String daemonType, final Principal principal) throws UnknownDaemonException,
			DaemonConnectException, UnknownHostException {

		/*
		 * Prepare to use either the headless or UI Agent Controller connector 
		 * service depending if the workbench is running, an automation
		 * service used to connect to the Agent Controller (secure or unsecure)
		 * and returns back the node once connected.
		 */
		boolean isWorkbenchRunning = PlatformUI.isWorkbenchRunning();
		AutomationClientAdapter client = new AutomationClientAdapter();

		// Configure state to pass into the connect service
		Properties properties = new Properties();
		properties.setProperty("host", this.name);//$NON-NLS-1$
		properties.setProperty("port", daemonType);//$NON-NLS-1$
		if(isWorkbenchRunning){
			properties.put("showErrors", new Boolean(true));//$NON-NLS-1$
		}
		MutableObject mutableNode = new MutableObject();
		properties.put("mutableNode", mutableNode);//$NON-NLS-1$

		// Execute either the headless or UI Agent Controller connector 
		// service depending if the workbench is running to obtain the node:
		try {

			if(isWorkbenchRunning){
				client.execute("org.eclipse.hyades.ui.connector", properties);//$NON-NLS-1$
			}
			else{
				client.execute("org.eclipse.tptp.platform.common.headlessConnector", properties);//$NON-NLS-1$		
			}
			this.node = (Node) mutableNode.getAndClear();

			// If node is null throw a daemon connect exception
			if (this.node == null) {
				throw new DaemonConnectException();
			}

		} catch (Throwable t) {

			/*
			 * Fall-back to legacy node connection method (unsecure only) using
			 * the connection method previous to version 4.1
			 */
			this.node = this.connectUsingLegacyMethod(daemonType, principal);

			// If node is still null throw exception
			if (this.node == null) {
				throw new DaemonConnectException(); // need to handle this
			}

		}

		// Return null session to indicate exception and already handledl
		if (this.node == null) {
			return null;
		}

		/* Create a process. This will be our session hosting process */
		Process rprocess = ProcessFactory.createProcess(this.node, "HyadesTestingSession");//$NON-NLS-1$

		/*
		 * if we are running in debug mode redirect the stdout/stderr of the
		 * remote process
		 */
		if (DEBUG) {
			captureConsoleOutput(rprocess);
		}

		/*
		 * Create our agent for communication with the remote session. We add
		 * the agent here with the understanding that it will be instantiated on
		 * the remote process by the RemoteSession. The autoattach means we will
		 * grab an exclusive lock on it at registration time in the Daemon
		 * process.
		 */
		Agent ragent = AgentFactory.createAgent(rprocess, "RemoteSession", "HyadesSession");//$NON-NLS-1$
		ragent.setAutoAttach(true);

		/*
		 * Create our ISession and give it the agent it will use to communicate
		 * with
		 */
		ISession sessionInstance = new SessionStub(new SessionImpl(this, ragent));

		Marshaller.addInstanceToMap(((SessionStub) sessionInstance).getUniqueId(), sessionInstance);

		/*
		 * As a parameter to the process we will provide the uniqueId of the
		 * session stub so that the proper associatione between the session
		 * objects can occur
		 */
		try {
			rprocess.setParameters("org.eclipse.hyades.execution.remote.NodeImpl "//$NON-NLS-1$
					+ ((SessionStub) sessionInstance).getUniqueId().toString());
		} catch (ProcessActiveException e) {
			/* We can ignore this as we have not launched the process yet */
		}

		/* Add a process listener */
		ProcessListener processListener = new ProcessListener() {

			public void processExited(Process process) {
				/*
				 * If the process is exiting and we are still locked below then
				 * there were problems instantiating the agent in the remote
				 * process. Allow the method to return.
				 */
				releaseSessionLock();
			}

			public void processLaunched(Process process) {
			}

		};

		rprocess.addProcessListener(processListener);

		/*
		 * Add a state change listener to the ISession so we know when the
		 * session is active and good to be used
		 */
		IExecutionComponentStateChangeListener stateChangeListener = new IExecutionComponentStateChangeListener() {
			public void stateChanged(ExecutionComponentStateChangeEvent event) {
				/*
				 * When we go to active state make sure we signal the session
				 * lock
				 */
				if (event.getState() == IExecutionComponent.READY) {
					releaseSessionLock();
				}
			}

		};

		sessionInstance.addExecutionComponentStateChangeListener(stateChangeListener);

		/* We are all setup now. Launch the process. */
		try {
			rprocess.launch();

			/* Confirm our process is running correctly */
			rprocess.getProcessId();
		} catch (InactiveProcessException e) {
			/* Our process failed to launch */
			// Cascade the exception
			this.cleanup(this.node, rprocess, sessionInstance, processListener, stateChangeListener);
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		} catch (NoSuchApplicationException e) {
			/* There is no alias in the config file */
			// Cascade the exception
			this.cleanup(this.node, rprocess, sessionInstance, processListener, stateChangeListener);
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		} catch (ProcessActiveException e) {
			/* We just created the process above and this should not be posible. */
			// Cascade the exception
			this.cleanup(this.node, rprocess, sessionInstance, processListener, stateChangeListener);
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		} catch (NotConnectedException e) {
			/* This will not occur */
			// Cascade the exception
			this.cleanup(this.node, rprocess, sessionInstance, processListener, stateChangeListener);
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		}

		/* Block until the session is up and running completely */
		while (!connected) {
			synchronized (sessionLock) {
				try {
					sessionLock.wait();
				} catch (InterruptedException e) {
					/* We can ignore this */
				}
			}
		}

		rprocess.removeProcessListener(processListener);
		sessionInstance.removeExecutionComponentStateChangeListener(stateChangeListener);

		return sessionInstance;
	}

	/**
	 * This method is provided strictly to simplify debugging the stdout and
	 * stderr of the remote process hosting the session. This will take the
	 * remote console output and dump it to the local processes stdout stream.
	 * 
	 * @param process -
	 *            the remote process to redirect its stdout/stderr to the local
	 *            stdout.
	 */
	private void captureConsoleOutput(Process process) {
		Console console = process.getConsole();
		console.setDataProcessor(new DataProcessor() {

			public void incommingData(byte[] buffer, int length, InetAddress peer) {
				System.out.print("====>");
				System.out.print(new String(buffer, 0, length));
				System.out.flush();

			}

			public void incommingData(char[] buffer, int length, InetAddress peer) {
				System.out.print("====>");
				System.out.print(new String(buffer, 0, length));
				System.out.flush();

			}

			public void invalidDataType(byte[] data, int length, InetAddress peer) {

			}

			public void waitingForData() {

			}

		});

	}

	/**
	 * Cleanup attempt to launch session hosting process
	 * 
	 * @param node
	 *            node that was connected to
	 * @param process
	 *            process created in attempt to launch
	 * @param sessionInstance
	 *            session instance created
	 * @param processListener
	 *            process listener registered
	 * @param stateChangeListener
	 *            state change listener registered
	 */
	private void cleanup(Node node, Process process, ISession sessionInstance, ProcessListener processListener,
			IExecutionComponentStateChangeListener stateChangeListener) {
		try {
			process.removeProcessListener(processListener);
			Marshaller.removeInstanceFromMap(((SessionStub) sessionInstance).getUniqueId());
			sessionInstance.removeExecutionComponentStateChangeListener(stateChangeListener);
			this.node.getConnection().disconnect();
		} catch (Throwable t) {
			// Do not let anything escape but log to console for now
			t.printStackTrace();
		}
	}

	/**
	 * Connects to the node using this connection specifier
	 * 
	 * @param specifier
	 *            the specifier used for connection purposes, uniquely
	 *            identifies an agent controller instance, only the port is read
	 *            currently, the host is specified using the node's name at
	 *            construction time, the host for this node is maintained
	 *            immutable
	 * @return the session to this agent controller instance
	 */
	public ISession connect(ConnectionSpecifier specifier) throws UnknownDaemonException, DaemonConnectException,
			UnknownHostException {
		return this.connect(String.valueOf(specifier.getPort()), null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.INode#connect(java.lang.String,
	 *      java.security.Principal)
	 */
	public ISession connect(String daemonType, Principal principal) throws UnknownDaemonException, DaemonConnectException,
			UnknownHostException {

		ISession sessionInstance = this._connect(daemonType, principal);

		if (sessionInstance != null) {
			try {
				this.fileManager = FileManagerFactory.getInstance().create(this.node.getConnection());
			} catch (ServerNotAvailableException e) {
				DaemonConnectException exc = new DaemonConnectException();
				exc.initCause(e);
				throw exc;
			}
		}

		/* Return the ISession for use */
		return sessionInstance;
	}

	public ISession connect(String daemonType, Principal principal, int iterations) throws UnknownDaemonException,
			DaemonConnectException, UnknownHostException {

		ISession sessionInstance = this._connect(daemonType, principal);

		if (sessionInstance != null) {
			try {
				this.fileManager = FileManagerFactory.getInstance().createTimed(this.node.getConnection(), iterations);
			} catch (ServerNotAvailableException e) {
				DaemonConnectException exc = new DaemonConnectException();
				exc.initCause(e);
				throw exc;
			}
		}

		/* Return the ISession for use */
		return sessionInstance;
	}

	/**
	 * Obtains and connects to node using the legacy method (does not support
	 * secure connections)
	 * 
	 * @param daemonType
	 *            the port to connect to
	 * @param principal
	 *            the security principal to use
	 * @return the node connected to
	 * @throws Exception
	 */
	public Node connectUsingLegacyMethod(String daemonType, Principal principal) throws UnknownDaemonException,
			DaemonConnectException, UnknownHostException {
		Node node = null;
		try {
			if (principal == null) {
				node = NodeFactory.createNode(name);
			} else {
				node = NodeFactory.createNode(name, principal);
			}
		} catch (DuplicateUserException e) {
			/* The prinical already exists */
			throw e;
		} catch (UnknownHostException e) {
			/* We could not resolve the remote host */
			throw e;
		}

		boolean isConnected = false;
		if(node instanceof org.eclipse.hyades.internal.execution.local.control.NodeImpl) {
			org.eclipse.hyades.internal.execution.local.control.NodeImpl nodeImpl = (org.eclipse.hyades.internal.execution.local.control.NodeImpl)node;
			isConnected = nodeImpl.isConnected() && nodeImpl.isUserAuthenticated();
		}

		if(!isConnected) {
			/* Connect on the proper port. */
			try {
				node.connect(Integer.parseInt(daemonType));
			} catch (SecureConnectionRequiredException e) {
				/*
				 * We need s secure connection to the node. Collect the user
				 * information and add a user and keystoremanager to the node.
				 */
				throw e;
			} catch (LoginFailedException e) {
				/*
				 * We are trying to connect with the wrong authentication
				 * credentials. Update the username and password and try again
				 */
				throw e;
			} catch (AgentControllerUnavailableException e) {
				/*
				 * there are several subclasses of this exception we need to handle
				 * properly.
				 */
				throw e;
			}
		}

		return node;
	}

	/**
	 * Retrieves the file manager associated with this node
	 * 
	 * @return file manager is returned and is potentially a file manager
	 *         extended instance
	 */
	public IFileManager getFileManager() {
		return fileManager;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.INode#getName()
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the underlying node. This method is for internal use only.
	 * 
	 * @return the underlying node object
	 */
	public Node getUnderlyingNode() {
		return this.node;
	}

	/**
	 * Release the session lock
	 */
	private void releaseSessionLock() {
		synchronized (sessionLock) {
			connected = true;
			sessionLock.notify();
		}
	}

}
