/**********************************************************************
Copyright (c) 2006, 2008 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: RemoteObjectStub.java,v 1.6 2008/03/20 18:49:50 dmorris Exp $

Contributors:
 IBM Corporation - initial implementation
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.execution.local;

import java.io.IOException;

import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.CallData;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;

/**
 * This is the parent of the hierarchy of stubs -- objects that handle remote
 * invocation semantics for other objects.
 */
public abstract class RemoteObjectStub implements IRemoteObject {
	
	/**
	 * a unique identifier used to reference this stub (and hence the
	 * target object) 
	 */
	protected Integer uniqueId = new Integer(hashCode());
	
	/**
	 * Remote objects are created in the context of the same session as the
	 * factory that creates them.
	 */
	protected ISession sessionContext;
	
	/**
	 * the object for which this stub is providing remote execution semantics
	 */
	protected Object delegate;
	
	public RemoteObjectStub(){}
	
	/**
	 * get the id unique to this instance
	 * @return
	 */
	public Integer getUniqueId() {
		return uniqueId;
	}

	/**
	 * Delegate a remote method call to the target instance. This method
	 * orchestrates the marshalling of the method call and the remote invocation
	 * of the method against the target.
	 * 
	 * @param callArgs
	 * @param call
	 * @return
	 * @throws RemoteInvocationException
	 */
	public ReturnData delegateRemoteCall(Class[] argTypes, Object[] callArgs, String call)
		throws RemoteInvocationException {

		// Gather the necessary data, marshal it and submit it for remote
		// invocation.
		CallData callData = new CallData(getUniqueId(), argTypes, callArgs, call);
	
		/* Create our custom command */
		BinaryCustomCommand command = new BinaryCustomCommand();
		try {
			command.setData(Marshaller.marshalMethodCall(callData));
		}
		catch ( IOException e ) {
			throw new RemoteInvocationException(e);
		}
		
		/* Find our session to send the invocation over */		
		try {
			SessionStub sessStub = ((SessionStub)getSessionContext());
			((SessionImpl)sessStub.getDelegate()).invokeRemote(command);
			
		}
		catch(ClassCastException e) {
			throw new RemoteInvocationException(LocalResourceBundle.RemoteObjectStub_CLASS_NOT_A_CHILD_);
		}
		
		// Wait for notification that the remote call has returned, unmarshal
		// the return data, return the result of the remote call.
		ReturnData returnData = null;
		
		
		try {
			// bugzilla 162605, added new method for proper 
			// synchronization and delivery of ReturnData
			returnData = Marshaller.waitForReturnDataWithTimeout(callData);
			
		} catch (InterruptedException e) {
			// Timed out waiting for response from remote call.
			throw new RemoteInvocationException(e);
		}
		 
		

		return returnData;
	}
	
	/**
	 * Get the instance for which this stub is providing remote invocation
	 * semantics.
	 *  
	 * @return
	 */
	public Object getDelegate() {
		return delegate;
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setDelegate(java.lang.Object)
	 */
	public void setDelegate(Object delegate) {
		this.delegate = delegate;
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {}
	
	/**
	 * Get the local session context for this remote object.
	 * Remote objects are created in the context of a session. The session
	 * context of a remote object is the same as that of the factory that
	 * created the remote object.
	 * 
	 * @return a <code>SessionImpl</code> instance
	 * 
	 * @see org.eclipse.hyades.execution.local.SessionImpl
	 */
	public ISession getSessionContext() {
		return sessionContext;
	}
	
	/**
	 * Set the session context for this remote object. The session context
	 * should be set to the same as that of the factory that created the
	 * remote object.
	 */
	public void setSessionContext(ISession session) {
		sessionContext = session;
	}

}
