/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SessionContextToSessionMap.java,v 1.2 2005/02/25 22:17:09 hleung Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.local;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.hyades.execution.core.ISession;

/**
 * This singleton caches sessions keyed on on a session context object. There
 * are operational static methods that are kept for compatability purposes only
 * and are now deprecated to be removed in the future. One static method to
 * retrieve the singleton instance will be maintained in future versions per the
 * singleton design pattern.
 * 
 * This singleton is thread-safe whether used via the singleton exposed session
 * cache access methods or the deprecated class session cache access methods.
 * The deprecated class session cache access methods simply forward the calls on
 * to their synchronized singleton exposed session cache access methods.
 * 
 * @author Scott E. Schneider
 */
public final class SessionContextToSessionMap {

    /**
     * Holds a single instance of the session context to session map class; this
     * will be initialized before the get instance method is invoked for the
     * first time enabling lazy intantiation of this singleton instance.
     */
    private static final SessionContextToSessionMap singleton = new SessionContextToSessionMap();

    /**
     * Adds the given session using the specified context as the key
     * 
     * @param context
     *            the key used for later retrieval of the session
     * @param session
     *            the session to be associated with the provided session context
     * @deprecated use the public instance methods instead of the public class
     *             methods
     */
    public static void addSession(SessionContext context, ISession session) {
        SessionContextToSessionMap.getInstance().addSessionEntry(context,
                session);
    }

    /**
     * Provides access to the singleton instance of this class
     * 
     * @return the singleton instance is returned
     */
    public static SessionContextToSessionMap getInstance() {
        return SessionContextToSessionMap.singleton;
    }

    /**
     * Returns the session associated with the given context as stored in the
     * internal cache
     * 
     * @param context
     *            the context that is the key for the session
     * @return the session mapped to the session context provided
     * @deprecated use the public instance methods instead of the public class
     *             methods
     */
    public static ISession getSession(SessionContext context) {
        return SessionContextToSessionMap.getInstance()
                .getSessionEntry(context);
    }

    /**
     * Removes the specified session from the internal cache
     * 
     * @param session
     *            the session to remove
     * @return true if it was successfully removed
     * @deprecated use the public instance methods instead of the public class
     *             methods
     */
    public static boolean removeSession(ISession session) {
        return SessionContextToSessionMap.getInstance().removeSessionEntry(
                session);
    }

    /**
     * Remove the session associated with the specified session context
     * 
     * @param context
     *            the context which is the key in identifying the session to
     *            remove
     * @deprecated use the public instance methods instead of the public class
     *             methodsf
     */
    public static void removeSession(SessionContext context) {
        SessionContextToSessionMap.getInstance().removeSessionEntry(context);
    }

    /**
     * Internally maintained hashtable for session to context map.
     */
    private Hashtable sessionFromContext = new Hashtable();

    /**
     * Limit construction of instance to private only, access singleton instance
     * via static get instance method.
     */
    private SessionContextToSessionMap() {
    }

    /**
     * Adds an entry in the map using the context as the key and the session as
     * the value.
     * 
     * @param context
     *            the key in the map for this entry
     * @param session
     *            the session in the map for this entry
     */
    public synchronized void addSessionEntry(SessionContext context,
            ISession session) {
        this.sessionFromContext.put(context, session);
    }

    /**
     * Gets a session entry given the session context, the session context is
     * used as the key in finding the session object.
     * 
     * @param context
     *            the key to retrieve the session
     * @return the session associated with the given context
     */
    public synchronized ISession getSessionEntry(SessionContext context) {
        return (ISession) this.sessionFromContext.get(context);
    }

    /**
     * Remove the session from the map given the session object
     * 
     * @param session
     *            the session to remove from the cache
     * @return indicates whether the session was removed
     */
    public synchronized boolean removeSessionEntry(ISession session) {
        Enumeration enumeration = this.sessionFromContext.keys();
        for (SessionContext context = null; enumeration.hasMoreElements();) {
            context = (SessionContext) enumeration.nextElement();
            ISession value = this.getSessionEntry(context);
            if (value != null) {
                if (value.equals(session)) {
                    this.removeSessionEntry(context);
                    break;
                }
            }
        }
        return false;
    }

    /**
     * Remove the session from the map given the specified context
     * 
     * @param context
     *            the key used to identify the session to remove
     */
    public synchronized void removeSessionEntry(SessionContext context) {
        this.sessionFromContext.remove(context);
    }
}