/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Console.java,v 1.3 2008/04/14 21:29:04 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;

import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.local.control.Node;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;

public class Console extends Thread implements Constants {

	protected ServerSocket _sock;

	protected DataProcessor _processor;

	protected long _ip = 0;

	protected long _port = 0;

	protected Node _node;

	private boolean _started = false;

	private boolean _valid = true;

	private Socket _activeConnection;

	/**
	 * Console constructor comment.
	 */
	public Console() {
		super();
		/* Don't block the VM from exiting */
		this.setName("Console"); /* 9707 *///$NON-NLS-1$
		this.setDaemon(true);
	}

	/**
	 * Construct a console that knows about the node that will try to
	 * communicate with it
	 */
	public Console(Node node) {
		this();
		this._node = node;
	}

	/**
	 * Console constructor comment.
	 * 
	 * @param name
	 *            java.lang.String
	 */
	public Console(String name) {
		super(name);
		/* Don't block the VM from exiting */
		this.setDaemon(true);
	}

	/**
	 * Console constructor comment.
	 * 
	 * @param group
	 *            java.lang.ThreadGroup
	 * @param name
	 *            java.lang.String
	 */
	public Console(ThreadGroup group, String name) {
		super(group, name);
		/* Don't block the VM from exiting */
		this.setDaemon(true);
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:44:15 PM)
	 * 
	 * @return long
	 */
	public long getIP() throws ConsoleNotStartedException {
		if (_ip == 0) {
			throw new ConsoleNotStartedException();
		}
		return _ip;
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:43:59 PM)
	 * 
	 * @return long
	 */
	public long getPort() throws ConsoleNotStartedException {
		if (_port == 0) {
			throw new ConsoleNotStartedException();
		}
		return _port;
	}

	/**
	 * Return the server socket used by the console thread
	 * 
	 * @return java.net.ServerSocket
	 * @throws ConsoleNotStartedException
	 */
	public ServerSocket getServerSocket() throws ConsoleNotStartedException {
		if (_sock == null) {
			throw new ConsoleNotStartedException(); /* 9707 */
		}

		return _sock;
	}
	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:43:25 PM)
	 */
	public void run() {
		byte[] buffer = new byte[1024];
		short port = 0/* DATA_PORT_NUM_CLIENT */;
		boolean started = false;
		boolean complete;

		port = 0;

		/* Start the server */
		while (!started) {
			try {
				started = true;
				_sock = new ServerSocket(port, 1);
			} catch (Exception e) {
				if(port != 0) {
					port++;
				}
				started = false;
			}
		}

		/* Store the address information of this console */
		_port = _sock.getLocalPort();
		InetAddress localHost = null;
		try {
			localHost = InetAddress.getLocalHost();
			byte[] bytes = localHost.getAddress();
			_ip = (long) (bytes[3] << 24 & 0xff000000) 
				| (long) (bytes[2] << 16 & 0x00ff0000) 
				| (long) (bytes[1] << 8 & 0x0000ff00) 
				| (long) (bytes[0] & 0x000000ff);
		} catch (UnknownHostException e) {
			/* Shouldn't happen as we are looking up the localhost */
		}

		/**
		 * If server cannot reach to client do not start console server,
		 * basically the launch process command will use 0 as IP and port which
		 * indicates no console traffic will be brought back to the client side
		 * (and therefore no chance for the firewall to block it since the
		 * server reach command is indicating this is what will happen if
		 * attempted) -- if server can indeed reach to client then proceed
		 * typically (using console server)
		 */
		boolean serverCanReach = false;
		try {
			org.eclipse.hyades.internal.execution.local.control.NodeImpl node;
			node = new org.eclipse.hyades.internal.execution.local.control.NodeImpl(_node.getName(), _node.getAllInetAddresses());
			node.connect(_node.getConnection().getPort());

			IFileManagerExtended fileManager = FileManagerFactory.getInstance().create(node.getConnection());
			serverCanReach = fileManager.determineServerReach(localHost.getHostAddress(), (int) this._port);
			node.getConnection().disconnect();
		} catch (Exception e) {
			e.printStackTrace();
		}

		/* Inform the start method we are up */
		synchronized (this) {
			_started = true;
			this.notify();
		}

		if (serverCanReach) {

			while (_valid) {
				InputStream is = null;
				_activeConnection = null;
				try {
					_activeConnection = _sock.accept();
					is = _activeConnection.getInputStream();
					_activeConnection.setSoTimeout(1000);
					complete = false;
				} catch (Exception e) {
					/* The server socket is toast, return */
					return;
				}
				while (!complete) {
					int length = 0;
					try {
						length = is.read(buffer);
					} catch (InterruptedIOException e) {
						/* Read timeout, don't want to wait too long */
						if (_processor != null) {
							_processor.waitingForData();
						}
					} catch (IOException e) {
						try {
							_activeConnection.close();
						} catch (IOException e1) {
						}
						complete = true;
					}

					/* Is the connection closed? */
					if (length < 0) {
						try {
							_activeConnection.close();
						} catch (IOException e) {
						}
						complete = true;
					} else if (length > 0) {
						if (_processor != null) {
							_processor.incommingData(buffer, length, _activeConnection.getInetAddress());
						}
					}

					/* Have we been asked to stop. */
					if (!_valid) {
						try {
							_activeConnection.close();
						} catch (IOException e1) {
						}
						complete = true;
					}
				}
			}

		} else {

			/*
			 * If server cannot reach back, close new server socket and reset ip
			 * and port to zero, these values will be attached to the launch
			 * process command, if they are zero then server will not attempt to
			 * connect to console started in this object
			 */
			this._ip = 0;
			this._port = 0;
			this.close();

		}

	}

	public void setDataProcessor(DataProcessor processor) {
		_processor = processor;

	}

	public DataProcessor getDataProcessor() {
		return _processor;
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 5:39:25 PM)
	 */
	public void start() {
		/* We need to wait until the server is fully up */
		synchronized (this) {
			_valid = true;
			super.start();
			do {
				try {
					this.wait();
				} catch (InterruptedException e) {
				}
			} while (!_started);
		}
	}

	/**
	 * Insert the method's description here. Creation date: (5/28/01 8:01:54 PM)
	 * 
	 * @param data
	 *            java.lang.String
	 */
	public void write(String data) {
		if (_activeConnection != null) {
			try {
				_activeConnection.getOutputStream().write(data.getBytes("UTF-8")); // Bug 73074
				                                                                   //$NON-NLS-1$ 
			} catch (Exception e) {
			}
		}
	}

	/* Closes this console */
	public void close() {
		if (_sock != null) {
			try {
				_sock.close();
			} catch (IOException e) {
			}
		}
		_valid = false;
	}

}
