/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LocalConsole.java,v 1.2 2008/03/20 18:49:51 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.UnknownHostException;

public class LocalConsole extends Console {
	private ConsoleOutReader _outReader = null;
	private ConsoleErrReader _errReader = null;
	private ConsoleWaitingThread _waitThread = null;
	private DataProcessor _processor = null;
	private boolean _done = false;
	private InputStream _stdout;
	private InputStream _stderr;
	private OutputStream _stdin;

	private int _bytesWritten = 0;

	/*
	 * Allows the console threads to wait until the streams are established
	 */
	private Object _processor_lock = new Object();
	private Object _stdin_lock = new Object();
	private Object _stdout_lock = new Object();
	private Object _stderr_lock = new Object();

	public LocalConsole() {
		this("Local Console");//$NON-NLS-1$
	}

	public LocalConsole(String name) {
		super(name);
		this.setDaemon(true);
	}

	public LocalConsole(ThreadGroup group, String name) {
		super(group, name);
		this.setDaemon(true);
	}

	private synchronized int getConsoleBytesWritten() {
		return _bytesWritten;
	}

	public long getIP() {
		return 0;
	}

	public long getPort() {
		return 0;
	}

	public ServerSocket getServerSocket() {
		return null;
	}

	public void run() {
		_outReader = new ConsoleOutReader();
		_outReader.start();
		_errReader = new ConsoleErrReader();
		_errReader.start();
		_waitThread = new ConsoleWaitingThread();
		_waitThread.start();
	}

	public void start() {
		run();
	}

	private synchronized void setConsoleBytesWritten(int bytes) {
		_bytesWritten = bytes;
	}

	public void setDataProcessor(DataProcessor processor) {
		synchronized (_processor_lock) {
			_processor = processor;
			_processor_lock.notifyAll();
		}
	}

	public void setStdOut(InputStream stdout) {
		synchronized (_stdout_lock) {
			_stdout = stdout;
			_stdout_lock.notifyAll();
		}
	}

	public void setStdErr(InputStream stderr) {
		synchronized (_stderr_lock) {
			_stderr = stderr;
			_stderr_lock.notifyAll();
		}
	}

	public void setStdIn(OutputStream stdin) {
		synchronized (_stdin_lock) {
			_stdin = stdin;
			_stdin_lock.notifyAll();
		}
	}

	public DataProcessor getDataProcessor() {
		return _processor;
	}

	public void write(String data) {
		if (_stdin == null) {
			synchronized (_stdin_lock) {
				try {
					_stdin_lock.wait();
				} catch (InterruptedException e) {
					return;
				}
			}
		}

		byte b[];
		try {
			b = data.getBytes("UTF-8");//$NON-NLS-1$
		} catch (UnsupportedEncodingException e) {
			b = data.getBytes();
		}

		try {
			_stdin.write(b);
			_stdin.flush();
		} catch (IOException e) {
			return;
		}
	}

	/* Closes this console */
	public void close() {
		_done = true;
		if (_outReader != null) {
			_outReader.interrupt();
		}
		if (_errReader != null) {
			_errReader.interrupt();
		}
		if (_waitThread != null) {
			_waitThread.interrupt();
		}
	}

	class ConsoleOutReader extends Thread {
		private byte[] buffer = new byte[1024];

		public ConsoleOutReader() {
			setName("Console Output Reader");//$NON-NLS-1$
		}

		public void run() {
			synchronized (_stdout_lock) {
				if (_stdout == null) {
					try {
						_stdout_lock.wait();
					} catch (InterruptedException e) {
					}
				}
			}

			while (!_done) {
				int byteRead = -1;
				try {
					byteRead = _stdout.read(buffer);
				} catch (IOException e1) {
				}

				if (byteRead == -1) {
					_done = true;
				} else {
					try {
						synchronized (_processor_lock) {
							if (_processor == null) {
								try {
									_processor_lock.wait(30000);
								} catch (InterruptedException e) {
								} // 30 sec max?
							}
							if (_processor != null) {
								_processor.incommingData(buffer, byteRead, InetAddress.getLocalHost());
								setConsoleBytesWritten(byteRead);
							} else {
								System.out.print(new String(buffer, 0, byteRead));
							}
						}
					} catch (UnknownHostException e) {
						// Error handling console
					}
				}
			}
		}
	}

	class ConsoleErrReader extends Thread {
		private byte[] buffer = new byte[1024];

		public ConsoleErrReader() {
			setName("Console Error Reader");//$NON-NLS-1$
		}

		public void run() {
			synchronized (_stderr_lock) {
				if (_stderr == null) {
					try {
						_stderr_lock.wait();
					} catch (InterruptedException e) {
					}
				}
			}

			while (!_done) {
				int byteRead = -1;
				try {
					byteRead = _stderr.read(buffer);
				} catch (IOException e1) {
				}

				if (byteRead == -1) {
					_done = true;
				} else {
					try {
						synchronized (_processor_lock) {
							if (_processor == null) {
								try {
									_processor_lock.wait(30000);
								} catch (InterruptedException e) {
								} // 30 sec max?
							}
							if (_processor != null) {
								_processor.incommingData(buffer, byteRead, InetAddress.getLocalHost());
								setConsoleBytesWritten(byteRead);
							} else {
								System.err.print(new String(buffer, 0, byteRead));
							}
						}
					} catch (UnknownHostException e) {
						// Error handling console
					}
				}
			}
		}
	}

	class ConsoleWaitingThread extends Thread {
		public ConsoleWaitingThread() {
			setName("Console Waiting Notification");//$NON-NLS-1$
		}

		public void run() {
			while (!_done) {
				try {
					sleep(3000); // 3 sec console polling
				} catch (Exception e) {
				}

				if (getConsoleBytesWritten() == 0) {
					synchronized (_processor_lock) {
						if (_processor != null) {
							_processor.waitingForData();
						}
					}
				} else {
					setConsoleBytesWritten(0);
				}
			}
		}
	}

}
