/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TCPDataServer.java,v 1.5 2008/04/14 21:29:04 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.hyades.execution.local.CommunicationDebug;
import org.eclipse.hyades.execution.local.pub.internal.resources.LocalPublicResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * Insert the type's description here.
 * Creation date: (9/15/00 12:57:03 PM)
 * @author:
 */
public class TCPDataServer {
	private String MESSAGE_VALUE_ID = "messageValue";//$NON-NLS-1$

	private ServerSocket _sock = null;

	private TCPDataProcessor _server = null;

	private BufferFlusher _flusher = null;

	private int _port;

	private boolean _isDataServerRunning = true; // Bug 90153

	/* Buffers for getting incomming data off the
	 socket reader and onto a dispatch thread
	 */
	private static final short NUM_BUFFERS = 32;

	private SingleBuffer[] _bufferArray;

	private short _currentFullBuffers = 0;

	/* Different types of data */
	public static final byte BINARY_DATA = 0;

	public static final byte UTF8_STRING_DATA = 1;

	public static final byte UNICODE_STRING_DATA = 2;

	private int totalBytes = 0; // Bug 90153

	public static final int BUFFER_SIZE = 8 * Constants.MAX_MESSAGE_LENGTH; //64K

	class SingleBuffer implements Constants {
		public InetAddress addr;

		public int length = 0;

		public int size = MAX_MESSAGE_LENGTH;

		public byte[] data = new byte[MAX_MESSAGE_LENGTH];

	}

	class TCPDataProcessor extends Thread implements Constants {
		protected ServerSocket _socket;

		boolean _processing = true;

		protected boolean _shutdown = false; /* 9707 */

		protected short _currentFillerBuffer = 0;

		public void setSocket(ServerSocket sock) {
			_socket = sock;
		}

		/* 9707 */
		public void resumeProcessing() {
			synchronized (this) {
				_processing = true;
			}
		}

		/* 9707 */
		public void pauseProcessing() {
			synchronized (this) {
				_processing = false;
			}
		}

		/* 9707 */
		public boolean isProcessing() {
			synchronized (this) {
				if (CommunicationDebug.INSTANCE.debug) {
					System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_IS_PROCESSING_, String.valueOf(_processing), this));
				}
				return _processing;
			}
		}

		/* 9707 */
		public void shutdown() {
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_SHUT_DOWN_, this));
			}
			_shutdown = true;
			pauseProcessing();
			try {
				_socket.close();
			} catch (IOException e) {
			}
			interrupt();
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_SHUT_DOWN_AFTER_, this));
			}
		}

		public void run() {
			_isDataServerRunning = true; // Bug 90153

			/* Run forever */
			outer: while (!_shutdown || (_currentFullBuffers != 0)) { /* 9707 */
				if (isProcessing()) {
					Socket incommingConnection = null;
					InputStream is = null;
					try {
						incommingConnection = _socket.accept();
						is = incommingConnection.getInputStream();
						incommingConnection.setSoTimeout(1000);
						//System.out.println("----Incomming connection");
					} catch (SocketException e) {
						/* The server socket is toast, stop processing */
						pauseProcessing();
						_isDataServerRunning = false; // Bug 90153
						continue;
					} catch (IOException e) {
						_isDataServerRunning = false; // Bug 90153
						continue;
					}

					while (true) {
						/* If all the buffers are full wait for the first one to be emptied */
						while (_currentFullBuffers == NUM_BUFFERS) {
							synchronized (_bufferArray[0]) {
								try {
									_bufferArray[0].wait();
								} catch (InterruptedException e) {
								}
							}
						}

						/* Fill the next buffer */
						_bufferArray[_currentFillerBuffer].addr = incommingConnection.getInetAddress();
						try {
							_bufferArray[_currentFillerBuffer].length = is.read(_bufferArray[_currentFillerBuffer].data);
							/*System.out.println("---Read "+_bufferArray[_currentFillerBuffer].length+" bytes");*/
							int _bytes = _bufferArray[_currentFillerBuffer].length; // Bug 90153
							totalBytes += _bytes;
							//System.out.println("---------------------Read "+_bytes+" into _bufferArray["+_currentFillerBuffer+"]");
							//System.out.println("---------------------Read "+totalBytes+" bytes in total");
							if (_bytes == -1) {
								_isDataServerRunning = false;
							}
						} catch (InterruptedIOException e) {
							if (CommunicationDebug.INSTANCE.debug) {
								e.printStackTrace(System.out);
							}
							/* Read timeout, don't want to wait too long */

							/* This is used to terminate the thread if the process is killed abnormally */
							if (_shutdown && (_currentFullBuffers == 0)) {
								pauseProcessing();
								_isDataServerRunning = false; // Bug 90153
								try {
									incommingConnection.close(); /* 9707 */
								} catch (IOException e1) {
									// Handle IOException
								}
								break outer;
							}
						} catch (IOException e) {
							if (CommunicationDebug.INSTANCE.debug) {
								e.printStackTrace(System.out);
							}
							/* Socket is toast, we are done */
							pauseProcessing();
							_isDataServerRunning = false; // Bug 90153
							try {
								incommingConnection.close(); /* 9707 */
							} catch (IOException e1) {
								// Handle IOException
							}
							break outer;
						}

						/* Is the connection closed? */
						if (_bufferArray[_currentFillerBuffer].length < 0) {
							pauseProcessing();
							_isDataServerRunning = false; // Bug 90153
							try {
								/* Will hit here when detaching agent */
								incommingConnection.close();
							} catch (IOException e) {
							}
							break outer;
						}
						synchronized (_bufferArray[0]) {
							if (_bufferArray[_currentFillerBuffer].length > 0) {
								/* Move on to the next buffer */
								_currentFillerBuffer++;
								if (_currentFillerBuffer == NUM_BUFFERS) {
									_currentFillerBuffer = 0;
								}
								_currentFullBuffers++;

								/* Is this the first buffer filled? */
								if (_currentFullBuffers == 1) {
									_bufferArray[0].notifyAll();
								}
							}
						}
					}
				} else {
					try {
						if (CommunicationDebug.INSTANCE.debug) {
							System.out.println(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_RUN_);
						}
						/* Monitoring is stopped, keep this thread in sleep state */
						sleep(1000); /* 9707 */
					} catch (InterruptedException e) {
					}
				}
			}
		}

	} /* end class TCPDataProcessor */

	class BufferFlusher extends Thread implements Constants {
		private DataProcessor _processor = null;

		private byte[] _binaryForwardBuffer = new byte[MAX_MESSAGE_LENGTH];

		private char[] _stringForwardBuffer = new char[MAX_MESSAGE_LENGTH];

		private byte[] _messageHeader = new byte[MESSAGE_HEADER_LENGTH];

		private long _currentBufferSize = MAX_MESSAGE_LENGTH;

		private short _currentFlusherBuffer = 0;

		private int _currentHeaderOffset = 0;

		private int _bytesWritten = 0;

		public void setProcessor(DataProcessor processor) {
			_processor = processor;
		}

		/** Load the _messageHeader from a buffer of data
		 * @param   data - the byte[] holding the raw data.
		 * @param offset - where the header starts in the raw data buffer.
		 * @param length - the length of the raw data buffer.
		 * @returns      - the new offset in the raw data buffer to continue
		 *                 reading from.
		 */
		protected int loadMessageHeader(byte[] data, int offset, int limit) {
			/* Load all we can into the header */
			while (offset < limit && _currentHeaderOffset < MESSAGE_HEADER_LENGTH) {
				_messageHeader[_currentHeaderOffset++] = data[offset++];
			}
			return offset;
		}

		/** Get the length of the current message
		 */
		protected long getMessageLength() {
			return Message.readRALongFromBuffer(_messageHeader, 5);
		}

		/** Get the message type.  There are currently three types
		 of messages.  BINARY_DATA, UTF8_STRING_DATA, UNICODE_STRING_DATA.
		 */
		protected byte getMessageType() {
			return _messageHeader[9];

		}

		/** Check the message magic number.  If the magic number is incorrect we need
		 to go into recovery mode
		 */
		protected boolean checkMessageMagic() {
			return true;
		}

		/** Recursively process a buffer of raw data.
		 */
		protected int processData(byte[] data, int offset, int limit, InetAddress addr) {
			long messageLength;
			byte type;
			int current;

			current = offset;

			/* Is there data to process */
			if (offset >= limit) {
				return limit;
			}

			/* Is this a new message? */
			if (_currentHeaderOffset < MESSAGE_HEADER_LENGTH) {
				/* Load the message header */
				current = this.loadMessageHeader(data, current, limit);

				/* Did we get the entire header, if not return */
				if (current == limit) {
					return current;
				}

				/* Resize and compress the forward buffer if nessesary */
				if (getMessageLength() >= _currentBufferSize) {
					type = getMessageType();

					if (type == BINARY_DATA || type == UTF8_STRING_DATA) {
						byte[] replacement = new byte[(int) getMessageLength()];
						/* Shift the available data to the front of the buffer */
						System.arraycopy(data, current, replacement, 0, (limit - current));
						_bytesWritten = limit - current;
						_binaryForwardBuffer = replacement;
					} else {
						char[] replacement = new char[(int) getMessageLength()];
						/* Shift the available data to the front of the buffer */
						for (int i = 0; i < limit - current + 1; i++) {
							try {
								replacement[i] = (char) data[i + current];
							} catch (Exception e) {
								System.out.println(LocalPublicResourceBundle.TCPDataServer_BUFFERFLUSHER_DATA_REPLACEMENT_);
								System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_BUFFERFLUSHER_DATA_REPLACEMENT_LENGTH1_, 
										String.valueOf(replacement.length), String.valueOf(data.length))
										         + NLS.bind(LocalPublicResourceBundle.TCPDataServer_BUFFERFLUSHER_DATA_REPLACEMENT_LENGTH2_,
										        		 String.valueOf(i), String.valueOf(current)));
								e.printStackTrace(System.out);
								throw new RuntimeException(e);
							}
						}
						_bytesWritten = limit - current;
						_stringForwardBuffer = replacement;
					}
					return limit;
				}
			}

			/* Validate the message header, if we are in recovery
			 mode try and look at the next offset */
			if (!checkMessageMagic()) {
				System.out.println(LocalPublicResourceBundle.TCPDataServer_BUFFERFLUSHER_CORRUPT_DATA_);
				_currentHeaderOffset = 0;
				return processData(data, offset + 1, limit, addr);

			}

			/* How long is the current message */
			messageLength = getMessageLength();

			/* What is the message type */
			type = getMessageType();

			/* Process the entire buffer */
			while (current < limit) {

				if (type == BINARY_DATA || type == UTF8_STRING_DATA) {
					/* Copy as many bytes as possible into the forwarding buffer */
					while (current < limit && _bytesWritten < messageLength) {
						_binaryForwardBuffer[_bytesWritten++] = data[current++];
					}
					/* Are we at the end of the message? If so forward to the handler */
					if (_bytesWritten == messageLength) {
						_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
						_bytesWritten = 0;
						_currentHeaderOffset = 0;
						/* Continue processing this data buffer */
						current = this.processData(data, current, limit, addr);
					}
				} else if (type == UNICODE_STRING_DATA) {
					/* Copy as many bytes as possible into the forwarding buffer */
					while (offset < limit && _bytesWritten < messageLength) {
						_stringForwardBuffer[_bytesWritten >> 1] = (char) ((char) data[current++] | (char) (data[current++] << 8));
						_bytesWritten += 2;
					}
					/* Are we at the end of the message? If so forward to the handler */
					if (_bytesWritten == messageLength) {
						_processor.incommingData(_stringForwardBuffer, _bytesWritten, addr);
						_bytesWritten = 0;
						_currentHeaderOffset = 0;
						/* Continue processing this data buffer */
						current = this.processData(data, current, limit, addr);
					}
				} else {
					/* Invalid message type */
					/* Copy as many bytes as possible into the forwarding buffer */
					while (offset < limit && _bytesWritten < messageLength) {
						_binaryForwardBuffer[_bytesWritten++] = data[current++];
					}
					/* Are we at the end of the message? If so forward to the handler */
					if (_bytesWritten == messageLength) {
						_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
						_bytesWritten = 0;
						_currentHeaderOffset = 0;
						/* Continue processing this data buffer */
						current = this.processData(data, current, limit, addr);
					}
				}
			}
			return current;
		}

		public void run() {
			if (CommunicationDebug.INSTANCE.debugMessageValue) {
				MESSAGE_VALUE_ID = MESSAGE_VALUE_ID + "_" + this.hashCode();
			}
			//outer:	while(isProcessing() || (_currentFullBuffers != 0) || _isFinished == false) { /* 237169 make sure buffer is empty before exiting */
			outer: while (isProcessing() || (_currentFullBuffers != 0)) { // Bug 90153
				/* If there are no current buffers to empty wait */
				if (_currentFullBuffers == 0) {
					_processor.waitingForData();
					do {
						synchronized (_bufferArray[0]) {
							try {
								_bufferArray[0].wait(1000);
							} catch (InterruptedException e) {
								return;
							}
						}
						if (!isProcessing() && _currentFullBuffers == 0) {
							break outer;
						}
					} while (_currentFullBuffers == 0);
				}

				/* Empty the current buffer */
				if (_bufferArray[_currentFlusherBuffer].length > 0) {
					//System.out.println("---- Flushing "+ _bufferArray[_currentFlusherBuffer].length+" bytes from _bufferArray["+_currentFlusherBuffer+"]");
					if (CommunicationDebug.INSTANCE.debugMessageValue) {
						CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID, _bufferArray[_currentFlusherBuffer].data, 0, _bufferArray[_currentFlusherBuffer].length);
					}
					processData(_bufferArray[_currentFlusherBuffer].data, 0, _bufferArray[_currentFlusherBuffer].length, _bufferArray[_currentFlusherBuffer].addr);
					/* Mark the buffer as empty */
					_bufferArray[_currentFlusherBuffer].length = 0;

				}

				synchronized (_bufferArray[0]) {

					_currentFullBuffers--;

					/* Increment the flusher to the next buffer */
					_currentFlusherBuffer++;
					if (_currentFlusherBuffer == NUM_BUFFERS) {
						_currentFlusherBuffer = 0;
					}

					/* If the buffers were half full before this flush notify the
					 filler it can continue. Generally this could be NUMBUFFERS
					 but not as efficient as more thread switches happen
					 */
					if (_currentFullBuffers == 0) {
						_bufferArray[0].notifyAll();
					}
				}
			}

			/* Notify that the flusher is exiting */
			//System.out.println("Notify that the flusher is exiting!!!");
			//System.out.println("isProcessing() = "+isProcessing()+", _currentFullBuffers = " +_currentFullBuffers);
			if (_processor instanceof DataServerListener) {
				((DataServerListener) _processor).dataServerExited();
			}
		}

		public void shutdown() {
			//interrupt(); // Bug 90153
		}

	} /* end class RingBufferFiller */

	/**
	 * 
	 * @author slavescu
	 *
	 */
	class TCPDataProcessorNew extends TCPDataProcessor {
		protected ExtendedDataServerListener _extendedProcessor;

		protected DataProcessor _processor;

		//		protected int _currentProcessingOffset;
		protected InetAddress _currentBufferAddr;

		protected byte[] _currentBufferData;

		protected int _currentBufferLength;
		protected long _currentMessageLength;
		protected byte _currentMessageType;
		protected int _currentHeaderOffset;
		protected char[] _stringForwardBuffer;
		protected byte[] _binaryForwardBuffer;
		protected int _bytesWritten;
		protected int _bytesRead;
		protected boolean _useExtendedProcessor;
		protected int _processed;
		protected byte[] _copyBuffer; // used only when the processor doesn't implement ExtendedDataServerListener
		protected boolean crimsonParser;
		protected boolean breakOuter;
		protected byte[] _inputStreamModeBuffer;
		protected int _inputStreamModeBufferLength,_inputStreamModeRecieverBufferLength;
		protected int _inputStreamModeBufferPos;

		public void run() {
			crimsonParser = isCrimsonParser();
			if (CommunicationDebug.INSTANCE.debugMessageValue) {
				MESSAGE_VALUE_ID = MESSAGE_VALUE_ID + "_newDataProcessor_debugMessageValue_" + this.hashCode();//$NON-NLS-1$
			}
			//			else
			//				if(CommunicationDebug.INSTANCE.debugSocketInputStream)
			//				{
			//					MESSAGE_VALUE_ID=MESSAGE_VALUE_ID+"_newDataProcessor_debugSocketInputStream"+this.hashCode();
			//				}

			_isDataServerRunning = true; // Bug 90153
			_currentBufferData = new byte[BUFFER_SIZE];
			initLocalVariables();
			if (_processor instanceof ExtendedDataServerListener) {
				_useExtendedProcessor = true;
				_extendedProcessor = (ExtendedDataServerListener) _processor;
			} else
				_copyBuffer = new byte[BUFFER_SIZE];

			/* Run forever */
			outer: while (!_shutdown && _isDataServerRunning) { /* 9707 */
				if (CommunicationDebug.INSTANCE.debug) {
					System.out.println(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_STEP_OUTER_);
				}
				if (isProcessing()) {
					final Socket incommingConnection ;
					InputStream is = null;
					try {
						incommingConnection = _socket.accept();
						is = incommingConnection.getInputStream();
						//						incommingConnection.setSoTimeout(1000);
						//System.out.println("----Incomming connection");
					} catch (SocketException e) {
						/* The server socket is toast, stop processing */
						pauseProcessing();
						_isDataServerRunning = false; // Bug 90153
						continue;
					} catch (IOException e) {
						_isDataServerRunning = false; // Bug 90153
						continue;
					}
					_currentBufferAddr = incommingConnection.getInetAddress();

					if (!CommunicationDebug.INSTANCE.debugUseEventMode && _useExtendedProcessor &&  !crimsonParser) {
						_inputStreamModeBuffer = new byte[BUFFER_SIZE];
						final InputStream delegatedInputStream = is;
						InputStream isFiltered = new InputStream(){
							byte[] oneByte=new byte[1];
							public int available() throws IOException {
								return delegatedInputStream.available();
							}

							public void close() throws IOException {
								delegatedInputStream.close();
							}

							public synchronized void mark(int readlimit) {
								delegatedInputStream.mark(readlimit);
							}

							public boolean markSupported() {
								return delegatedInputStream.markSupported();
							}

							public int read(byte[] b) throws IOException {
								return read(b,0,b.length);
							}

							public synchronized void reset() throws IOException {
								delegatedInputStream.reset();
							}

							public long skip(long n) throws IOException {
								return delegatedInputStream.skip(n);
							}

							public synchronized int read(byte[] b, int off, int len) throws IOException {
								while(true){
									if(_inputStreamModeBufferPos!=_inputStreamModeBufferLength)
									{
										int ret = copyInputData(b,off,len);
										return ret;
									}
									else
									{
										_inputStreamModeBufferPos = 0;
										_inputStreamModeBufferLength = 0;
									}
										
									if (_currentBufferLength == _currentBufferData.length) {
										breakOuter=false;
										extendBuffer(incommingConnection);
										if(breakOuter)
											return -1;
									}
									_bytesRead = delegatedInputStream.read(_currentBufferData, _currentBufferLength, _currentBufferData.length - _currentBufferLength);
									if (_bytesRead > 0) {
										if (CommunicationDebug.INSTANCE.debugMessageValue) {
											CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID, _currentBufferData, _currentBufferLength, _bytesRead);
										}
										_currentBufferLength += _bytesRead;
										// check if the header was received
										if (_currentBufferLength > MESSAGE_HEADER_LENGTH) {
											processCurrentBuffer();
											int ret = copyInputData(b,off,len);
											if(ret==0)
												continue;
											return ret;
										}
									} else if (_bytesRead < 0) {
										/* The connection is closed */
										done(incommingConnection);
										return -1;
									}
									return 0;
								}
							}

							protected int copyInputData(byte[] b, int off, int len) {
								int ret=0;
								len = Math.min(len,_inputStreamModeBufferLength-_inputStreamModeBufferPos);
								System.arraycopy(_inputStreamModeBuffer,_inputStreamModeBufferPos,b,off,len);
//								if(CommunicationDebug.INSTANCE.debugMessageValue)
//								{
//									CommunicationDebug.INSTANCE.writeBinaryLog("incommingData", b,off,len);
//								}
								_inputStreamModeBufferPos+=len;
								return ret+len;
							}

							public int read() throws IOException {
								if(read(oneByte,0,1)==-1)
									return -1;
								else
									return oneByte[0];
							}
						};
						try {
							_extendedProcessor.incommingStream(isFiltered,_currentBufferAddr);
						} catch (Exception e) {
							if (CommunicationDebug.INSTANCE.debug) {
								System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_END_EXCEPTION_, _extendedProcessor));
								e.printStackTrace(System.out);
							}
							done(incommingConnection);
							break outer;
						}
						if(CommunicationDebug.INSTANCE.debug)
						{
							System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_END_, _extendedProcessor));
						}
					} else
						while (true) {
							try {
								//							if(CommunicationDebug.INSTANCE.debugSocketInputStream)
								//							{
								//								_bytesRead = is.read(_currentBuffer.data, 0, _currentBuffer.data.length);
								//								if(_bytesRead>0)
								//									CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID,_currentBuffer.data,0,_bytesRead);
								//								else {
								//									/* The connection is closed */
								//									done(incommingConnection);
								//									break outer;
								//								}
								//							}
								//							else
								//							{
								if (_processed != 0 && _currentMessageLength > Integer.MAX_VALUE) {
									_currentHeaderOffset = 0;
									recover();
									continue;
								} else {
									if (_currentBufferLength == _currentBufferData.length) {
										breakOuter=false;
										extendBuffer(incommingConnection);
										if(breakOuter)
											break outer;
									}
									//									if(CommunicationDebug.INSTANCE.debug)
									//									{
									//										System.out.println("TCPDataProcessorNew.run() - about to read _currentBufferData.length - _currentBufferLength="+(_currentBufferData.length - _currentBufferLength));
									//									}
									_bytesRead = is.read(_currentBufferData, _currentBufferLength, _currentBufferData.length - _currentBufferLength);
									//									if(CommunicationDebug.INSTANCE.debug)
									//									{
									//										System.out.println("TCPDataProcessorNew.run() - bytesRead="+_bytesRead);
									//									}
									if (_bytesRead > 0) {
										if (CommunicationDebug.INSTANCE.debugMessageValue) {
											CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID, _currentBufferData, _currentBufferLength, _bytesRead);
										}
										_currentBufferLength += _bytesRead;
										// check if the header was received
										if (_currentBufferLength > MESSAGE_HEADER_LENGTH) {
											_processed = 0;
											processCurrentBuffer();
										}
									} else if (_bytesRead < 0) {
										/* The connection is closed */
										done(incommingConnection);
										break outer;
									}
								}
								//							}
							} catch (InterruptedIOException e) {
								if (CommunicationDebug.INSTANCE.debug) {
									System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_TIMEOUT_, String.valueOf(e.bytesTransferred)));
									e.printStackTrace(System.out);
								}
								/* Read timeout, don't want to wait too long */
								/* This is used to terminate the thread if the process is killed abnormally */
								if (e.bytesTransferred > 0) {
									_bytesRead = e.bytesTransferred;
									if (CommunicationDebug.INSTANCE.debugMessageValue) {
										CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID, _currentBufferData, _currentBufferLength, _bytesRead);
									}
									_currentBufferLength += _bytesRead;
									// check if the header was received
									if (_currentBufferLength > MESSAGE_HEADER_LENGTH) {
										_processed = 0;
										processCurrentBuffer();
									}

								}
								//								else
								//								if(_shutdown)
								//								{
								//									/* The connection is closed */
								//									done(incommingConnection);
								//									break outer;
								//								}
							} catch (IOException e) {
								if (CommunicationDebug.INSTANCE.debug) {
									e.printStackTrace(System.out);
								}
								done(incommingConnection);
								break outer;
							}

						}
				} else {
					try {
						if (CommunicationDebug.INSTANCE.debug) {
							System.out.println(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_MONITORING_STOPPED_);
						}
						/* Monitoring is stopped, keep this thread in sleep state */
						sleep(1000); /* 9707 */
					} catch (InterruptedException e) {
					}
				}
			}
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_EXIT_OUTER_LOOP_);
			}

			try {
				if (CommunicationDebug.INSTANCE.debug) {
					System.out.println(LocalPublicResourceBundle.TCPDataServer_DATA_PROCESSOR_CLOSE_SOCKET_);
				}
				_socket.close();
			} catch (IOException e) {
				if (CommunicationDebug.INSTANCE.debug) {
					e.printStackTrace(System.out);
				}
			}

			if (_processor instanceof DataServerListener) {
				((DataServerListener) _processor).dataServerExited();
			}
			if (CommunicationDebug.INSTANCE.debugMessageValue) {

				try {
					CommunicationDebug.INSTANCE.getBinaryLog(MESSAGE_VALUE_ID).flush();
					CommunicationDebug.INSTANCE.removeBinaryLog(MESSAGE_VALUE_ID);
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}

		}

		private void extendBuffer(Socket incommingConnection) {
			int remainingData = _currentBufferLength - _currentHeaderOffset;
			if (_currentMessageLength > _currentBufferLength - MESSAGE_HEADER_LENGTH) {
				//grow to _currentMessageLength+MESSAGE_HEADER_LENGTH
				byte[] b = null;
				try {
					b = new byte[(int) _currentMessageLength + MESSAGE_HEADER_LENGTH];
				} catch (Throwable e) {
					if (CommunicationDebug.INSTANCE.debug) {
						System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_CURRENT_HEADER_OFFSET_, String.valueOf(_currentHeaderOffset)));
						System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_CURRENT_BUFFER_LENGTH_, String.valueOf(_currentBufferLength)));
						System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_GET_MESSAGE_LENGTH_, String.valueOf(getMessageLength())));
						System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_GET_MESSAGE_TYPE_, String.valueOf(getMessageType())));
						e.printStackTrace(System.out);
						//dump current buffer in a separate file
						CommunicationDebug.INSTANCE.writeBinaryLog(MESSAGE_VALUE_ID + "_" + _currentHeaderOffset, _currentBufferData, 0, _currentBufferLength);
					}
					//											recover();
					//											continue;
					breakOuter=true;
					return;
				}
				//copy remainig data to new location
				System.arraycopy(_currentBufferData, _currentHeaderOffset, b, 0, _currentBufferLength - _currentHeaderOffset);
				_currentBufferData = b;
			} else {
				if (remainingData < _currentHeaderOffset) {
					//shift remainig data to the begining of the buffer
					System.arraycopy(_currentBufferData, _currentHeaderOffset, _currentBufferData, 0, remainingData);
				} else {
					//shift remainig data to the begining of the buffer
					for (int i = 0; i < remainingData; i++) {
						_currentBufferData[i] = _currentBufferData[i + _currentHeaderOffset];
					}
				}
			}
			_currentBufferLength = remainingData;
			_currentHeaderOffset = 0;
		}

		protected boolean isCrimsonParser() {
				SAXParserFactory factory = SAXParserFactory.newInstance();
				try {
					SAXParser p = factory.newSAXParser();;
					if (p!=null && p.getClass().getName().equals("org.apache.crimson.jaxp.SAXParserImpl")) {//$NON-NLS-1$
						return true;
					}
				} catch (Exception e) {
				}
			return false;
		}

		private void done(Socket incommingConnection) {
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_DONE_ENTER_, this));
			}
			/* Socket is toast or connection closed , we are done */
			pauseProcessing();
			_isDataServerRunning = false; // Bug 90153
			try {
				incommingConnection.close(); /* 9707 */
				if (CommunicationDebug.INSTANCE.debug) {
					new Throwable(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_DONE_CONNECTION_CLOSED_, this)).printStackTrace(System.out);
				}
			} catch (IOException e1) {
				if (CommunicationDebug.INSTANCE.debug) {
					e1.printStackTrace(System.out);
				}
			}
		}

		protected int recover() {
			_currentHeaderOffset++;
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_RECOVER_, String.valueOf(_currentHeaderOffset)));
			}
			//incorrect length, try to recover
			if (_currentHeaderOffset == _currentBufferLength) {
				//ignore this buffer and continue reading
				initLocalVariables();
				_currentHeaderOffset = 0;
			} else {
				_currentMessageLength = -1;
				_processed = 0;
				// process from new possition
				processCurrentBuffer();
			}
			return _currentHeaderOffset;
		}

		private void initLocalVariables() {
			_currentHeaderOffset = 0;
			_currentMessageLength = -1;
			_bytesRead = 0;
			_bytesWritten = 0;
			_currentBufferLength = 0;
			_processed = 0;
		}

		public void setProcessor(DataProcessor processor) {
			if (_processor instanceof ExtendedDataServerListener) {
				_useExtendedProcessor = true;
				_extendedProcessor = (ExtendedDataServerListener) _processor;
			} else {
				_useExtendedProcessor = false;
				this._processor = processor;
				if (_copyBuffer == null)
					_copyBuffer = new byte[BUFFER_SIZE];
			}
		}

		/** Get the length of the current message
		 */
		protected long getMessageLength() {
			return Message.readRALongFromBuffer(_currentBufferData, _currentHeaderOffset + 5);
		}

		/** Get the message type.  There are currently three types
		 of messages.  BINARY_DATA, UTF8_STRING_DATA, UNICODE_STRING_DATA.
		 */
		protected byte getMessageType() {
			return _currentBufferData[_currentHeaderOffset + 9];
		}

		/** Check the message magic number.  If the magic number is incorrect we need
		 to go into recovery mode
		 */
		protected boolean checkMessageMagic() {
			//			/* Check the message magic number */
			//			long messageKey = (long) (_currentBufferData[_currentHeaderOffset] << 24 & 0xff000000) |
			//				(long) (_currentBufferData[_currentHeaderOffset + 1] << 16 & 0x00ff0000) | 
			//				(long) _currentBufferData[_currentHeaderOffset + 2] << 8 & 0x0000ff00 | 
			//				(long) _currentBufferData[_currentHeaderOffset + 3];
			//			/* If the magic is incorect we need to go into recovery mode */
			//			/* TO_DO: RE-enable and test*/
			//			if (messageKey == RA_MAGIC ) {
			//				return false;
			//			}
			//			return Message.readRALongFromBuffer(_currentBufferData, _currentHeaderOffset) == RA_MAGIC ;
			if (_currentBufferData[_currentHeaderOffset] != RA_MAGIC_0)
				return false;
			if (_currentBufferData[_currentHeaderOffset + 1] != RA_MAGIC_1)
				return false;
			if (_currentBufferData[_currentHeaderOffset + 2] != RA_MAGIC_2)
				return false;
			if (_currentBufferData[_currentHeaderOffset + 3] != RA_MAGIC_3)
				return false;
			return true;
		}

		/**
		 * process the current input buffer and sends all the full messages to the data processor
		 */
		protected void processCurrentBuffer() {
			/* Is there data to process */
			if (_currentBufferLength <= 0) {
				_processed = 0;
				return;
			}
			int _stopProcessingMarker = _currentBufferLength - MESSAGE_HEADER_LENGTH;
			do {
				if (_currentMessageLength < 0) {
					/* Validate the message header*/
					do {
						if (_currentHeaderOffset > _stopProcessingMarker) {
							_processed = 0;
							return;
						}
						if (checkMessageMagic()) {
							break;
						}
						//recovery mode, try and look at the next offset
						if (CommunicationDebug.INSTANCE.debug)
							System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_INVALID_MAGIC_NUMBER_, String.valueOf(_currentHeaderOffset)));
						_currentHeaderOffset++;
					} while (true);
					_currentMessageLength = getMessageLength();
					_currentMessageType = getMessageType();
				}
				if (_currentHeaderOffset + _currentMessageLength <= _stopProcessingMarker) {
					_bytesWritten = writeMessage(_currentHeaderOffset, _currentBufferData);
					_processed += _bytesWritten;
					_currentHeaderOffset = _currentHeaderOffset + MESSAGE_HEADER_LENGTH + _bytesWritten;
//					_currentMessageLength = -1;
				} else {
					return;
				}
			} while (true);
		}

		/**
		 * sends a full message to the data processor
		 */
		protected int writeMessage(int _currentHeaderOffset, byte[] _currentBufferData) {
			int _bytesWritten = 0;
			if (_currentMessageType == UNICODE_STRING_DATA) {
				int endOffset = _currentHeaderOffset + MESSAGE_HEADER_LENGTH + (int) _currentMessageLength;
				for (int i = _currentHeaderOffset + MESSAGE_HEADER_LENGTH; i < endOffset;) {
					_stringForwardBuffer[_bytesWritten >> 1] = (char) ((char) _currentBufferData[i++] | (char) (_currentBufferData[i++] << 8));
					_bytesWritten += 2;

				}
				/* Are we at the end of the message? If so forward to the handler */
				if (_bytesWritten == _currentMessageLength) {
					_processor.incommingData(_stringForwardBuffer, _bytesWritten, _currentBufferAddr);
					_currentMessageLength = -1;
//					_bytesWritten = 0;
//					_currentHeaderOffset = 0;
//					/* Continue processing this data buffer */
//					current = this.processData(data, current, limit, addr);
				}

			} else {
				_bytesWritten = (int) _currentMessageLength;
				if (_useExtendedProcessor) {
					if(!CommunicationDebug.INSTANCE.debugUseEventMode && !crimsonParser)
					{
						if(_inputStreamModeBuffer.length-_inputStreamModeBufferLength<_bytesWritten)
						{
							byte[] temp = new byte[_inputStreamModeBufferLength+_bytesWritten+1024];
							System.arraycopy(_inputStreamModeBuffer,0,temp,0,_inputStreamModeBufferLength);
							_inputStreamModeBuffer=temp;
						}
						System.arraycopy(_currentBufferData,_currentHeaderOffset + MESSAGE_HEADER_LENGTH, _inputStreamModeBuffer,_inputStreamModeBufferLength,_bytesWritten);
						_inputStreamModeBufferLength+=_bytesWritten;
					}
					else
						_extendedProcessor.incommingData(_currentBufferData, _currentHeaderOffset + MESSAGE_HEADER_LENGTH, _bytesWritten, _currentBufferAddr);
				} else {
					/* BINARY_DATA, UTF8_STRING_DATA or Invalid message type */
					/* Copy as many bytes as possible into the forwarding buffer */
					if (_copyBuffer.length < _bytesWritten)
						_copyBuffer = new byte[_bytesWritten];
					System.arraycopy(_currentBufferData, _currentHeaderOffset + MESSAGE_HEADER_LENGTH, _copyBuffer, 0, _bytesWritten);
					_processor.incommingData(_copyBuffer, _bytesWritten, _currentBufferAddr);
				}
				_currentMessageLength = -1;
			}
			return _bytesWritten;
		}

		public void shutdown() {
			if (CommunicationDebug.INSTANCE.debug) {
				System.out.println(NLS.bind(LocalPublicResourceBundle.TCPDataServer_DATA_SERVER_NEW_SHUT_DOWN_, this));
			}
			_shutdown = true;

		}

	} /* end class TCPDataProcessorNew */

	/**
	 * TCPDataServer constructor comment.
	 */
	public TCPDataServer() {
		super();
	}

	public int getPort() {
		return _port;
	}

	public InetAddress getServerAddress() {
		return _sock.getInetAddress();
	}
	
	/**
	 * Begins listening for data on the given port. If the server is unable to listen on the port, an exception is thrown to the calling method.
	 * @param processor DataProcessor with which to process the received data
	 * @param port Port on which to listen (0 is random).
	 */	
	public void startServer(DataProcessor processor, int port) throws SocketException, IOException {
		startServer(processor, port, false);
	}

	/**
	 * Begins listening for data on the given port.
	 * If the server is unable to listen on the port, the server will increment the port (or re-randomize if port is 0) and attempt to listen again.  
	 * @param processor DataProcessor with which to process the received data
	 * @param port Port on which to listen (0 is random)
	 * @param incrementFromInitialPort If listening on 'port' fails, whether or not to increment and attempt again.
	 */
	public void startServer(DataProcessor processor, int port, boolean incrementFromInitialPort) throws SocketException, IOException {
		if (!CommunicationDebug.INSTANCE.debugUseOldDataServer) {
			startServerNew(processor, port, incrementFromInitialPort);
		} else {
			/* Create the buffering mechanism */
			_bufferArray = new SingleBuffer[NUM_BUFFERS];
			for (int i = 0; i < NUM_BUFFERS; i++)
				_bufferArray[i] = new SingleBuffer();

			/* Start the server */
			if(!incrementFromInitialPort) {
				_sock = new ServerSocket(port);
			} else {
				boolean portConnected = false;
				
				while(!portConnected) {
					try {
						_sock = new ServerSocket(port);
						portConnected = true;
					} catch(Exception e) {
						if(port != 0) {
							port++;
						}
					}
				}
			}

			_port = _sock.getLocalPort();

			_server = new TCPDataProcessor();
			_server.setSocket(_sock);
			_server.setName("TCPDataFiller");//$NON-NLS-1$

			/* Set up the data flusher */
			_flusher = new BufferFlusher();
			_flusher.setProcessor(processor);
			_flusher.setName("TCPDataFlusher");//$NON-NLS-1$

			/* Don't block exit of the VM */
			_server.setDaemon(true);
			_flusher.setDaemon(true);

			/* RKD:
			 Because these two threads intensely contend for the same resources
			 we need to increase the priority of the flusher, otherwise I have
			 noticed that the flusher doesn't run often enough and the data sits
			 in the buffers far too long.  A relitive increase of 3 seems to much
			 better */

			_flusher.setPriority(_server.getPriority() + 3);

			_server.start();
			_flusher.start();
		}
	}
	
	/**
	 * Begins listening for data on the given port.  
	 * @param processor DataProcessor with which to process the received data
	 * @param port Port on which to listen (0 is random)
	 * @param incrementFromInitialPort If listening on 'port' fails, whether or not to increment and attempt again
	 */
	private void startServerNew(DataProcessor processor, int port, boolean incrementFromInitialPort) throws SocketException, IOException {
		/* Create the buffering mechanism */
		_bufferArray = new SingleBuffer[NUM_BUFFERS];
		for (int i = 0; i < NUM_BUFFERS; i++)
			_bufferArray[i] = new SingleBuffer();

		/* Start the server */
		
		
		if(!incrementFromInitialPort) {
			_sock = new ServerSocket(port);
		} else {
			boolean portConnected = false;
			
			while(!portConnected) {
				try {
					_sock = new ServerSocket(port);
					portConnected = true;
				} catch(Exception e) {
					if(port != 0) {
						port++;
					}
				}
			}
		}

		_port = _sock.getLocalPort();

		_server = new TCPDataProcessorNew();
		_server.setSocket(_sock);
		_server.setName("TCPDataFiller");//$NON-NLS-1$
		((TCPDataProcessorNew) _server).setProcessor(processor);

		/* Don't block exit of the VM */
		_server.setDaemon(true);

		_server.start();
	}

	public void startServer(DataProcessor processor) throws SocketException, IOException {
		startServer(processor, 0);
	}

	public boolean isProcessing() {
		return _isDataServerRunning; // Bug 90153
		//		return _server.isAlive();
	}

	public void stopServer() {
		_server.pauseProcessing(); /* 9707 */
	}

	/* 9707 */
	public void resumeServer() {
		_server.resumeProcessing();
	}

	/* 9707 */
	public void resumeServer(DataProcessor processor) {
		if (!CommunicationDebug.INSTANCE.debugUseOldDataServer) {
			resumeServerNew(processor);
		} else {
			_flusher.setProcessor(processor);
			_server.resumeProcessing();
		}
	}

	private void resumeServerNew(DataProcessor processor) {
		_server.resumeProcessing();
		((TCPDataProcessorNew) _server).setProcessor(processor);
	}

	/* 9707 */
	public void shutdownServer() {
		//System.out.println("TCPDataServer is shutted down!");
		if (!CommunicationDebug.INSTANCE.debugUseOldDataServer) {
			shutdownServerNew();
		} else {
			_server.shutdown();
			_flusher.shutdown();
		}
	}

	private void shutdownServerNew() {
		_server.shutdown();
	}
}
