/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Agent.java,v 1.1 2007/04/09 14:04:25 samwai Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.control;

import org.eclipse.hyades.execution.local.common.CustomCommand;
import org.eclipse.hyades.execution.local.common.DataProcessor;

public interface Agent {

	/**
	 * Add a listener that will inform you when this Agent becomes active/inactive.
	 * 
	 * @see AgentListener
	 */
	void addAgentListener(AgentListener listener);

	/**
	 * Remove a listener previously added with addAgentListener.
	 * 
	 * @see AgentListener
	 */
	void removeAgentListener(AgentListener listener);

	/**
	 * Get the process which this agent is a member of.
	 * 
	 * @return the Process if it exists, null otherwise.
	 */
	Process getProcess();

	void setProcess(Process p);

	/**
	 * If autoAttach is set, this is like registering an interest in this agent. It may not exist yet within the process, but you want to be attached to the agent as soon as it becomes active.
	 */
	boolean isAutoAttach();

	void setAutoAttach(boolean auto);

	/**
	 * Attach to the agent.
	 */
	void attach() throws InactiveAgentException, InactiveProcessException;

	/**
	 * Detach from the agent.
	 */
	void detach() throws InactiveAgentException, InactiveProcessException;

	/**
	 * Retrieve the type name of this agent.
	 * 
	 * @return the type of the agent if known, null otherwise.
	 */
	String getType();

	void setType(String type);

	/**
	 * Retrieve the name of this agent.
	 * 
	 * @return the name of the agent if known, null otherwise.
	 */
	String getName();

	void setName(String name);

	/**
	 * Retrieve the UUID of this agent.
	 * 
	 * @return the uuid of the agent if it is known, null otherwise.
	 */
	String getUUID();

	void setUUID(String uuid);

	/**
	 * Determine whether this agent is currently active.
	 */
	boolean isActive();

	void setActive(boolean isActive);

	/**
	 * Determine if this agent is currently being monitored.
	 */
	boolean isMonitored();

	void setMonitored(boolean isMonitored);

	/**
	 * Determine is this agent is currently attached to a client.
	 */
	boolean isAttached();

	boolean isAttached(boolean remote); // Bug 54376

	void setAttached(boolean isAttached);

	/**
	 * Start monitoring the data output of this agent using the specified DataProcessor. You must be attached to the agent before you can start monitoring it.
	 */
	void startMonitoring(DataProcessor processor) throws InactiveAgentException;

	/**
	 * Stop monitoring this agent.
	 */
	void stopMonitoring() throws InactiveAgentException;

	/**
	 * Set the configuration for the agent.
	 */
	void setConfiguration(AgentConfiguration config);

	/**
	 * Get the configuration object for this agent.
	 * 
	 * @return the AgentConfiguration if it exists, null otherwise.
	 */
	AgentConfiguration getConfiguration();

	/**
	 * Publish's the AgentConfiguration to an active agent
	 */
	void publishConfiguration() throws InactiveAgentException;

	/**
	 * Send a custom command to the agent for processing.
	 */
	void invokeCustomCommand(CustomCommand command) throws InactiveAgentException;

	/**
	 * Determine if this agent is sending data to a profiling file
	 */
	public boolean isToProfileFile();

	/**
	 * Get the fully quality path of profiling file
	 * 
	 * @return String
	 */
	public String getProfileFile();

	/**
	 * Set the fully quality path of profiling file
	 * 
	 * @param _profileFile
	 *            The _profileFile to set
	 */
	public void setProfileFile(String _profileFile);

}
