/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NodeFactory.java,v 1.2 2008/03/20 18:49:58 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.control;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.Principal;
import java.util.Hashtable;

import org.eclipse.hyades.execution.security.DuplicateUserException;
import org.eclipse.hyades.execution.security.User;

/**
 * A factory for creating Nodes. Nodes can be specified by either name or InetAddress. Each Node is treated as a singleton and attempts to create a Node that already exists will return the same Node instance that already existed.
 */
public class NodeFactory {
	private static String _hostname = "localhost";//$NON-NLS-1$
	private static Hashtable _servers = new Hashtable();

	// static {
	// try {
	// _hostname = InetAddress.getLocalHost().getHostName();
	// _servers.put(_hostname, createNode(InetAddress.getLocalHost()));
	// }
	// catch(UnknownHostException e) {
	// /* We ignore this here and throw the error
	// when getLocalHost is called later.
	// */
	// }
	// }

	/**
	 * Create a Node for the suplied hostname. Nodes are are intended to be singletons in this application so if the Node exists it will be returned.
	 */
	public static Node createNode(String hostname) throws UnknownHostException {
		try {
			return createNode(hostname, null);
		} catch (DuplicateUserException e) {
			return containsNode(hostname, null);
		}

	}

	public static Node createNode(String hostname, Principal principal) throws UnknownHostException, DuplicateUserException {
		Node result = containsNode(hostname, principal);
		if (result == null) {
			InetAddress addr = InetAddress.getByName(hostname);
			return addNode(hostname, addr, principal);
		}
		throw new DuplicateUserException();
	}

	/**
	 * Create a Node for the suplied InetAddress. Nodes are are intended to be singletons in this application so if the Node exists it will be returned.
	 */
	public static Node createNode(InetAddress address) throws UnknownHostException {
		return createNode(address, null);
	}

	public static Node createNode(InetAddress address, Principal principal) throws UnknownHostException {
		String hostname = address.getHostName();
		Node result = containsNode(hostname, principal);
		if (result == null) {
			result = addNode(hostname, address, principal);
		}
		return result;
	}

	/**
	 * Returns the node that represents the local host.
	 * 
	 * @deprecated - use getLocalHost(User user)
	 */
	public static Node getLocalHost() throws UnknownHostException {
		return getLocalHost(null);
	}

	public static Node getLocalHost(Principal principal) throws UnknownHostException {
		/* The localhost is always in the first slot of the servers table */
		Node localNode = null;
		synchronized (_servers) {
			localNode = (Node) _servers.get(_hostname);
			if (localNode == null) {
				throw new UnknownHostException();
			}
		}
		return localNode;
	}

	public static void addNode(Node node) {
		String nodeName = node.getName();
		synchronized (_servers) {
			if (!_servers.containsKey(nodeName)) {
				_servers.put(nodeName, node);
			}
		}
	}

	private static Node addNode(String name, InetAddress addr, Principal principal) {
		if (name == null) {
			return null;
		}

		if (name.equals("localhost")) {//$NON-NLS-1$
			try {
				name = InetAddress.getLocalHost().getHostName();
				InetAddress[] addrs = InetAddress.getAllByName(name);
				addr = addrs[0];
			} catch (UnknownHostException e) {
				/* We can ignore this */
			}
		}

		synchronized (_servers) {
			// Check if node already exist
			if (_servers.containsKey(name)) {
				Node existingNode = (Node) _servers.get(name);
				// Update the principal
				if (principal instanceof User) {
					existingNode.setUser((User) principal);
				} else if (principal instanceof Application) {
					existingNode.setApplication((Application) principal);
				}
				return existingNode;
			} else {
				Node newNode = new NodeImpl(name, addr);
				if (principal instanceof User) {
					newNode.setUser((User) principal);
				} else if (principal instanceof Application) {
					newNode.setApplication((Application) principal);
				}
				addNode(newNode);
				return newNode;
			}
		}
	}

	/**
	 * Searches the Node list based upon the hostname.
	 * 
	 * @return the Node if it exists, null otherwise.
	 */
	private static Node containsNode(String name, Principal principal) {
		Node node = null;

		if (name == null) {
			return null;
		}

		synchronized (_servers) {
			/* If this is "localhost" try and resolve its real name first */
			if (name.equals("localhost")) {//$NON-NLS-1$
				try {
					name = InetAddress.getLocalHost().getHostName();
				} catch (UnknownHostException e) {
					/* We can ignore this */
				}
			}

			if (_servers.containsKey(name)) {
				node = (Node) _servers.get(name);
				if (principal == null) {
					// Node is the right one since no principal is supplied
				} else if ((principal instanceof User) && (node.getUser() != null)) {
					if (principal.getName().equals(node.getUser().getName())) {
						// Node is the right one since no principal is supplied
					} else {
						node = null;
					}
				} else if ((principal instanceof Application) && (node.getApplication() != null)) {
					if (principal.getName().equals(node.getApplication().getName())) {
						// Node is the right one since no principal is supplied
					} else {
						node = null;
					}
				}
			}
		}

		return node;
	}

	/**
	 * Searches the Node list based upon the InetAddress.
	 * 
	 * @return the Node if it exists, null otherwise.
	 */
	public static Node getNode(InetAddress addr) {
		return getNode(addr, null);
	}

	public static Node getNode(InetAddress addr, Principal principal) {
		/* Resolve the name and delegate */
		return containsNode(addr.getHostName(), principal);
	}

	public static Node getNode(String name, Principal principal) throws UnknownHostException {
		/* Resolve the name and delegate */
		return getNode(InetAddress.getByName(name), principal);
	}

}
