/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SecureConnectionImpl.java,v 1.3 2009/09/28 19:17:48 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.control;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.nio.channels.SocketChannel;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;

import org.eclipse.core.runtime.Assert;
import org.eclipse.hyades.execution.local.pub.internal.resources.LocalPublicResourceBundle;
import org.eclipse.hyades.execution.security.ISecureClientParameters;
import org.eclipse.hyades.execution.core.file.socket.ISocketChannel;
import org.eclipse.hyades.execution.core.file.socket.ISocketChannelFactory;
import org.eclipse.hyades.execution.core.file.socket.SocketChannelFactory;
import org.eclipse.hyades.execution.security.LoginFailedException;
import org.eclipse.hyades.execution.security.SecureConnectionRequiredException;
import org.eclipse.hyades.execution.security.UntrustedAgentControllerException;

public class SecureConnectionImpl extends ConnectionImpl implements ISocketChannelFactory {

	public SecureConnectionImpl() {
		super();
	}

	public void connect(Node node, int port) throws IOException, LoginFailedException, SecureConnectionRequiredException, UntrustedAgentControllerException {
		int protocolOffset = 0;

		/* Set the node */
		_node = node;

		/* Determine our acceptable protocols */
		String[] sslProtocols = getSSLProtocols();
		SSLContext sslContext = null;

		/* Connect to the remote machine */
		boolean done = false;

		/* Get the SSL context */
		do {
			try {
				sslContext = getSSLContext(sslProtocols[protocolOffset]);
				done = true;
			} catch (NoSuchAlgorithmException e) {
				protocolOffset++;
			}
		} while (!done && (protocolOffset < sslProtocols.length));

		if (sslContext == null) {
			throw new IOException(LocalPublicResourceBundle.SecureConnectionImpl_NOT_SUPPORTED_PROTOCOLS_EXCEPTION_);
		}

		_socket = getSSLSocket(sslContext, port);
		_port = port;

		/* Set the cipher suites */
		String[] cipherSuites = getCipherSuites();
		((SSLSocket) _socket).setEnabledCipherSuites(cipherSuites);
		((SSLSocket) _socket).setUseClientMode(true);

		/*
		 * The underlying JSSE code returns the SSL socket before all the SSL handshakes, including client authentication, are completed. The handshake carries on, asynchronously, in the background. If the handshake fails then the socket is not usable. We synchronize things by calling getSession() on the SSL socket. The thread calling getSession() is forced to wait until the underlying SSL handshake is completed, and if the handshake fails then getSession() returns a null.
		 */
		SSLSession session = ((SSLSocket) _socket).getSession();

		/*
		 * If we could not establish a session we should throw an exception
		 */
		if (session == null) {
			throw new UntrustedAgentControllerException();
		}
		if (session.getCipherSuite().equals("SSL_NULL_WITH_NULL_NULL")) {//$NON-NLS-1$
			throw new UntrustedAgentControllerException();
		}

		_node = node;
		this.init();
	}

	/**
	 * Creates a socket channel using the current _socket state as the abstracted element
	 * 
	 * @return
	 * @throws IOException
	 */
	private ISocketChannel create() throws IOException {
		Assert.isNotNull(this._socket);
		return this.create(this._socket);
	}

	public ISocketChannel create(InetSocketAddress address) throws IOException {

		Assert.isNotNull(address);
		Assert.isNotNull(this._node);

		/*
		 * Creates a new socket channel to the given address/port combination, this new socket channel is a new connection and does not affect this secure connection implementation's connection status. It does attempt to use this secure connection implementation's credentials if possible to ease connection, as well as any of state that helps establish a new socket channel from this connection class.
		 * 
		 * The node must point to the same host, although the connection port can be different, The address host specified is ignored and only the port information is used in this case.
		 */
		SecureConnectionImpl connection = new SecureConnectionImpl() {

			public void init() {

				// Override to avoid base class socket reader thread
				try {
					this._socket.setSoTimeout(60000);
					this._socket.setTcpNoDelay(true);
					this._socket.setReuseAddress(true);
				} catch (SocketException e) {
					//
				}

			}

		};

		/*
		 * Create a new connection on a different port -- it is expected that no exceptions will occur due to already being validated on the same node as spawning secure connection impl
		 */
		try {
			connection.connect(this._node, address.getPort());
			return connection.create();
		} catch (SecureConnectionRequiredException e) {
			//
		} catch (UntrustedAgentControllerException e) {
			//
		} catch (IOException e) {
			//
		} catch (LoginFailedException e) {
			//
		}
		return null;

	}

	public ISocketChannel create(Socket socket) throws IOException {
		return SocketChannelFactory.getInstance().create(socket);
	}

	public ISocketChannel create(SocketChannel realChannel) {
		return SocketChannelFactory.getInstance().create(realChannel);
	}

	public ISocketChannel create(InetSocketAddress address, int timeout) throws IOException {
		// TODO Auto-generated method stub
		return null;
	}

	private String[] getCipherSuites() {
		Node node = getNode();
		String[] cipherSuites = null;

		ISecureClientParameters params = node.getSecurityParameters();
		if (params != null) {
			cipherSuites = params.getEnabledCipherSuites();
		}
		if (cipherSuites == null) {
			cipherSuites = ((SSLSocket) _socket).getSupportedCipherSuites();
		}

		return cipherSuites;
	}

	private SSLContext getSSLContext(String protocol) throws NoSuchAlgorithmException {
		Node node = getNode();

		KeyManager[] kms;
		TrustManager[] tms;

		/* Attach using SSL and initiate a handshake */
		SSLContext sslContext;
		sslContext = SSLContext.getInstance(protocol);

		kms = node.getSecurityParameters().getKeystoreManager().getKeyManagers();
		tms = node.getSecurityParameters().getKeystoreManager().getTrustManagers();

		try {
			sslContext.init(kms, tms, null);
		} catch (KeyManagementException e) {
			e.printStackTrace();
			return null;
		}

		return sslContext;
	}

	private String[] getSSLProtocols() {
		Node node = getNode();

		/* Determine our acceptable protocols */
		String[] sslProtocols;
		ISecureClientParameters param = node.getSecurityParameters();
		if (param != null) {
			sslProtocols = param.getEnabledProtocols();
			if (sslProtocols == null) {
				sslProtocols = new String[] { "SSL" };//$NON-NLS-1$
			}
		} else {
			sslProtocols = new String[] { "SSL" };//$NON-NLS-1$
		}

		return sslProtocols;
	}

	private SSLSocket getSSLSocket(SSLContext sslContext, int port) {
		int offset = 0;
		Node node = getNode();
		InetAddress[] addrs = node.getAllInetAddresses();
		SSLSocket sslSocket = null;
		boolean done = false;

		do {
			try {
				if (sslContext == null) {
					sslSocket = (SSLSocket) SSLSocketFactory.getDefault().createSocket(addrs[offset], port);
					sslSocket.startHandshake();
					done = true;
				} else {
					sslSocket = (SSLSocket) sslContext.getSocketFactory().createSocket(addrs[offset], port);
					done = true;
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		} while (!done && (offset < addrs.length));

		return sslSocket;
	}
}
