/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileLocaterImpl.java,v 1.4 2005/09/10 02:40:48 sschneid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local.file;


import java.io.IOException;

import org.eclipse.hyades.execution.core.file.IFileLocater;
import org.eclipse.hyades.execution.core.file.IFileLocation;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.ResourceLocation;
import org.eclipse.hyades.internal.execution.local.control.CommandHandler;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.Node;

/**
 * @author rduggan
 * This is the client side implementation of the IFileManger.
 */
public class FileLocaterImpl implements IFileLocater {
	
	
	/* The connection object which this fileManager is dealing with */
	private Connection connection;
	
	/* The maximum time to wait for a file operation to start */
	private int timout=60000;
	
	
	public FileLocaterImpl(Connection connection) {
		this.connection=connection;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.file.IFileLocater#getFile(java.lang.String)
	 */
	public IFileLocation getFile(String filename) throws IOException{
		ManageFileCommand command=new ManageFileCommand();
		command.setOperation(ManageFileCommand.GET);
		command.setFilename(filename);
		ControlMessage message= new ControlMessage();
		message.appendCommand(command);
		final FileLocationImpl result=new FileLocationImpl();
		final Object lock=new Object();
		
		try {
			synchronized(lock) {
				connection.sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node, CommandElement command) {
						if(command.getTag()==org.eclipse.hyades.internal.execution.local.common.Constants.RA_RESOURCE_LOCATION) {
							ResourceLocation location=(ResourceLocation)command;
							result.setInetAddress(node.getInetAddress());
							result.setPort(location.getPort());
							result.setJobKey(location.getJobKey());
							synchronized(lock) {
								lock.notify();
							}
						}
					}
				});
				
				lock.wait(timout);
			}
		}
		catch(IOException e) {
			/* We are not connected or the connection is broken
			 * * TODO Need to handle this properly
			 */
			throw e;
		}
		catch(InterruptedException e) {
			/* We have timed out
			 * TODO Need to handle this properly
			 */
		}
		return result;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.file.IFileLocater#putFile(java.lang.String)
	 */
	public IFileLocation putFile(String filename) throws IOException{
		ManageFileCommand command=new ManageFileCommand();
		command.setOperation(ManageFileCommand.PUT);
		command.setFilename(filename);
		ControlMessage message= new ControlMessage();
		message.appendCommand(command);
		final FileLocationImpl result=new FileLocationImpl();
		final Object lock=new Object();
		
		try {
			synchronized(lock) {
				connection.sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node, CommandElement command) {
						if(command.getTag()==org.eclipse.hyades.internal.execution.local.common.Constants.RA_RESOURCE_LOCATION) {
							ResourceLocation location=(ResourceLocation)command;
							result.setInetAddress(node.getInetAddress());
							result.setPort(location.getPort());
							result.setJobKey(location.getJobKey());
							synchronized(lock) {
								lock.notify();
							}
						}
					}
				});
				
				lock.wait(timout);
			}
		}
		catch(IOException e) {
			/* We are not connected or the connection is broken
			 * * TODO Need to handle this properly
			 */
			throw e;
		}
		catch(InterruptedException e) {
			/* We have timed out
			 * TODO Need to handle this properly
			 */
		}
		return result;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.file.IFileLocater#deleteFile(java.lang.String)
	 */
	public void deleteFile(String filename) throws IOException {
		ManageFileCommand command=new ManageFileCommand();
		command.setOperation(ManageFileCommand.DELETE);
		command.setFilename(filename);
		ControlMessage message= new ControlMessage();
		message.appendCommand(command);
		
		try {
			connection.sendMessage(message, null);
		}
		catch(IOException e) {
			/* We are not connected or the connection is broken
			 * * TODO Need to handle this properly
			 */
			throw e;
		}
	}
}
