/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerExtendedTimedImpl.java,v 1.13 2008/03/20 18:49:59 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.IOException;
import java.io.PrintStream;
import java.net.InetSocketAddress;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.timer.IPerformanceTimer;
import org.eclipse.hyades.execution.core.timer.PerformanceTimer;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.control.Connection;

/**
 * Implementation of new client-side file manager, the factory is responsible
 * for createing the appropriate concrete class that abides by the file manager
 * interface.
 * 
 * @author Scott E. Schneider
 */
class FileManagerExtendedTimedImpl extends FileManagerExtendedImpl {

	/**
	 * Performance timer instance to use for all operations in this class
	 */
	private final IPerformanceTimer timer;

	/**
	 * Number of times to invoke each method for every call into a method, used
	 * strictly for obtaining a larger sample for performance measurements
	 */
	private final int iterations;

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 */
	FileManagerExtendedTimedImpl(Connection connection)
			throws ServerNotAvailableException {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString());
		this.iterations = 1;
	}

	/**
	 * Creates a new extended file manager implementation instance with the
	 * specified file server address (IP address and port combination) -- the
	 * specified number of iterations will configure the performance timer
	 * 
	 * @param fileServerAddress
	 *            the IP address and port combination to attempt connection with
	 * @param iterations
	 *            the multiplicative scaling of the command requests (the number
	 *            of iterations that each command is executed per request)
	 */
	FileManagerExtendedTimedImpl(InetSocketAddress fileServerAddress,
			int iterations) {
		super(fileServerAddress);
		this.timer = PerformanceTimer.getInstance(this.toString());
		this.iterations = iterations;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param stream
	 *            the print stream to output elasped time information to
	 */
	FileManagerExtendedTimedImpl(Connection connection, PrintStream stream)
			throws ServerNotAvailableException {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString(), stream);
		this.iterations = 1;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param iterations
	 *            number of times to exercise each method
	 */
	FileManagerExtendedTimedImpl(Connection connection, int iterations)
			throws ServerNotAvailableException {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString());
		this.iterations = iterations;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param stream
	 *            the print stream to output elasped time information to
	 * @param iterations
	 *            number of times to exercise each method
	 */
	FileManagerExtendedTimedImpl(Connection connection, PrintStream stream,
			int iterations) throws ServerNotAvailableException {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString(), stream);
		this.iterations = iterations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(Cookie cookie, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		this.timer.start("deleteFile (using cookie to access server state)"
				+ cookie);//$NON-NLS-1$
		super.deleteFile(cookie, monitor);
		this.timer.stop();
		this.timer.elapsed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		int size = remoteIdentifiers.size();
		this.timer.start("deleteFile with " + size + " files");//$NON-NLS-1$//$NON-NLS-2$
		for (int i = 1; i < this.iterations + 1; i++) {
			this.timer.start("iteration " + i);//$NON-NLS-1$
			super.deleteFile(remoteIdentifiers, monitor);
			this.timer.stop();
			this.timer.elapsed(size);
		}
		this.timer.stop();
		this.timer.elapsed(size);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		int size = localIdentifiers.size();
		if (size > 0) {
			this.timer.start("getFile with " + size + " files named "
					+ localIdentifiers.getArray()[0] + "..."
					+ localIdentifiers.getArray()[size - 1]);//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			for (int i = 1; i < this.iterations + 1; i++) {
				this.timer.start("iteration " + i);//$NON-NLS-1$
				super.getFile(localIdentifiers, remoteIdentifiers, options,
						monitor);
				this.timer.stop();
				this.timer.elapsed(size);
			}
			this.timer.stop();
			this.timer.elapsed(size);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
	 */
	public boolean isServerAvailable() throws IOException {
		boolean isServerAvailable = false;
		this.timer.start("isServerAvailable");//$NON-NLS-1$
		for (int i = 1; i < this.iterations + 1; i++) {
			this.timer.start("iteration " + i);//$NON-NLS-1$
			isServerAvailable = super.isServerAvailable() || isServerAvailable;
			this.timer.stop();
			this.timer.elapsed();
		}
		this.timer.stop();
		this.timer.elapsed();
		return isServerAvailable;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		int size = localIdentifiers.size();
		if (size > 0) {
			this.timer.start("putFile with " + size + " files named "
					+ localIdentifiers.getArray()[0] + "..."
					+ localIdentifiers.getArray()[size - 1]);//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			for (int i = 1; i < this.iterations + 1; i++) {
				this.timer.start("iteration " + i);//$NON-NLS-1$
				super.putFile(cookie, localIdentifiers, remoteIdentifiers,
						options, monitor);
				this.timer.stop();
				this.timer.elapsed(size);
			}
			this.timer.stop();
			this.timer.elapsed(size);
		}
	}

}
