/********************************************************************** 
 * Copyright (c) 2005, 2011 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerFactory.java,v 1.30 2011/01/21 19:39:01 mreid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.File;
import java.net.InetSocketAddress;
import java.util.HashMap;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.task.ProgressiveTask;
import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.execution.local.util.AgentControllerDescriptor;
import org.eclipse.hyades.execution.local.util.IAgentControllerDescriptor;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.osgi.util.NLS;

/**
 * The file manager factory is the recommended access point for obtaining an
 * instance of the file manager to use. The file manager will begin the starting
 * timeout at a low value and increment it by a constant amount as the new and
 * legacy file server protocol is attempted, if the communication succeeds then
 * the appropriate concrete file manager is created; if the communication fails
 * it will continue alternating between the legacy and new file server protocols
 * until a communication can be made. Each set of attempts will increment the
 * timeout value by a constant value until the maximum amount of query server
 * availability tries is exceeded.
 * 
 * @author Scott E. Schneider
 */
public final class FileManagerFactory implements IFileManagerFactory {

	/**
	 * Abstract file server provider base class for the more specific file
	 * server providers defined
	 * 
	 * @author Scott E. Schneider
	 */
	private static abstract class AbstractFileServerProvider implements IFileServerProvider {

		/**
		 * Lock used to notify waiting thread of attempt completion
		 */
		final Object attemptLock;

		/**
		 * Agent controller descriptor used to gather information about the
		 * agent controller instance, such as version
		 */
		IAgentControllerDescriptor descriptor;

		/**
		 * Stores the exception encountered during the provides call
		 */
		Throwable exception;

		/**
		 * The file manager constructed to attempt to connect to
		 */
		IFileManagerExtended fileManager;

		/**
		 * Keeps track if the server is available or not
		 */
		ServerValidity serverValidity = ServerValidity.UNKNOWN;

		/**
		 * Determines if the task should be canceled or not
		 */
		private boolean shouldCancel;

		{
			this.attemptLock = new Object(); // arbitrary lock object
		}

		/**
		 * Constructs a file server provider
		 * 
		 * @param fileManager
		 *            the file manager instance to attempt to use
		 */
		private AbstractFileServerProvider(IFileManagerExtended fileManager) {
			this.fileManager = fileManager;
		}

		/**
		 * Constructs a file server provider
		 * 
		 * @param fileManager
		 *            the file manager instance to attempt to use
		 * @param descriptor
		 *            the agent controller descriptor, used to find out
		 *            descriptive information concerning the agent controller
		 *            instance
		 */
		private AbstractFileServerProvider(IFileManagerExtended fileManager, IAgentControllerDescriptor descriptor) {
			this.fileManager = fileManager;
			this.descriptor = descriptor;
		}

		/**
		 * Indicates the server validity
		 */
		public ServerValidity getServerValidity() {
			return this.serverValidity;
		}

		/**
		 * Returns the task name to use to attempt connection
		 * 
		 * @return the task name
		 */
		abstract String getTaskName();

		/**
		 * Returns the runnable to use in the progressive task
		 * 
		 * @return the runnable to use in the progressive task
		 */
		abstract Runnable getTaskRunnable();

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.IFileServerProvider#provide(int)
		 */
		public IFileManagerExtended provide(int timeout) {

			/*
			 * Construct task that will query the server within a given timeout
			 * period; cancel is monitored every three seconds via the progress
			 * monitor. For each check of cancel, the progressive task
			 * increments the work units of the progress monitor instance by
			 * some default amount, see the progressive task class for details.
			 */
			ProgressiveTask queryServerAvailability = new ProgressiveTask(this.getTaskName(), this.getTaskRunnable(),
					new NullProgressMonitor() {
						public boolean isCanceled() {
							return AbstractFileServerProvider.this.shouldCancel;
						}
					}, 3000);

			// Synchronized so lock is held as task starts to execute
			synchronized (this.attemptLock) {

				// Wait for a given timeout period then assume not available
				queryServerAvailability.execute(ProgressiveTask.Synchronicity.ASYNCHRONOUS);

				// Wait for a given timeout period
				try {
					FileSystemServices.println(NLS.bind(LocalResourceBundle.FileManagerFactory_FILE_SYSTEM_SERVICE_, this.fileManager, String.valueOf(timeout)), this);
					this.attemptLock.wait(timeout);
				} catch (InterruptedException e) {
					// No handling needed
				}

			}

			// If not complete, cancel the task and assume not available
			this.shouldCancel = (this.serverValidity == ServerValidity.UNKNOWN);

			// Handle exceptions appropriately
			if (exception != null) {
				FileSystemServices.println(
						NLS.bind(LocalResourceBundle.FileManagerFactory_EXCEPTION_ENCOUNTERED_, exception.getLocalizedMessage()), this);
			}

			// Returns either the file manager or null
			return (this.serverValidity == ServerValidity.VALID) ? this.fileManager : null;

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.IFileServerProvider#reset()
		 */
		public void reset() {
			this.fileManager.reset();
		}
	}

	/**
	 * The file server provider interface, basically a callback interface that
	 * file server provides provide
	 * 
	 * @author Scott E. Schneider
	 */
	private static interface IFileServerProvider {
		/**
		 * The server validity type safe enumeration used for tracking server
		 * validity in provision of file system services
		 * 
		 * @author Scott E. Schneider
		 */
		static final class ServerValidity {

			/**
			 * Server is confirmed to be invalid
			 */
			public static final ServerValidity INVALID = new ServerValidity();

			/**
			 * Server validity is unknown, usually moves into the invalid or
			 * valid state after provision is initiated
			 * 
			 * @see #UNKNOWN
			 * @see #VALID
			 */
			public static final ServerValidity UNKNOWN = new ServerValidity();

			/**
			 * Server is confirmed as valid and can be used via the file manager
			 */
			public static final ServerValidity VALID = new ServerValidity();

			/**
			 * Limit instantiation
			 * 
			 */
			private ServerValidity() {
			}
		}

		/**
		 * Returns last confirmed server validity state
		 * 
		 * @return the current server validity state
		 */
		public ServerValidity getServerValidity();

		/**
		 * Invoking provide returns a file manager instance to use that it
		 * provides or null if none can be accessed
		 * 
		 * @param timeout
		 *            the amount of time to wait before abandoning the attemp to
		 *            provide a file server
		 * @return the file manager instance that is provided or null if none
		 *         can be found
		 */
		public IFileManagerExtended provide(int timeout);

		/**
		 * Resets this provider in anticipation of a previous provide call,
		 * cleans up any resources that a provider deems necessary and executes
		 * any other code that is needed before another provide attempt with the
		 * same provider is needed.
		 */
		public void reset();
	}

	/**
	 * The legacy file server provider that attempts to provide a connection to
	 * the legacy file server using the legacy protocol (protocol versions less
	 * than 3.3)
	 * 
	 * @author Scott E. Schneider
	 */
	private static class LegacyFileServerProvider extends AbstractFileServerProvider {

		/**
		 * Constructs a legacy file server provider given the agent controller
		 * connection
		 * 
		 * @param connection
		 *            the agent controller connection
		 */
		private LegacyFileServerProvider(Connection connection) {
			super(new FileManagerExtendedImpl.Adapter(new FileManagerLegacy(connection)), new AgentControllerDescriptor(
					connection.getNode()));
		}

		/**
		 * Returns the task name to use for the progressive task
		 */
		String getTaskName() {
			return "Query Legacy File Server Availability"; //$NON-NLS-1$
		}

		/**
		 * Returns the runnable to use for the progressive task
		 */
		Runnable getTaskRunnable() {
			return new Runnable() {
				public void run() {

					try {

						// Query agent controller descriptor for server version
						LegacyFileServerProvider.this.serverValidity = ((LegacyFileServerProvider.this.descriptor != null
								&& LegacyFileServerProvider.this.descriptor.isVersionAtMost("3.2") || LegacyFileServerProvider.this.descriptor == null) ? ServerValidity.VALID
								: ServerValidity.UNKNOWN);

						// Attempt to use legacy protocol
						if (LegacyFileServerProvider.this.serverValidity == ServerValidity.UNKNOWN) {
							File file = File.createTempFile("tptp", "tptp");//$NON-NLS-1$
							file.deleteOnExit();
							LegacyFileServerProvider.this.fileManager.getFile(file.getAbsolutePath(), ".");

							// If it doesn't fail, then the server is likely
							// valid
							LegacyFileServerProvider.this.serverValidity = ServerValidity.VALID;
						}

					} catch (Throwable e) {

						// If it fails then the file server is invalid
						LegacyFileServerProvider.this.serverValidity = ServerValidity.INVALID;
						exception = e;

					}

					// Notify waiting threads
					synchronized (LegacyFileServerProvider.this.attemptLock) {
						LegacyFileServerProvider.this.attemptLock.notifyAll();
					}

				}
			};
		}
	}

	/**
	 * The new file server provider attempts to connect to the new file system
	 * services server using the new protocol (with dynamic class definition
	 * streaming to server as required) -- this is typically the highest
	 * priority provider (most desirable and flexible) and requires agent
	 * controller packages of at least 4.1.
	 * 
	 * @author Scott E. Schneider
	 */
	private static class NewFileServerProvider extends AbstractFileServerProvider {

		/**
		 * Constructs a new file server provider using the given agent
		 * controller connection
		 * 
		 * @param connection
		 *            the agent controller connection
		 */
		private NewFileServerProvider(Connection connection) throws ServerNotAvailableException {
			super(new FileManagerExtendedImpl(connection), new AgentControllerDescriptor(connection.getNode()));
		}

		/**
		 * Constructs a new file server provider given a file manager instance
		 * to attempt connection to
		 * 
		 * @param fileManager
		 *            the file manager to attempt to use
		 */
		private NewFileServerProvider(IFileManagerExtended fileManager) {
			super(fileManager);
		}

		/**
		 * Constructs a new file server provider given a file manager instance
		 * to attempt connection to
		 * 
		 * @param fileManager
		 *            the file manager to attempt to use
		 * @param descriptor
		 *            agent controller descriptor
		 */
		private NewFileServerProvider(IFileManagerExtended fileManager, IAgentControllerDescriptor descriptor) {
			super(fileManager, descriptor);
		}

		/**
		 * Constructs a new new file server provider given the specified file
		 * server address
		 * 
		 * @param fileServerAddress
		 *            the IP address and port to attempt connection to
		 */
		private NewFileServerProvider(InetSocketAddress fileServerAddress) {
			super(new FileManagerExtendedImpl(fileServerAddress));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskName()
		 */
		String getTaskName() {
			return "Query New File Server Availability"; //$NON-NLS-1$
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskRunnable()
		 */
		Runnable getTaskRunnable() {
			return new Runnable() {
				public void run() {

					// Query agent controller descriptor for server version
					NewFileServerProvider.this.serverValidity = ((NewFileServerProvider.this.descriptor != null
							&& NewFileServerProvider.this.descriptor.isVersionAtLeast("4.1") || NewFileServerProvider.this.descriptor == null) ? ServerValidity.VALID
							: ServerValidity.INVALID);

					// Notify waiting threads
					synchronized (NewFileServerProvider.this.attemptLock) {
						NewFileServerProvider.this.attemptLock.notifyAll();
					}

				}
			};
		}
	}

	/**
	 * The new timed file server provider attempts to use the new file server in
	 * timed mode, allowing for performance measurements to output to the
	 * console
	 * 
	 * @author Scott E. Schneider
	 */
	private static class NewTimedFileServerProvider extends NewFileServerProvider {

		/**
		 * Iterations specified for the timed file server provider
		 */
		private int iterations = 0;

		/**
		 * Constructs a new timed file server provider
		 * 
		 * @param connection
		 *            the connection to the agent controller
		 * @param iterations
		 *            the number of iterations to execute each invoked operation
		 */
		private NewTimedFileServerProvider(Connection connection, int iterations) throws ServerNotAvailableException {
			super(new FileManagerExtendedTimedImpl(connection, iterations), new AgentControllerDescriptor(connection
					.getNode()));
			this.iterations = iterations;
		}

		/**
		 * Constructs a new timed file server provider without requiring an
		 * agent controller connection instance
		 * 
		 * @param fileServerAddress
		 *            the IP address and port to use to connect to the file
		 *            server on, if not attempting to connect to the agent
		 *            controller's fie system services server, be sure that the
		 *            file server is running
		 * 
		 * @param iterations
		 */
		private NewTimedFileServerProvider(InetSocketAddress fileServerAddress, int iterations) {
			super(new FileManagerExtendedTimedImpl(fileServerAddress, iterations));
			this.iterations = iterations;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskName()
		 */
		String getTaskName() {
			return "Query New Timed File Server Availability"; //$NON-NLS-1$
		}

		/**
		 * Override the provide method scaling the timeout by the number of
		 * iterations
		 */
		public IFileManagerExtended provide(int timeout) {
			return super.provide(timeout * this.iterations);
		}

	}

	/**
	 * The provisional new file server provider attempts to connect to the new
	 * file server using the new protocol (the protocol between versions 3.3 and
	 * 4.0.1 inclusive)
	 * 
	 * @author Scott E. Schneider
	 */
	private static class ProvisionalNewFileServerProvider extends AbstractFileServerProvider {

		/**
		 * Constructs a new file server provider using the given agent
		 * controller connection
		 * 
		 * @param connection
		 *            the agent controller connection
		 */
		private ProvisionalNewFileServerProvider(Connection connection) {
			super(new ProvisionalFileManagerExtended(connection), new AgentControllerDescriptor(connection.getNode()));
		}

		/**
		 * Constructs a new file server provider given a connection and file
		 * manager instance to attempt to connect to
		 * 
		 * @param fileManager
		 *            the file manager to attempt to use]
		 * @param descriptor
		 *            the agent controller descriptor, used to query for
		 *            metadata about the server such as version
		 */
		private ProvisionalNewFileServerProvider(IFileManagerExtended fileManager, IAgentControllerDescriptor descriptor) {
			super(fileManager, descriptor);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskName()
		 */
		String getTaskName() {
			return "Query Provisional New File Server Availability"; //$NON-NLS-1$
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskRunnable()
		 */
		Runnable getTaskRunnable() {
			return new Runnable() {
				public void run() {
					try {

						if (ProvisionalNewFileServerProvider.this.descriptor.isVersionWithin("3.3", "4.0.1")) {

							// Attempt file server available operation
							ProvisionalNewFileServerProvider.this.fileManager.isServerAvailable();

							// If it doesn't fail, then the server is valid
							ProvisionalNewFileServerProvider.this.serverValidity = ServerValidity.VALID;

						} else {

							// Invalid if outside of 3.3 <= x <= 4.0.1 range
							ProvisionalNewFileServerProvider.this.serverValidity = ServerValidity.INVALID;

						}

					} catch (Throwable e) {

						// Exception leaves server validity in unknown state
						ProvisionalNewFileServerProvider.this.serverValidity = ServerValidity.UNKNOWN;
						exception = e;

					}

					// Notify waiting threads
					synchronized (ProvisionalNewFileServerProvider.this.attemptLock) {
						ProvisionalNewFileServerProvider.this.attemptLock.notifyAll();
					}

				}
			};
		}
	}

	/**
	 * The provisional new timed file server provider attempts to use the
	 * provisional new file server in timed mode, allowing for performance
	 * measurements to output to the console
	 * 
	 * @author Scott E. Schneider
	 */
	private static class ProvisionalNewTimedFileServerProvider extends ProvisionalNewFileServerProvider {

		/**
		 * Iterations specified for the timed file server provider
		 */
		private int iterations = 0;

		/**
		 * Constructs a provisional new timed file server provider
		 * 
		 * @param connection
		 *            the connection to the agent controller
		 * @param iterations
		 *            the number of iterations to execute each invoked operation
		 */
		private ProvisionalNewTimedFileServerProvider(Connection connection, int iterations) {
			super(new ProvisionalFileManagerExtendedTimed(connection, iterations), new AgentControllerDescriptor(
					connection.getNode()));
			this.iterations = iterations;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.local.file.FileManagerFactory.AbstractFileServerProvider#getTaskName()
		 */
		String getTaskName() {
			return "Query Provisional New Timed File Server Availability"; //$NON-NLS-1$
		}

		/**
		 * Override the provide method scaling the timeout by the number of
		 * iterations
		 */
		public IFileManagerExtended provide(int timeout) {
			return super.provide(timeout * this.iterations);
		}

	}

	/**
	 * The singleton file manager factory instance
	 */
	private static final IFileManagerFactory factory;

	/**
	 * Value to increment the timeout when alternating back and forth between
	 * the legacy and new protocol to determine the proper protocol to use.
	 */
	private static final int QUERY_SERVER_AVAILABILITY_TIMEOUT_INCREMENT = 15000;

	/**
	 * Starting Timeout used when determining what file manager implementation
	 * to instantiate and use (starts at a low value anticipating that most
	 * networks will be local, then increments by the timeout increment falling
	 * back to the assumption that this network might not be a local network and
	 * it needs more time to detect the proper protocol to use).
	 */
	private static final int QUERY_SERVER_AVAILABILITY_TIMEOUT_START = 7500;

	/**
	 * The number of times to interrogate the file server to determine to the
	 * appropriate protocol to use. Each try includes an attempted ping to the
	 * legacy server using the legacy protocol and an attempted ping to the new
	 * server using the legacy protocol.
	 */
	private static final int QUERY_SERVER_AVAILABILITY_TRIES = 25;

	/**
	 * Create and store the singleton instance, not using on demand creation,
	 * always create on class load
	 */
	static {
		factory = new FileManagerFactory();
	}

	/**
	 * Singleton accessor method
	 * 
	 * @return
	 */
	public static IFileManagerFactory getInstance() {
		return FileManagerFactory.factory;
	}

	/**
	 * A cache of connection specific managers, this should use a soft hash map
	 * so the cache can be reduced in times of memory needs
	 */
	private final HashMap connectionSpecificManagers = new HashMap();

	/**
	 * A cache of iteration specific managers, this should use a soft hash map
	 * so the cache can be reduced in times of memory needs
	 */
	private final HashMap iterationSpecificManagers = new HashMap();

	/**
	 * Limit instantiation to same class for singleton pattern
	 */
	private FileManagerFactory() {
	}

	/**
	 * Create file manager factory method, uses the entry in the cache if the
	 * connection already has a file manager cached -- if a file manager cannot
	 * be established due to providers not being able to connect to the file
	 * transfer service, null will be returned.
	 * 
	 * @param connection
	 *            connection to use for this file manager instance
	 * @return the extended file manager interface to operate on, null if none
	 *         could be found
	 */
	public synchronized IFileManagerExtended create(Connection connection) throws ServerNotAvailableException {
		IFileManagerExtended fileManager = null;

		// Look into connections-specific file manager cache first
		String cacheKey = this.getConnectionKey(connection);
		fileManager = (IFileManagerExtended) this.connectionSpecificManagers.get(cacheKey);

		// If it is cached use it, if not create a new one and add to cache
		if (fileManager == null) {
			fileManager = this.create(new NewFileServerProvider(connection), new ProvisionalNewFileServerProvider(
					connection), new LegacyFileServerProvider(connection));
			if (fileManager != null) {
				this.connectionSpecificManagers.put(cacheKey, fileManager);
			}
		}

		// Attempt new file server followed by legacy file server
		return fileManager;
	}

	/**
	 * Create an extended file manager using the provider specified, the
	 * provider will attempt to provide a file manager
	 * 
	 * @param provider
	 *            the provider used to provide the file manager
	 * @return an instance of an extended file manager ready for use
	 */
	private IFileManagerExtended create(IFileServerProvider provider) {
		return this.create(provider, provider);
	}

	/**
	 * Create a new file manager instance given an agent controller connection
	 * and a primary and secondary file server (using the respective protocol)
	 * to attempt connection with
	 * 
	 * @param primary
	 *            the first file server to attempt to use
	 * @param secondary
	 *            the second choice file server to attempt to use
	 * @return the instance of the file manager to use
	 */
	private IFileManagerExtended create(IFileServerProvider primaryProvider, IFileServerProvider secondaryProvider) {

		// Allocate a file manager to store found instance
		IFileManagerExtended fileManager = null;

		// The primary and secondary providers are used as necessary
		for (int i = 0, timeout = FileManagerFactory.QUERY_SERVER_AVAILABILITY_TIMEOUT_START; fileManager == null
				&& i < FileManagerFactory.QUERY_SERVER_AVAILABILITY_TRIES; i++) {
			fileManager = primaryProvider.provide(timeout);
			primaryProvider.reset();
			if (fileManager == null) {
				fileManager = secondaryProvider.provide(timeout);
				secondaryProvider.reset();
			}
			FileSystemServices.println(NLS.bind(LocalResourceBundle.FileManagerFactory_CREATE_FILE_MANAGER_, fileManager), this);
			timeout = timeout + QUERY_SERVER_AVAILABILITY_TIMEOUT_INCREMENT;
		}

		// Return either the new or legacy file manager instance or null if none
		return fileManager;

	}

	/**
	 * Create a new file manager instance given an agent controller connection
	 * and a primary and secondary file server (using the respective protocol)
	 * to attempt connection with
	 * 
	 * @param primary
	 *            the first file server to attempt to use
	 * @param secondary
	 *            the second choice file server to attempt to use
	 * @param ternary
	 *            the third choice file server to attempt to use
	 * @return the instance of the file manager to use or null if known can be
	 *         acquired for use
	 */
	private IFileManagerExtended create(IFileServerProvider primaryProvider, IFileServerProvider secondaryProvider,
			IFileServerProvider ternaryProvider) {

		// Allocate a file manager to store found instance
		IFileManagerExtended fileManager = null;

		// The primary and secondary providers are used as necessary
		for (int i = 0, timeout = FileManagerFactory.QUERY_SERVER_AVAILABILITY_TIMEOUT_START; fileManager == null
				&& i < FileManagerFactory.QUERY_SERVER_AVAILABILITY_TRIES; i++) {
			fileManager = primaryProvider.provide(timeout);
			primaryProvider.reset();
			if (fileManager == null) {
				fileManager = secondaryProvider.provide(timeout);
				secondaryProvider.reset();
				if (fileManager == null) {
					fileManager = ternaryProvider.provide(timeout);
					ternaryProvider.reset();
				}
			}
			timeout = timeout + QUERY_SERVER_AVAILABILITY_TIMEOUT_INCREMENT;
		}

		// Report file manager eventually selected to debug out
		FileSystemServices.println(NLS.bind(LocalResourceBundle.FileManagerFactory_CREATE_FILE_MANAGER_, fileManager), this);

		// Return either the new or legacy file manager instance or null if none
		return fileManager;

	}

	/**
	 * Create a new extended file manager instance given the file server address
	 * to use (this specifies the IP address and the port) -- this is typically
	 * used for testing where the agent controller is not necessary up and
	 * running, although it can be used instead of the factory method that takes
	 * an agent controller connection.
	 * 
	 * @see #create(Connection)
	 */
	public synchronized IFileManagerExtended create(InetSocketAddress fileServerAddress) {
		return this.create(new NewFileServerProvider(fileServerAddress));
	}

	/**
	 * Creates a file manager used for performance timing measurements
	 * 
	 * @param connection
	 *            the connection to the agent controller
	 * @param iterations
	 *            the number of times each method will be exercised per
	 *            invocation in
	 * @return the file manager configured to use
	 */
	public synchronized IFileManagerExtended createTimed(Connection connection, int iterations) throws ServerNotAvailableException {
		IFileManagerExtended fileManager = null;

		// Look into iteration-specific file manager cache first
		String key = this.getConnectionIterationKey(connection, iterations);
		fileManager = (IFileManagerExtended) this.iterationSpecificManagers.get(key);

		// If it is cached use it, if not create a new one and add to cache
		if (fileManager == null) {
			fileManager = this.create(new NewTimedFileServerProvider(connection, iterations),
					new ProvisionalNewTimedFileServerProvider(connection, iterations), new LegacyFileServerProvider(
							connection));
			if (fileManager != null) {
				this.iterationSpecificManagers.put(key, fileManager);
			}
		}

		// Return either the new timed or legacy file manager instance
		return fileManager;

	}

	public synchronized IFileManagerExtended createTimed(InetSocketAddress fileServerAddress, int iterations) {
		return this.create(new NewTimedFileServerProvider(fileServerAddress, iterations));
	}

	/**
	 * Calculate connection-iteration key for use in iteration-specific hash map
	 * cache, concatenate connection identity with iterations taken as a string
	 * using a separator that will prevent combination values to ever be
	 * mistaken for other overlapping combinations (make the key unique per
	 * connection-iteration combination)
	 * 
	 * @param connection
	 *            the connection of interest
	 * @param iterations
	 *            the iterations of interest
	 * @return the string key to use in the cache, for connection specific
	 *         cache, see the peer method
	 * 
	 * @see #getConnectionKey(Connection)
	 */
	private String getConnectionIterationKey(Connection connection, int iterations) {
		return this.getConnectionKey(connection).toString().concat("~").concat(String.valueOf(iterations));
	}

	/**
	 * Calculate connection key for use in connection-specific hash map cache,
	 * retrieves the textual representation of the host this connection connects
	 * to, this will be sufficient uniquess for an appropriate cache.
	 * 
	 * BUG 144286 - Host string is not sufficiently unique. The AC may be reconfigured
	 * (e.g. change port, or insecure to secure connection mode) while the workbench is
	 * running. Add fully qualified class name and port # to the key to disambiguate these
	 * cases.
	 * 
	 * @param connection
	 *            the connection of interest
	 * @return a cache key to be used, for the iteration specific cache, see the
	 *         peer method
	 * 
	 * @see #getConnectionIterationKey(Connection, int)
	 */
	private String getConnectionKey(Connection connection) {
		return connection.getClass().getName() + "[" + connection.getNode().getInetAddress().toString() + ":" + connection.getPort() + "]";
	}

}