/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileManagerLocal.java,v 1.10 2008/03/20 18:49:59 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;

public class FileManagerLocal implements IFileManagerExtended {

	private void copyFile(FileChannel source, FileChannel destination) {
		try {
			source.transferTo(0, source.size(), destination);
			source.close();
			destination.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * This MUST take a fully qualified path
	 * 
	 * @param file
	 */
	private void prepareFile(File file) {
		// if (!file.getParentFile().exists()) {
		// file.getParentFile().mkdirs();
		// }

		if (file.isDirectory()) {
			file.mkdirs();
		} else {
			if (file.getParentFile() != null) {
				file.getParentFile().mkdirs();
			}
		}

		// if (file.exists()) {
		// file.delete();
		// }
	}

	public void deleteDirectory(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		deleteDirectory(remoteIdentifiers, null);
	}

	public void deleteDirectory(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		String remoteFiles[] = remoteIdentifiers.getArray();
		for (int i = 0; i < remoteFiles.length; i++) {
			deleteFile(remoteFiles[i]);
		}
	}

	public void deleteFile(Cookie cookie) throws IOException,
			ServerNotAvailableException {
		// TODO: Not yet implemented
	}

	public void deleteFile(Cookie cookie, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		// TODO: Not yet implemented
	}

	public void deleteFile(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		deleteFile(remoteIdentifiers, null);
	}

	public void deleteFile(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		String remoteFiles[] = remoteIdentifiers.getArray();
		for (int i = 0; i < remoteFiles.length; i++) {
			deleteFile(remoteFiles[i]);
		}
	}

	public void deleteFile(String absFileName) throws IOException {
		File file = new File(absFileName);
		if (file.exists()) {
			file.delete();
		}
	}

	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		int numLocalFiles = localIdentifiers.size();
		int numRemoteFiles = remoteIdentifiers.size();
		int numToCopy = numLocalFiles > numRemoteFiles ? numRemoteFiles
				: numLocalFiles; // make sure the array are the same size

		/*
		 * Copy the files
		 */
		for (int i = 0; i < numToCopy; i++) {
			String localFileName = localIdentifiers.get(i);
			String remoteFileName = remoteIdentifiers.get(i);

			/*
			 * Make sure remote paths are absolute
			 */
			File remoteFile = new File(remoteFileName);

			/*
			 * Make sure file path is absolute
			 */
			if (!remoteFile.isAbsolute()) {
				remoteFileName = new String(System
						.getProperty("java.io.tmpdir")//$NON-NLS-1$
						+ File.separator + remoteFileName);
				remoteIdentifiers.remove(i);
				remoteIdentifiers.add(i, remoteFileName); // Substitute the
															// file name with
															// its absolute path
			}

			getFile(localFileName, remoteFileName);
		}
	}

	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		getFile(localIdentifiers, remoteIdentifiers);
	}

	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		getFile(localIdentifiers, remoteIdentifiers);
	}

	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		getFile(localIdentifiers, remoteIdentifiers);
	}

	public void getFile(String localName, String remoteName) throws IOException {
		File localFile = new File(localName);
		String remoteNameResolved = remoteName;

		File remoteFile = new File(remoteNameResolved);

		/*
		 * Make sure file path is absolute
		 */
		if (!remoteFile.isAbsolute()) {
			remoteNameResolved = new String(System
					.getProperty("java.io.tmpdir")//$NON-NLS-1$
					+ File.separator + remoteName);
			remoteFile = new File(remoteNameResolved);
		}

		/*
		 * Make sure we are not copying over ourselves
		 */
		if (localFile.equals(remoteFile)) {
			return;
		}

		/*
		 * Make sure the parent directory of the local file exists
		 */
		prepareFile(localFile);

		/*
		 * Use the java.nio to copy the files
		 */
		FileInputStream sourceStream = new FileInputStream(remoteFile);
		FileOutputStream destinationStream = new FileOutputStream(localFile);
		copyFile(sourceStream.getChannel(), destinationStream.getChannel());
	}

	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier)
			throws IOException, ServerNotAvailableException {
		return new boolean[0];
	}

	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		return new boolean[0];
	}

	public boolean isServerAvailable() throws IOException {
		return true; // TODO: Local file server is always available
	}

	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		return listContent(remoteIdentifiers, null);
	}

	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		return remoteIdentifiers;
	}

	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective) throws IOException,
			ServerNotAvailableException {
		// TODO: No need to modify permission for local case?
	}

	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		// TODO: No need to modify permission for local case?
	}

	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		int numLocalFiles = localIdentifiers.size();
		int numRemoteFiles = remoteIdentifiers.size();
		int numToCopy = numLocalFiles > numRemoteFiles ? numRemoteFiles
				: numLocalFiles; // make sure the array are the same size

		/*
		 * Copy the files
		 */
		for (int i = 0; i < numToCopy; i++) {
			String localFileName = localIdentifiers.get(i);
			String remoteFileName = remoteIdentifiers.get(i);

			/*
			 * Make sure remote paths are absolute
			 */
			File remoteFile = new File(remoteFileName);

			/*
			 * Make sure file path is absolute
			 */
			if (!remoteFile.isAbsolute()) {
				remoteFileName = new String(System
						.getProperty("java.io.tmpdir")//$NON-NLS-1$
						+ File.separator + remoteFileName);
				remoteIdentifiers.remove(i);
				remoteIdentifiers.add(i, remoteFileName); // Substitute the
															// file name with
															// its absolute path
			}

			putFile(localFileName, remoteFileName);
		}
	}

	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		putFile(localIdentifiers, remoteIdentifiers);
	}

	public void putFile(String localName, String remoteName) throws IOException {
		File localFile = new File(localName);
		String remoteNameResolved = remoteName;

		File remoteFile = new File(remoteNameResolved);

		/*
		 * Make sure file path is absolute
		 */
		if (!remoteFile.isAbsolute()) {
			remoteNameResolved = new String(System
					.getProperty("java.io.tmpdir")//$NON-NLS-1$
					+ File.separator + remoteName);
			remoteFile = new File(remoteNameResolved);
		}

		/*
		 * Make sure we are not copying over ourselves
		 */
		if (localFile.equals(remoteFile)) {
			return;
		}

		/*
		 * Make sure the parent directory of the local file exists
		 */
		prepareFile(remoteFile);

		/*
		 * Use the java.nio to copy the files
		 */
		FileInputStream sourceStream = new FileInputStream(localFile);
		FileOutputStream destinationStream = new FileOutputStream(remoteFile);
		copyFile(sourceStream.getChannel(), destinationStream.getChannel());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#reset()
	 */
	public void reset() {
		//
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#determineServerReach(java.lang.String,
	 *      int)
	 */
	public boolean determineServerReach(String host, int port) {
		return true;
	}

}
