/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ProvisionalFileManagerExtended.java,v 1.7 2006/07/29 01:57:49 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileLocation;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.internal.execution.core.file.FileServerCommandFactory;
import org.eclipse.hyades.internal.execution.core.file.IFileServerCommand;
import org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory;
import org.eclipse.hyades.internal.execution.core.file.IQueryServerStatusCommand;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.ResourceLocation;
import org.eclipse.hyades.internal.execution.local.control.CommandHandler;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.Node;

/**
 * File manager extended implementation class used for file manager operations
 * and intended to be created via a factory such as the file manager factory.
 * This implements extends the legacy file manager implementation and implements
 * a new interface that extends from the legacy interface. Refer to the file
 * manager extended interface for more details and documentation. Also refer to
 * the file manager extended test within the org.eclipse.hyades.test.
 * execution.local package.
 * 
 * @author Scott E. Schneider
 */
class ProvisionalFileManagerExtended implements IFileManagerExtended {

	/**
	 * An adapter that adapts a legacy file manager implementation so it
	 * implements the file manager extended interface
	 * 
	 * @author Scott E. Schneider
	 */
	static class Adapter implements IFileManagerExtended {

		/**
		 * The legacy implementation to delegate calls to
		 */
		private final IFileManager fileManager;

		/**
		 * Wrap the legacy file manager with new interface
		 * 
		 * @param fileManager
		 *            the legacy file manager to delegate calls into
		 */
		Adapter(IFileManager fileManager) {
			this.fileManager = fileManager;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
		 *      #deleteDirectory(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public void deleteDirectory(FileIdentifierList remoteIdentifiers)
				throws IOException, ServerNotAvailableException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
		 *      #deleteDirectory(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void deleteDirectory(FileIdentifierList remoteIdentifiers,
				IProgressMonitor monitor) throws IOException,
				ServerNotAvailableException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie)
		 */
		public void deleteFile(Cookie cookie) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void deleteFile(Cookie cookie, IProgressMonitor monitor)
				throws IOException {

			// Validate arguments
			if (cookie == null || monitor == null) {
				throw new NullPointerException();
			}

			// Not yet implemented
			throw new UnsupportedOperationException();

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public void deleteFile(FileIdentifierList remoteIdentifiers)
				throws IOException {
			for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers
					.hasNext();) {
				String remoteFileName = (String) identifiers.next();
				this.deleteFile(remoteFileName);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void deleteFile(FileIdentifierList remoteIdentifiers,
				IProgressMonitor monitor) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManager#deleteFile(java.lang.String)
		 */
		public void deleteFile(String remoteFileName) throws IOException {
			this.fileManager.deleteFile(remoteFileName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#determineServerReach(java.lang.String,
		 *      int)
		 */
		public boolean determineServerReach(String host, int port) {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public void getFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers) throws IOException {
			for (Iterator localIdentifier = localIdentifiers.iterator(), remoteIdentifier = remoteIdentifiers
					.iterator(); localIdentifier.hasNext();) {
				String localFileName = (String) localIdentifier.next();
				String remoteFileName = (String) remoteIdentifier.next();
				this.getFile(localFileName, remoteFileName);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void getFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
		 */
		public void getFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void getFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options,
				IProgressMonitor monitor) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManager#getFile(java.lang.String,
		 *      java.lang.String)
		 */
		public void getFile(String localName, String remoteName)
				throws IOException {
			this.fileManager.getFile(localName, remoteName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#
		 *      isDirectoryExistent(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier)
				throws IOException, ServerNotAvailableException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
		 *      #isDirectoryExistent(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public boolean[] isDirectoryExistent(
				FileIdentifierList remoteIdentifier, IProgressMonitor monitor)
				throws IOException, ServerNotAvailableException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
		 */
		public boolean isServerAvailable() throws IOException {
			return true;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public FileIdentifierList listContent(
				FileIdentifierList remoteIdentifiers) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public FileIdentifierList listContent(
				FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      java.lang.String)
		 */
		public void modifyPermission(FileIdentifierList remoteIdentifiers,
				String permissionDirective) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void modifyPermission(FileIdentifierList remoteIdentifiers,
				String permissionDirective, IProgressMonitor monitor)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
		 */
		public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options,
				IProgressMonitor monitor) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
		 */
		public void putFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers) throws IOException {
			for (Iterator localIdentifier = localIdentifiers.iterator(), remoteIdentifier = remoteIdentifiers
					.iterator(); localIdentifier.hasNext();) {
				String localFileName = (String) localIdentifier.next();
				String remoteFileName = (String) remoteIdentifier.next();
				this.putFile(localFileName, remoteFileName);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void putFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
		 */
		public void putFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options)
				throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
		 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
		 *      org.eclipse.core.runtime.IProgressMonitor)
		 */
		public void putFile(FileIdentifierList localIdentifiers,
				FileIdentifierList remoteIdentifiers, Option[] options,
				IProgressMonitor monitor) throws IOException {
			throw new UnsupportedOperationException();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManager#putFile(java.lang.String,
		 *      java.lang.String)
		 */
		public void putFile(String localName, String remoteName)
				throws IOException {
			this.fileManager.putFile(localName, remoteName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#reset()
		 */
		public void reset() {
			//
		}

	}

	/**
	 * Amount of time in milliseconds to wait for the agent controller to
	 * respond to the resource location command request to be completed
	 */
	private static final int DEFAULT_TIMEOUT = 60000;

	/**
	 * To save the cost of newing off many null progress monitor instances, one
	 * is created and used and treated as immutable (the state will never be
	 * read).
	 */
	private static final NullProgressMonitor NULL_PROGRESS_MONITOR;

	/**
	 * Initialize singleton kept instance of null progress monitor, within the
	 * file manager extended implementation class to be reused for all cases
	 * where a progress monitor is not specified.
	 */
	static {
		NULL_PROGRESS_MONITOR = new NullProgressMonitor();
	}

	/**
	 * The associated file server command factory to use for command execution
	 */
	private final IFileServerCommandFactory factory;

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 */
	ProvisionalFileManagerExtended(Connection connection) {
		this.factory = FileServerCommandFactory.getInstance(this
				.queryFileServerLocation(connection));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
	 *      #deleteDirectory(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public void deleteDirectory(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		deleteDirectory(remoteIdentifiers,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
	 *      #deleteDirectory(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteDirectory(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {

		// Validate arguments
		if (remoteIdentifiers == null || monitor == null) {
			throw new NullPointerException();
		}

		// Create delete file command, execute and then dispose
		IFileServerCommand command = this.factory.createDeleteDirectoryCommand(
				remoteIdentifiers, monitor);
		command.execute();
		command.dispose();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie)
	 */
	public void deleteFile(Cookie cookie) throws IOException,
			ServerNotAvailableException {
		this.deleteFile(cookie,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(Cookie cookie, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {

		// Validate arguments
		if (cookie == null || monitor == null) {
			throw new NullPointerException();
		}

		// Not yet implemented
		throw new UnsupportedOperationException();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		this.deleteFile(remoteIdentifiers,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {

		// Validate arguments
		if (remoteIdentifiers == null || monitor == null) {
			throw new NullPointerException();
		}

		// Create delete file command, execute and then dispose
		IFileServerCommand command = this.factory.createDeleteFileCommand(
				remoteIdentifiers, monitor);
		command.execute();
		command.dispose();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManager#deleteFile(java.lang.String)
	 */
	public void deleteFile(String remoteName) throws IOException {
		try {
			this.deleteFile(FileIdentifierList.create(remoteName));
		} catch (ServerNotAvailableException e) {
			throw new IOException(e.getLocalizedMessage());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#determineServerReach(java.lang.String, int)
	 */
	public boolean determineServerReach(String host, int port) {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		this.getFile(localIdentifiers, remoteIdentifiers,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		this.getFile(localIdentifiers, remoteIdentifiers,
				IFileManagerExtended.Option.NONE, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		this.getFile(localIdentifiers, remoteIdentifiers, options,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void getFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {

		// Validate arguments
		if (localIdentifiers == null || remoteIdentifiers == null
				|| options == null || monitor == null) {
			throw new NullPointerException();
		} else {
			if (localIdentifiers.size() != remoteIdentifiers.size()) {
				throw new IllegalArgumentException();
			}
		}

		// Create get file command, execute it and dispose it
		IFileServerCommand command = this.factory.createGetFileCommand(
				localIdentifiers, remoteIdentifiers, options, monitor);
		command.execute();
		command.dispose();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManager#getFile(java.lang.String,
	 *      java.lang.String)
	 */
	public void getFile(String localName, String remoteName) throws IOException {
		try {
			this.getFile(FileIdentifierList.create(localName),
					FileIdentifierList.create(remoteName));
		} catch (ServerNotAvailableException e) {
			throw new IOException(e.getLocalizedMessage());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
	 *      #isDirectoryExistent(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier)
			throws IOException, ServerNotAvailableException {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended
	 *      #isDirectoryExistent(org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public boolean[] isDirectoryExistent(FileIdentifierList remoteIdentifier,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
	 */
	public boolean isServerAvailable() throws IOException {
		IQueryServerStatusCommand command = null;
		try {
			command = this.factory.createQueryServerStatusCommand();
			command.execute();
		} catch (ServerNotAvailableException e) {
			command = null;
		} finally {
			if (command != null) {
				command.dispose();
			}
		}
		command.dispose();
		return (command != null ? command.isServerAvailable() : false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers)
			throws IOException, ServerNotAvailableException {
		return this.listContent(remoteIdentifiers,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {

		// Validate arguments
		if (remoteIdentifiers == null || monitor == null) {
			throw new NullPointerException();
		}

		// Not yet implemented
		throw new UnsupportedOperationException();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      java.lang.String)
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective) throws IOException,
			ServerNotAvailableException {
		this.modifyPermission(remoteIdentifiers, permissionDirective,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void modifyPermission(FileIdentifierList remoteIdentifiers,
			String permissionDirective, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {

		// Validate arguments
		if (remoteIdentifiers == null || permissionDirective == null
				|| monitor == null) {
			throw new NullPointerException();
		} else {
			if (permissionDirective.trim().length() <= 0) {
				throw new IllegalArgumentException();
			}
		}

		// Not implemented yet
		throw new UnsupportedOperationException();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		this.putFile(cookie, localIdentifiers, remoteIdentifiers, Option.NONE);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		this.putFile(cookie, localIdentifiers, remoteIdentifiers, Option.NONE,
				monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		this.putFile(cookie, localIdentifiers, remoteIdentifiers, options,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {

		// Validate arguments
		if (cookie == null || localIdentifiers == null
				|| remoteIdentifiers == null || options == null
				|| monitor == null) {
			throw new NullPointerException();
		} else {
			if (localIdentifiers.size() != remoteIdentifiers.size()) {
				throw new IllegalArgumentException();
			}
		}

		// Create put file command, execute and dispose it
		IFileServerCommand command = this.factory.createPutFileCommand(cookie,
				localIdentifiers, remoteIdentifiers, options, monitor);
		command.execute();
		command.dispose();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers) throws IOException,
			ServerNotAvailableException {
		this.putFile(localIdentifiers, remoteIdentifiers,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
			throws IOException, ServerNotAvailableException {
		this.putFile(localIdentifiers, remoteIdentifiers,
				IFileManagerExtended.Option.NONE, monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options)
			throws IOException, ServerNotAvailableException {
		this.putFile(localIdentifiers, remoteIdentifiers, options,
				ProvisionalFileManagerExtended.NULL_PROGRESS_MONITOR);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(FileIdentifierList localIdentifiers,
			FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException,
			ServerNotAvailableException {
		this.putFile(Cookie.NONE, localIdentifiers, remoteIdentifiers, options,
				monitor);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManager#putFile(java.lang.String,
	 *      java.lang.String)
	 */
	public void putFile(String localName, String remoteName) throws IOException {
		try {
			this.putFile(FileIdentifierList.create(localName),
					FileIdentifierList.create(remoteName));
		} catch (ServerNotAvailableException e) {
			throw new IOException(e.getLocalizedMessage());
		}
	}

	/**
	 * Query the file server location and other information important for
	 * locating and connecting to the file server, the port is variable and is
	 * usually changed by setting a new port in the agent controller
	 * configuration files
	 * 
	 * @param connection
	 *            the connection to use to communicate with the agent controller
	 * @return the file server location
	 */
	private IFileLocation queryFileServerLocation(final Connection connection) {

		// Create a command just to receive resource location command
		ManageFileCommand command = new ManageFileCommand();
		command.setOperation(ManageFileCommand.PUT);
		command.setFilename("");
		ControlMessage message = new ControlMessage();
		message.appendCommand(command);

		// Allocate a result to store the file server location object
		final FileLocationImpl result = new FileLocationImpl();
		final Object lock = new Object();

		// Send the message to the agent controller and wait for response
		try {

			// Command is asynchronous, need to wait for response
			synchronized (lock) {

				// Send the message with anonymous implementation handler
				connection.sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node,
							CommandElement command) {
						if (command.getTag() == org.eclipse.hyades.internal.execution.local.common.Constants.RA_RESOURCE_LOCATION) {
							ResourceLocation location = (ResourceLocation) command;
							result.setInetAddress(node.getInetAddress());
							result.setPort(location.getPort());
							result.setJobKey(location.getJobKey());
							synchronized (lock) {
								lock.notify();
							}
						}
					}
				});

				// Wait up to the timeout value
				lock.wait(ProvisionalFileManagerExtended.DEFAULT_TIMEOUT);

			}

		} catch (IOException e) {

			return null; // a connection is not possible, setting to null

		} catch (InterruptedException e) {

			return null; // a null location indicates connection not possible

		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#reset()
	 */
	public void reset() {
		this.factory.reset();
	}

}