/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AgentControllerDescriptor.java,v 1.2 2005/09/24 05:43:17 sschneid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.local.util;

import java.util.StringTokenizer;

import org.eclipse.hyades.internal.execution.local.common.SetNVPairCommand;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;

/**
 * The agent controller descriptor class wraps a node instance and queries it
 * for agent controller metadata as required. Information such as agent
 * controller version can be discovered.
 * 
 * @author Scott E. Schneider
 * @author Joseph P. Toomey
 * @author Bianca Xue Jiang
 */
public class AgentControllerDescriptor implements IAgentControllerDescriptor {

	/**
	 * <code>ORG_ECLIPSE_HYADES_DATACOLLECTION</code> -- plugin that is
	 * queried when determining version of the agent controller before 4.0.
	 */
	private static final String ORG_ECLIPSE_HYADES_DATACOLLECTION = "org.eclipse.hyades.datacollection"; //$NON-NLS-1$

	/**
	 * <code>ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK</code> -- plugin
	 * that is queried when determining version of the agent controller in 4.0
	 * or later
	 */
	private static final String ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK = "org.eclipse.hyades.execution"; //$NON-NLS-1$

	/**
	 * <code>VERSION</code> -- string that is searched for when determining
	 * version of the agent controller
	 */
	private static final String VERSION = "version"; //$NON-NLS-1$

	/**
	 * Determines if the version passed in, as a string, is at least the version
	 * required that is passed in
	 * 
	 * @param required
	 *            the inclusive lower limit
	 * @param version
	 *            the version that must be at least the lower limit value
	 * @return indicates if the version specified meets the required version,
	 *         since the version is represented by a string, a simple numeric
	 *         comparison is not good enough (supports major and minor
	 *         versioning such 3.3 and 4.0)
	 */
	private static boolean versionIsAtLeast(String required, String version) {

		if (required == null || version == null) {
			throw new NullPointerException();
		}

		StringTokenizer stReq = new StringTokenizer(required, "."); //$NON-NLS-1$
		StringTokenizer stVer = new StringTokenizer(version, "."); //$NON-NLS-1$
		/*
		 * Check major version
		 */
		int majReq = 0;
		int majVer = -1;
		// Get the major versions
		if (stReq.hasMoreTokens() && stVer.hasMoreTokens()) {
			majReq = Integer.valueOf(stReq.nextToken()).intValue();
			majVer = Integer.valueOf(stVer.nextToken()).intValue();
		} else {
			return false; // can't find major version
		}

		if (majVer < majReq) {
			return false;
		} else if (majVer > majReq) {
			return true;
		}

		int min = java.lang.Math.min(stReq.countTokens(), stVer.countTokens());
		for (int i = 0; i < min; i++) {
			int minReq = 0;
			int minVer = -1;
			// Get the minor versions
			if (stReq.hasMoreTokens()) { // do this only if minor version
				// exists
				minReq = Integer.valueOf(stReq.nextToken()).intValue();
				if (stVer.hasMoreTokens()) {
					minVer = Integer.valueOf(stVer.nextToken()).intValue();
					// + minVer ); //$NON-NLS-1$
					if (minVer < minReq) {
						return false;
					} else if (minVer > minReq) {
						return true;
					} else if (i == (min - 1)) // last time thru, check for
					// equality
					{
						if (!stReq.hasMoreTokens()) {
							return minVer == minReq;
						} else {
							return false;
						}

					}
				}

			}
		}
		return false;
	}

	/**
	 * The node used as the data source for the description information
	 */
	private Node node;

	/**
	 * Constructs an agent controller descriptor object, used to gather metdata
	 * about the agent controller such as version
	 * 
	 * @param node
	 *            the node to query
	 */
	public AgentControllerDescriptor(Node node) {
		this.node = node;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#getProperty(java.lang.String,
	 *      java.lang.String)
	 */
	public String getProperty(String name, String type) {
		return this.getPropertyStringValue(name, type);
	}

	/**
	 * Retrieve a string property value given a node, name and type
	 * 
	 * @param node
	 *            the node to query
	 * @param name
	 *            the name of the property
	 * @param type
	 *            the type of the property
	 * @return the property value
	 */
	private String getPropertyStringValue(String name, String type) {
		SetNVPairCommand[] values = null;
		try {
			values = this.node.getPropertyValues(name, type);
		} catch (NotConnectedException e) {
		}
		if (values != null && values.length > 0) {
			return values[0].getValue();
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#getVersion()
	 */
	public String getVersion() {
		String name_3_3 = ORG_ECLIPSE_HYADES_DATACOLLECTION;
		String name_4_0 = ORG_ECLIPSE_TPTP_PLATFORM_COLLECTION_FRAMEWORK;
		String type = VERSION;
		String value = this.getPropertyStringValue(name_4_0, type);
		if (value == null) {
			value = getPropertyStringValue(name_3_3, type);
		}
		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#isVersionAtLeast(java.lang.String)
	 */
	public boolean isVersionAtLeast(String lowerLimitInclusive) {
		return AgentControllerDescriptor.versionIsAtLeast(lowerLimitInclusive, this.getVersion());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#isVersionAtMost(java.lang.String)
	 */
	public boolean isVersionAtMost(String upperLimitInclusive) {
		String version = this.getVersion();
		return !AgentControllerDescriptor.versionIsAtLeast(upperLimitInclusive, this.getVersion())
				|| upperLimitInclusive.equals(version);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#isVersionEqual(java.lang.String)
	 */
	public boolean isVersionEqual(String version) {
		return this.getVersion().trim().equals(version.trim());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.util.IAgentControllerDescriptor#isVersionWithin(java.lang.String,
	 *      java.lang.String)
	 */
	public boolean isVersionWithin(String lowerLimitInclusive, String upperLimitInclusive) {
		return this.isVersionAtLeast(lowerLimitInclusive) && this.isVersionAtMost(upperLimitInclusive);
	}

}
