/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SecureServer.java,v 1.7 2008/03/20 18:49:49 dmorris Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.security;


import java.io.IOException;
import java.net.SocketException;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.HandshakeCompletedEvent;
import javax.net.ssl.HandshakeCompletedListener;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;


/**
 * @author rduggan
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public final class SecureServer implements Runnable {
	private SSLServerSocket sslsocket;
	private IConnectionHandler connectionHandler;
	private boolean shutdown = false;


	private IKeystoreManager loadKeystore(ISecureServerParameters params) 
		 throws java.security.KeyStoreException, 
		        java.security.NoSuchAlgorithmException,
		        java.security.UnrecoverableKeyException,
		        java.security.KeyManagementException,
		        java.security.cert.CertificateException,
		        java.io.IOException
	{
		IKeystoreManager manager=params.getKeystoreManager();
//		manager.setProvider(params.getSecurityProvider());
		manager.loadKeystore(params.getKeystoreFileName(), params.getKeystoreFilepassword()); 
		return manager;
	}


	public void init(ISecureServerParameters params) 
		throws java.security.KeyStoreException, 
				java.security.NoSuchAlgorithmException,
				java.security.UnrecoverableKeyException,
				java.security.KeyManagementException,
				java.security.cert.CertificateException,
				java.io.IOException{
		int protocolOffset = 0;
		IKeystoreManager keyManager=null;
		
		/* add the proper security provider */
//		Security.addProvider(params.getSecurityProvider());
		
		
		/* Load our keystore */
		keyManager=loadKeystore(params);
		
		/* Find the protocols we can use */ 
		String[] sslProtocols = params.getEnabledProtocols();
		if(sslProtocols==null) {
			sslProtocols=new String[] {"SSL"};//$NON-NLS-1$
		}
		
		/* Setup our SSLContext */
		SSLContext sslContext=null;
		do {
		
			try {
				sslContext=SSLContext.getInstance(sslProtocols[protocolOffset]/*, params.getSecurityProvider()*/);
				break;
			}
			catch(NoSuchAlgorithmException e) {
				protocolOffset++;
				if(protocolOffset==sslProtocols.length) {
					throw e;		
				}		
			}		
		}while(protocolOffset<sslProtocols.length);
			
		/* Initailize our context with the specified key and trust managers */
		sslContext.init(keyManager.getKeyManagers(), keyManager.getTrustManagers(), null);

		
		/* Create out Secure socket to accept connections */
		sslsocket=(SSLServerSocket)sslContext.getServerSocketFactory().createServerSocket(params.getPort());
		
		/* Configure our socket to recieve only the type of client connections specified */
		sslsocket.setNeedClientAuth(params.clientAuthenticationRequired());
		try {
			sslsocket.setWantClientAuth(params.clientAuthenticationRequired());
		} catch (Throwable e) {
			// TODO: This is a workaround on AS/400, as the 1.3.1 JVM has trouble
			// locating the setWantClientAuth() method even though it exists.
		}
		
		/* Use the cypher suites specified */
		String[] cyphers=params.getEnabledCipherSuites();
		if(cyphers!=null) {
			sslsocket.setEnabledCipherSuites(params.getEnabledCipherSuites());
		}
		else {
			sslsocket.setEnabledCipherSuites(sslsocket.getEnabledCipherSuites());
		}
		
		/* Enable session creation */
		sslsocket.setEnableSessionCreation(true);
	
		/* Determine who is going to handle incoming connections */
		connectionHandler=params.getConnectionHandler();
	}
	
	public void run() {
		
		while(true) {
			
			try {
				SSLSocket socket=(SSLSocket)sslsocket.accept();	
				
				socket.addHandshakeCompletedListener(new HandshakeCompletedListener() {

					public void handshakeCompleted(HandshakeCompletedEvent event) {
					}
					
				});
				
				//socket.startHandshake();
				
			
				/* The underlying JSSE code returns the SSL socket before all
				 * the SSL handshakes, including client authentication, are
				 * completed.  The handshake carries on, asynchronously, in
				 * the background. If the handshake fails then the socket is
				 * not usable. We synchronize things by calling getSession()
				 * on the SSL socket.  The thread calling getSession() is
				 * forced to wait until the underlying SSL handshake is completed,
				 *  and if the handshake fails then getSession() returns a null.
				 */
			    SSLSession session = socket.getSession();
			    
				if (session==null ||session.getCipherSuite().equals("SSL_NULL_WITH_NULL_NULL")) {//$NON-NLS-1$
				}
				else if(connectionHandler!=null) {
					connectionHandler.connectionAccepted(socket);
				}
			} catch (SocketException e) {
				if ( this.shutdown )
				{
					/* Synchronization ensures that this method will be last exiting method
					   in preparation for a JVM termination in the native code. */
					synchronized (this) {
						break;
					}
				}
			} catch(IOException e) {
				/* TODO Handle this exception with a log message */
			}
		}
	}

	/**
	 * Signals the server to terminate gracefully.  This method is called from
	 * the native code upon a Agent Controller shutdown.  The method is synchronized to ensure
	 * the run() method will be last exiting method, after which the JVM
	 * termination will occur.
	 */
	public synchronized void quit() {

		/* Signal the server to terminate by generating an exception on the server socket. */
		try {
			this.shutdown = true;
			if (this.sslsocket != null)
				sslsocket.close();
		} catch (Throwable e) {
		}
    }
	
}
