/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileServer.java,v 1.19 2008/03/20 18:49:59 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.file;

import org.eclipse.hyades.internal.execution.file.defaults.FileServerParametersExtendedImpl;

/**
 * This class acts as a bridge between the expected file server class from the
 * agent controller native code and the real preferred implementation of the
 * file server class. Both the file server and the implementing file servers
 * implement the expected file server interface. This was refactored to support
 * increased speed and flexibilty to test project deployment by enhancing the
 * file transfer service. Refer to FileManagerExtendedTest for a performance
 * test that also exercises all supported features of the new file transfer
 * service. Support might be given to allow switching out of file server
 * implementations/strategies at run-time in the future. For example, a bit of
 * the header could indicate the file transfer service implementation desired.
 * The name of this class can not be changed without changing the native code
 * that contains the fully-qualified name of this class embedded in it.
 * 
 * The run method creates a new thread and starts up the file server
 * implementation -- the server is then up and running waiting for connections
 * and command requests.
 * 
 * @author Scott E. Schneider
 */
public class FileServer implements IFileServer {

	/**
	 * The server port in use listening for connections
	 */
	private static int serverPort = -1;

	/**
	 * Retrieves the server port in use; assuming one file server per virtual
	 * machine, if not, will return the last started file server's port number
	 * in use
	 * 
	 * @return the file server port to connect to to access the file server, the
	 *         value will be -1 if it has not yet been assigned (which is likely
	 *         indicative that the file server is not yet running yet)
	 */
	public static synchronized int getServerPort() {
		return FileServer.serverPort;
	}

	/**
	 * Set the server port to the argument passed in
	 * 
	 * @param serverPort
	 *            the port used to listen for connections on, could be a known
	 *            port or a port that is assigned by the local machine in the
	 *            ephemeral port range
	 */
	static synchronized void setServerPort(int serverPort) {
		FileServer.serverPort = serverPort;
	}

	/**
	 * An instance of the file server, the bridge class instance to the actual
	 * file server implementation
	 */
	private IFileServer fileServer;

	/**
	 * Construct a file server class, the appropriate file server implementation
	 * is plugged-in, either the new file server extended implementation or the
	 * file server legacy implementation (that does not use java.nio)
	 */
	public FileServer() {
		this.fileServer = this.createFileServer();
	}

	/**
	 * Constructs a file server on the given port, primarily used for testing,
	 * although could be used to create other peripheral task-oriented specific
	 * file servers when needed on other non agent controller resident hosts.
	 * 
	 * @param port
	 *            the port to listen for file server requests on, a port value
	 *            of zero will pick an ephemeral port to use as offered by the
	 *            local machine
	 */
	public FileServer(int port) {
		this();
		IFileServerParameters fileServerParameters = new FileServerParametersExtendedImpl() {
			// since protected, need to instantiate subclass
		};
		if (port < 0) {
			throw new IllegalArgumentException();
		}
		fileServerParameters.setPort(port);
		this.fileServer.init(fileServerParameters);
	}

	/**
	 * Factory method to create the concrete file server instance
	 * 
	 * @return a file server instance ready for use
	 */
	private IFileServer createFileServer() {
		return this.createFileServerExtended();
	}

	/**
	 * Factory method for the extended file server implementation
	 * 
	 * @return the file server extended implementation
	 */
	private IFileServer createFileServerExtended() {
		return new FileServerExtended();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.file.IFileServer#getErrorType()
	 */
	public int getErrorType() {
		return this.fileServer.getErrorType();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.file.IFileServer#getFileServerStatus()
	 */
	public int getFileServerStatus() {
		return this.fileServer.getFileServerStatus();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.file.IFileServer#init(org.eclipse.hyades.internal.execution.file.IFileServerParameters)
	 */
	public void init(IFileServerParameters parameters) {
		this.fileServer.init(parameters);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.file.IFileServer#quit()
	 */
	public void quit() {
		this.fileServer.quit();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.file.IFileServer#run()
	 */
	public void run() {

		// Create a new thread to run the server's run method
		Thread runFileServer = new Thread() {
			public void run() {
				FileServer.this.fileServer.run();
			}
		};
		runFileServer.setName("File System Services Connection Server");//$NON-NLS-1$
		runFileServer.setDaemon(true);

		// Start the file server
		runFileServer.start();

	}

}