/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileServerLegacy.java,v 1.3 2005/05/20 22:53:09 jptoomey Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.file;

import java.io.IOException;
import java.net.BindException;
import java.net.ServerSocket;
import java.net.Socket;

import org.eclipse.hyades.execution.security.IConnectionHandler;
import org.eclipse.hyades.internal.execution.local.common.Constants;

/**
 * @author Giridhar.S
 *
 * This is the file server class which is instantiated from Agent Controller during startup, and listens for 
 * incoming file transfer requests from clients, and creates client handlers for each client requests.
 * The client handlers perform the actual file transfer.
 * 
 */
class FileServerLegacy implements Runnable, FileServiceConstants, Constants, IFileServer {
	
	/* The port on which the file server listens for incoming requests. Default = 10005 */
	private int port;
	
	/* The server socket on which the server listens for requests from clients */
	private ServerSocket serverSocket;

	/* The single file connection handler, which accepts recieves client information and
	 * passes these to a file client handler which does the file processing */	
	private IConnectionHandler connHandler;
	
	/* The status of the file server */
	private int serverStatus; 
	private int errorType;
	private int initStatus;
	
	FileServerLegacy() {
		super();
		serverStatus = RA_FS_STATUS_OK;
		errorType = RA_FS_STATUS_OK;
		initStatus = RA_FS_UNINITIALIZED;
	}
	
	/**
	 * Method to initialize file server with parameters.
	 * @param ifsp - FileServerParametersImpl object containing required information.
	 */
	/* The init method is called from native code (Agent Controller) */
	
	public void init(IFileServerParameters ifsp)	{
		/* We get references to FileConnectionHandlerImpl and the configured file transfer port*/
		connHandler = (IConnectionHandler) ifsp.getConnectionHandler();
		port = ifsp.getPort();
	}
	
	/* The file server just listens for incoming requests on the configured port */
	public synchronized void run() {
		
		try {
			serverSocket = new ServerSocket(port);
			
			/* Since the file server is implemented as a thread, the file server status 
			 * might be requested before the file server thread has started completely. To
			 * guard against this inconsistency, if the file server is not intialized and
			 * a status check is requested, we block till the file server has bound to
			 * the server socket (i.e. initialized), and then we notify all waiting threads. */
			
			initStatus = RA_FS_INITIALIZED;
			this.notifyAll();
		
			/* Run forever */
			while (true){
				Socket newSocket = serverSocket.accept();							/* Wait for client requests */
				connHandler.connectionAccepted(newSocket);
			}
		}
		catch (BindException be)	{
			/* The user might want to use another port */
			serverStatus = RA_FS_STATUS_ERROR;
			errorType = RA_BIND_EXCEPTION;
			
			/* We have to notify all waiting threads */
			initStatus = RA_FS_INITIALIZED;
			this.notifyAll();
		}
		catch (IOException e) {
			serverStatus = RA_FS_STATUS_ERROR;
			errorType = RA_IO_ERROR;
			
			/* We have to notify all waiting threads */
			initStatus = RA_FS_INITIALIZED;
			this.notifyAll();
		}
	}
	
	/* Returns the status of the file server */
	public synchronized int getFileServerStatus()	{
		/* We wait for a specific amount of time, and then move on */
		final long timeout = 5000L;
		
		if (initStatus != RA_FS_INITIALIZED)
			try {
				this.wait(timeout);	
			} catch (InterruptedException e) {
				/* Should not happen */
			}
		return serverStatus;
	}
	
	/* Returns the type of error encountered. Might be useful to give a hint to
	 * the user about the type of error that occurred */
	public int getErrorType()	{
		return errorType;
	}
	
	/* Quit method called from RAC */
	public synchronized void quit()	{
		try {
			serverSocket.close();
		} catch (IOException e) {
			/* Is this necessary? */
		}
	}
}