/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileClientHandlerExtendedImpl.java,v 1.6 2008/03/21 12:52:39 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.file.defaults;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.Observable;
import java.util.Observer;

import org.eclipse.hyades.execution.core.loader.ScopedChannelClassLoader;
import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.internal.execution.core.file.FileSystemServices;
import org.eclipse.hyades.internal.execution.core.file.socket.ISocketChannel;
import org.eclipse.hyades.internal.execution.file.FileServiceConstants;
import org.eclipse.osgi.util.NLS;

/**
 * Handles a client request coming in to the file server, in the typical case
 * this runs on a new thread that is used for the purpose of servicing requests
 * and is not the same thread that the connection is accepted on.
 * 
 * Requires the use of reflection for any dynamic package members since the use
 * of the type directly (importing it and using it) will force a class load
 * using the non- dynamic method -- this will bring in the wrong file server
 * command interface and file server command factory at the minimum.
 * 
 * @author Scott E. Schneider
 */
class FileClientHandlerExtendedImpl implements Runnable {

	/**
	 * Channel to communicate with client
	 */
	private final ISocketChannel clientChannel;

	/**
	 * Creates a new file clietn handler implementation instance that will
	 * likely run on a new thread just to handle the request
	 * 
	 * @param clientChannel
	 *            the channel to communicate with the client
	 */
	FileClientHandlerExtendedImpl(ISocketChannel clientChannel) {
		this.clientChannel = clientChannel;
	}

	/**
	 * Executes a file system services command, appropriately consuming class
	 * metadata from provider if dynamic is specified.
	 * 
	 * @param identity
	 *            the fully qualified class name of the file server command to
	 *            execute
	 * @param isDynamic
	 *            determines if the class metadata is downloaded from the
	 *            provider to the consumer as classes are loaded and linked
	 *            dynamically
	 * @throws ClassNotFoundException
	 * @throws NoSuchMethodException
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IOException
	 */
	private void executeCommand(String identity, boolean isDynamic)
			throws ClassNotFoundException, NoSuchMethodException,
			InvocationTargetException, IllegalAccessException, IOException {

		// Allocate space for potential custom classloader use
		ScopedChannelClassLoader.Consumer consumerClassLoader = null;

		// Declare observer variable
		Observer observer = null;

		// Ready variable to store command factory class
		Class commandFactoryClass = null;

		// Switch over classloader if dynamic set to true
		if (isDynamic == true) {

			// Instantiate named identity using filer server command factory
			String scopingExpression = identity.substring(0, identity
					.lastIndexOf('.'));
			consumerClassLoader = new ScopedChannelClassLoader.Consumer(
					scopingExpression, this.clientChannel, this.clientChannel);

			// Enabled custom classloader, signify begin of loading
			if (consumerClassLoader != null) {

				// Add listeners for informative and debug purposes
				observer = new Observer() {
					public void update(Observable observable, Object argument) {
						FileSystemServices.println(
								NLS.bind(LocalResourceBundle.FileClientHandlerExtendedImpl_CLASS_PROVIDED_FROM_CLIENT_, argument), this);
					}
				};
				consumerClassLoader.addObserver(observer);

				// Enable consumer part of class loader
				consumerClassLoader.setEnabled(true);

				// Set thread's context class loader to custom one
				Thread.currentThread().setContextClassLoader(
						consumerClassLoader.getContextClassLoader());

				// Bootstrap point into the dynamic package
				commandFactoryClass = Thread
						.currentThread()
						.getContextClassLoader()
						.loadClass(
								"org.eclipse.hyades.internal.execution.core.file.dynamic.FileServerCommandFactory");//$NON-NLS-1$

			}

		} else {

			// Retrieve file server command factory for non-dynamic
			commandFactoryClass = Class
					.forName("org.eclipse.hyades.internal.execution.core.file.FileServerCommandFactory");//$NON-NLS-1$

		}

		Method getInstance = commandFactoryClass.getDeclaredMethod(
				"getInstance", null);//$NON-NLS-1$
		Object commandFactory = getInstance.invoke(null, null);

		// Invoke factory method reflectively
		Method createFileServerCommand = commandFactoryClass.getDeclaredMethod(
				"createFileServerCommand", new Class[] { String.class,
						ISocketChannel.class });//$NON-NLS-1$
		Object command = createFileServerCommand.invoke(commandFactory,
				new Object[] { identity, this.clientChannel });

		// Execute the command on the server-side reflectively
		Method execute = command.getClass().getMethod("execute", null);//$NON-NLS-1$

		// If custom classloader in use, disable to signify end of loading
		if (isDynamic == true) {
			if (consumerClassLoader != null) {
				consumerClassLoader.setEnabled(false);
				consumerClassLoader.deleteObserver(observer);
			}
		}

		// Execute file server command
		execute.invoke(command, null);


	}

	/**
	 * Service client file server request, need to bootstrap file server command
	 * factory by first figuring oupt the command by the command identity sent
	 * in the first 128 bytes of data. After this the file server command
	 * factory can take the handling from here. The command that will eventually
	 * be instantiated to handle the client request will encapsulate the process
	 * of communication and request and response handling.
	 */
	public void run() {

		// Allocate buffer to hold command identity length
		ByteBuffer identityLength = ByteBuffer.allocate(4);

		/*
		 * Load command via file server command factory and execute, this should
		 * be changed to use the simple communication abstraction that abstracts
		 * the mechanism for communication (see AbstractFileServerCommand.State)
		 */
		try {

			// A variable to track dynamic mode enablement
			boolean isDynamic = false;

			// Read command identity into buffer
			while (identityLength.hasRemaining()) {
				this.clientChannel.read(identityLength);
			}
			identityLength.flip();

			// Read command identity data now
			int length = identityLength.getInt();

			// Output debug console information about command identity
			FileSystemServices.println(
					NLS.bind(LocalResourceBundle.FileClientHandlerExtendedImpl_COMMAND_IDENTITY_LENGTH_, String.valueOf(Math.abs(length))), this);

			/*
			 * If legacy client file transfer service protocol is in use,
			 * delegate connection off to the legacy file connection handler.
			 */
			if (length >= 0 && length < 5) {
				FileSystemServices
						.println(
								LocalResourceBundle.FileClientHandlerExtendedImpl_COMMAND_BEING_DELEGATED_, this);

				// Code to delegate to legacy handler goes here
				ByteBuffer fileNameLength = ByteBuffer.allocate(4);
				while (fileNameLength.hasRemaining()) {
					this.clientChannel.read(fileNameLength);
				}
				fileNameLength.flip();
				ByteBuffer fileName = ByteBuffer.allocate(fileNameLength
						.getInt());
				while (fileName.hasRemaining()) {
					this.clientChannel.read(fileName);
				}
				ByteBuffer legacyException = ByteBuffer.allocate(1);
				legacyException.put((byte) FileServiceConstants.RA_IO_ERROR);
				legacyException.flip();
				this.clientChannel.write(legacyException);
				return; // return early if legacy

			} else {

				// Assumes that a command length will not exceed 256 bytes
				if (length >= -256 && length < 0) {

					/*
					 * A negative length indicates dynamic commands in effect,
					 * the absolute value of the negative length is the length
					 * of the command, yet the negative attribute of the number
					 * gives more information, that the command is using dynamic
					 * commands, expecting to download the command code as
					 * required
					 */
					isDynamic = true;
					length = Math.abs(length);

					/*
					 * Set up scoped socket classloader for this thread,
					 * basically this type of classloader will first check to
					 * see if the class to be loaded in a responsibility of
					 * itself, if not it will delegate the request to the next
					 * classloader in line -- this allows for classes in certain
					 * packages, for instance, to be retrieve over the socket
					 * channel (allowing for dynamic update of appropriate
					 * command class to use per client request, client and
					 * server command classes will match which is necessary so
					 * they have the same protocol version) while keeping other
					 * packaged classes using the standard classloader
					 */

				}
			}

			ByteBuffer identityData = ByteBuffer.allocate(length);
			while (identityData.hasRemaining()) {
				this.clientChannel.read(identityData);
			}
			identityData.flip();

			// Allocate a string for the command identity received
			String identity = null;

			/*
			 * The following code is a fix carried over from the legacy file
			 * client handler implementation, bugzilla_65922 -- z/OS triggered
			 * change
			 */
			try {

				// First try with UTF-8 encoding
				identity = new String(identityData.array(), 0, length, "UTF-8"); //$NON-NLS-1$

			} catch (UnsupportedEncodingException e) {

				// Any issues will trigger fallback to default encoding
				identity = new String(identityData.array(), 0, length);

			}

			// Clear up the buffers (not seemingly required)
			identityLength.clear();
			identityData.clear();

			// Execute command using static or dynamically loaded classes
			this.executeCommand(identity, isDynamic);

		} catch (Throwable t) {

			t.printStackTrace();

		} finally {

			try {

				// Clean-up appropriately
				this.clientChannel.close();

			} catch (IOException e) {

				e.printStackTrace();

			}

		}

	}

}