/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Console.java,v 1.15 2010/03/12 16:37:43 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.common;

/**
 * Insert the type's description here. Creation date: (2/19/01 2:42:57 PM)
 * 
 * @author: Richard K. Duggan
 */
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.tptp.platform.iac.administrator.AdminPlugin;
import org.eclipse.tptp.platform.iac.administrator.internal.common.CommonConstants;
import org.osgi.framework.Bundle;

import org.eclipse.hyades.internal.execution.local.control.Process;

public class Console extends Thread implements Constants {

	protected ServerSocket _sock;

	protected DataProcessor _processor;

	protected long _ip = 0;

	protected long _port = 0;

	protected Node _node;

	private boolean _started = false;

	private boolean _valid = true;

	private Socket _activeConnection;
	
	private Process _debugProcessReference = null;

	/**
	 * Console constructor comment.
	 */
	public Console() {
		super();
		/* Don't block the VM from exiting */
		this.setName("Console"); /* 9707 *///$NON-NLS-1$
		this.setDaemon(true);
	}

	/**
	 * Construct a console that knows about the node that will try to
	 * communicate with it
	 */
	public Console(Node node) {
		this();
		this._node = node;
	}

	/**
	 * Console constructor comment.
	 * 
	 * @param name
	 *            java.lang.String
	 */
	public Console(String name) {
		super(name);
		/* Don't block the VM from exiting */
		this.setDaemon(true);
	}

	/**
	 * Console constructor comment.
	 * 
	 * @param group
	 *            java.lang.ThreadGroup
	 * @param name
	 *            java.lang.String
	 */
	public Console(ThreadGroup group, String name) {
		super(group, name);
		/* Don't block the VM from exiting */
		this.setDaemon(true);
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:44:15 PM)
	 * 
	 * @return long
	 */
	public long getIP() throws ConsoleNotStartedException {
		if (_ip == 0) {
			throw new ConsoleNotStartedException();
		}
		return _ip;
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:43:59 PM)
	 * 
	 * @return long
	 */
	public long getPort() throws ConsoleNotStartedException {
		if (_port == 0) {
			throw new ConsoleNotStartedException();
		}
		return _port;
	}

	/**
	 * Return the server socket used by the console thread
	 * 
	 * @return java.net.ServerSocket
	 * @throws ConsoleNotStartedException
	 */
	public ServerSocket getServerSocket() throws ConsoleNotStartedException {
		if (_sock == null) {
			throw new ConsoleNotStartedException(); /* 9707 */
		}

		return _sock;
	}

	private static int retrServSockStartFromUI() {
		int result = 0;
		
		// If we are running inside Eclipse, this bundle will be available... 
		Bundle uiBundle = Platform.getBundle("org.eclipse.tptp.platform.common.ui");
		if(uiBundle != null) {
			IPreferenceStore store = AdminPlugin.getDefault().getPreferenceStore();
			if(store != null) {
				
				boolean firewallCheckBoxEnabled = store.getBoolean(CommonConstants.DATAPOOL_PORT_ENABLED);
				boolean randomPortSelected = store.getBoolean(CommonConstants.DATAPOOL_PORT_RANDOM);
				String customPortValue = store.getString(CommonConstants.DATAPOOL_PORT_NUMBER);
				
				if(firewallCheckBoxEnabled && !randomPortSelected) {
					try {
						result = Integer.parseInt(customPortValue);
					} catch(NumberFormatException e) { /* ignore exception, result is 0 */}
				}
			}
		}
		
		return result;
	}
	
	/** For all network interfaces, return their local InetAddress */
	private static List/*<InetAddress>*/ getLocalHostAddresses()  {
		ArrayList result = new ArrayList();
		
		NetworkInterface ni = null;
		try {
			for (Enumeration ifaces = NetworkInterface.getNetworkInterfaces(); ifaces.hasMoreElements();) {
				ni = (NetworkInterface) ifaces.nextElement();
				InetAddress ia = null;
				
				for (Enumeration ipaddr = ni.getInetAddresses(); ipaddr.hasMoreElements();) {
					
					ia = (InetAddress) ipaddr.nextElement();
					
					result.add(ia);
				}
			}
		} catch(SocketException se) { /* Ignore this error */}
		
		return result;
	}
	
	private static InetAddress getLocalHostAddress() {
		List /*<InetAddress>*/ ial = getLocalHostAddresses();
		
		try {
			
			// This call returns only the first portion of your hostname, everything before the first dot
			String shortenedHostName = InetAddress.getLocalHost().getHostName().toLowerCase();
			
			// For each address in the list..
			for(Iterator i = ial.iterator(); i.hasNext();) {
				InetAddress ia = (InetAddress)i.next(); 

				String iaHN = ia.getHostName().toLowerCase();
				
				// .. If one of the addresses begins with the shortened host name, followed by the separator, we have a match
				if(iaHN.startsWith(shortenedHostName + ".")) {
					return ia;
				}
					
			}
			
		} catch (Exception e) {
			/* Any exceptions here will return null, reverting the calling function to its previous behaviour */
		}
		
		return null;
	}
	
	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:43:25 PM)
	 */
	public void run() {
		byte[] buffer = new byte[1024];
		int port = 0/* DATA_PORT_NUM_CLIENT */;
		boolean started = false;
		boolean complete;

		port = retrServSockStartFromUI();

		/* Start the server */
		while (!started) {
			try {
				started = true;
				_sock = new ServerSocket(port, 1);
			} catch (Exception e) {
				if(port != 0) {
					port++;
				}
				started = false;
			}
		}

		/* Store the address information of this console */
		_port = _sock.getLocalPort();

		if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
			HyadesExecMessageDebug.writeDebugMessage("Console ServerSocket listening on port " +debugFooter());
		}
		
		InetAddress localHost = null;
		
		InetAddress localHostAddress = getLocalHostAddress();
		if(localHostAddress != null) {
			localHost = localHostAddress;
		} else {

			try {
				localHost = InetAddress.getLocalHost();
			} catch (UnknownHostException e) {
				/* Shouldn't happen as we are looking up the localhost */
			}
		}

		if(localHost != null) {
			byte[] bytes = localHost.getAddress();
			_ip = (long) (bytes[3] << 24 & 0xff000000)
					| (long) (bytes[2] << 16 & 0x00ff0000)
					| (long) (bytes[1] << 8 & 0x0000ff00)
					| (long) (bytes[0] & 0x000000ff);
		}

		/**
		 * If server cannot reach to client do not start console server,
		 * basically the launch process command will use 0 as IP and port which
		 * indicates no console traffic will be brought back to the client side
		 * (and therefore no chance for the firewall to block it since the
		 * server reach command is indicating this is what will happen if
		 * attempted) -- if server can indeed reach to client then proceed
		 * typically (using console server)
		 */
		boolean serverCanReach = false;
		try {
			IFileManagerExtended fileManager = FileManagerFactory.getInstance()
					.create(this._node.getConnection());

			String hostname = null;
			try {
				
				if(localHostAddress != null) {
					hostname = localHostAddress.getHostName();
				} else {
					hostname = InetAddress.getLocalHost().getCanonicalHostName();
				}
				
			} catch (UnknownHostException e) { }
			
			if(hostname == null) {
				hostname = localHost.getHostAddress();
			}
			
			serverCanReach = fileManager.determineServerReach(hostname, (int) this._port);
		} catch (ServerNotAvailableException e) {
			e.printStackTrace();
		}

		/* Inform the start method we are up */
		synchronized (this) {
			_started = true;
			this.notify();
		}

		if (serverCanReach) {

			while (_valid) {
				InputStream is = null;
				_activeConnection = null;
				try {
					if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
						HyadesExecMessageDebug.writeDebugMessage("Console's ServerSocket accept() called. "+debugFooter());
					}
					
					_activeConnection = _sock.accept();
					if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
						HyadesExecMessageDebug.writeDebugMessage("Console accept() call returned a socket. "+debugFooter());
					}
					is = _activeConnection.getInputStream();
					_activeConnection.setSoTimeout(1000);
					complete = false;
				} catch (Exception e) {
					/* The server socket is toast, return */
					return;
				}
				while (!complete) {
					int length = 0;
					try {
						length = is.read(buffer);
					} catch (InterruptedIOException e) {
						/* Read timeout, don't want to wait too long */
						if (_processor != null) {
							_processor.waitingForData();
						}
					} catch (IOException e) {
						try {
							if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
								HyadesExecMessageDebug.writeDebugMessage("Console activeConnection closed. "+debugFooter());
							}							
							_activeConnection.close();
						} catch (IOException e1) {
						}
						complete = true;
					}

					/* Is the connection closed? */
					if (length < 0) {
						try {
							if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
								HyadesExecMessageDebug.writeDebugMessage("Console activeConnection closed. "+debugFooter());
							}							
							_activeConnection.close();
						} catch (IOException e) {
						}
						complete = true;
					} else if (length > 0) {
						if (_processor != null) {
							
							if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
								String str = new String(buffer, 0, length);
								String processId = "---";

								try {
									processId = (_debugProcessReference != null) ? _debugProcessReference.getProcessId() : "---";
								} catch (InactiveProcessException e) { }
								
								String processHash = _debugProcessReference != null ? (""+_debugProcessReference.hashCode()) : "---"; 
									
								HyadesExecMessageDebug.writeDebugMessage("Received console data: ["+str+"] (remote ip:"+_activeConnection.getInetAddress()+") (process id:"+processId+") "+debugFooter()+"\n");
							}							
							_processor.incommingData(buffer, length,
									_activeConnection.getInetAddress());
						}
					}

					/* Have we been asked to stop. */
					if (!_valid) {
						try {
							if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
								HyadesExecMessageDebug.writeDebugMessage("Console activeConnection closed. "+debugFooter());
							}
							_activeConnection.close();
						} catch (IOException e1) {
						}
						complete = true;
					}
				}
			}

		} else {

			/*
			 * If server cannot reach back, close new server socket and reset ip
			 * and port to zero, these values will be attached to the launch
			 * process command, if they are zero then server will not attempt to
			 * connect to console started in this object
			 */
			this._ip = 0;
			this._port = 0;
			this.close();

		}

	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 2:56:30 PM)
	 * 
	 * @param processor
	 *            org.eclipse.hyades.internal.execution.local.common.DataProcessor
	 */
	public void setDataProcessor(DataProcessor processor) {
		_processor = processor;

	}

	public DataProcessor getDataProcessor() {
		return _processor;
	}

	/**
	 * Insert the method's description here. Creation date: (2/19/01 5:39:25 PM)
	 */
	public void start() {
		/* We need to wait until the server is fully up */
		synchronized (this) {
			_valid = true;
			super.start();
			do {
				try {
					this.wait();
				} catch (InterruptedException e) {
				}
			} while (!_started);
		}
	}

	/**
	 * Insert the method's description here. Creation date: (5/28/01 8:01:54 PM)
	 * 
	 * @param data
	 *            java.lang.String
	 */
	public void write(String data) {
		if (_activeConnection != null) {
			try {
				if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && data != null) {
					HyadesExecMessageDebug.writeDebugMessage("Output to console: ["+data.trim()+"] ");
				}
				
				_activeConnection.getOutputStream().write(
						data.getBytes("UTF-8")); // Bug 73074//$NON-NLS-1$
			} catch (Exception e) {
			}
		}
	}
	
	/* Closes this console */
	public void close() {
		if (_sock != null) {
			try {
				if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_CONSOLE_DEBUG) {
					HyadesExecMessageDebug.writeDebugMessage("close() called on Console server " + debugFooter());
				}
				_sock.close();
			} catch (IOException e) {
			}
		}
		_valid = false;
	}

	/** For debug purposes only */
	public void setDebugProcess(Process debugProcess) {
		_debugProcessReference = debugProcess;
	}
	
	private String debugFooter() {
		String str = " port:["+_port+"] console-hash:["+this.hashCode()+"] thread-hash["+Thread.currentThread().hashCode()+"]";
		return str;
	}
}
