/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ControlMessage.java,v 1.17 2010/10/28 19:47:50 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.common;

import java.util.Iterator;

import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;


/**
 * Insert the type's description here.
 * Creation date: (6/2/00 3:06:05 PM)
 * @author: Administrator
 */
public class ControlMessage extends Message {

	CommandElement[] _entries=null;
	
	/* The length field */
	long _length;

	/* The authentication key */
	protected RAString _key=new RAString("");
/**
 * ControlMessage constructor comment.
 */
public ControlMessage() {
	super();
	_type=RA_CONTROL_MESSAGE;
}
/**
 * Insert the method's description here.
 * Creation date: (6/2/00 4:35:08 PM)
 * @param command org.eclipse.hyades.internal.execution.local.common.CommandElement
 */
public void appendCommand(CommandElement command)  {
	if(_entries==null) {
		_entries=new CommandElement[1];
		_entries[0]=command;
		return;
	}
	synchronized(_entries) {
//		if(this.getSize()+command.getSize() >= MAX_MESSAGE_LENGTH)
//			throw new MessageOverflowException();
			
		CommandElement[] _oldElements=_entries;
		_entries=new CommandElement[_oldElements.length+1];
		for(int i=0; i<_oldElements.length; i++) {
			_entries[i]=_oldElements[i];
		}
		_entries[_oldElements.length]=command;
	}
}
/**
 * Insert the method's description here.
 * Creation date: (6/8/00 2:35:25 PM)
 * @return CommandElement
 * @param offset int
 */
public CommandElement getCommand(int offset) {
	return _entries[offset];
}
/**
 * Insert the method's description here.
 * Creation date: (6/8/00 6:40:36 PM)
 * @return int
 */
public int getCommandCount() {
	return _entries.length;
}
/**
 * Insert the method's description here.
 * Creation date: (10/31/00 5:57:46 PM)
 */
public RAString getKey() {
	return _key;
}
/* 185463 begin */
/**
 * Returns length of message read from message itself.
 * Creation date: (09/27/00 5:57:46 PM)
 */
public long getLength() {
	return _length;
}
/* 185463 end */

/**
 * Insert the method's description here.
 * Creation date: (6/8/00 10:21:47 AM)
 * @return short
 */
public int getSize() {
	int size=super.getSize();

	/* The length field */
	size+=sizeofLong;
	/* Key size */
	size+=(int)_key.getSize();

	/* Area for the count */
	size+=sizeofLong;

	/* The commands */

	/*synchronized(_entries) {*/
		for(int i=0; i<_entries.length; i++)
			size+=_entries[i].getSize();
	/*}*/
	return  size;
}

/**
 * readFromBuffer - checks if the full message is contained in the buffer
 * @param buffer data buffer to read from
 * @param offset offset with the data buffer to start reading from
 * @param length length of the data in the buffer
 */
public int readFromBuffer(byte[] buffer, int offset, int length) {
	int current=super.readFromBuffer(buffer, offset);

	long msgLength=Message.readRALongFromBuffer(buffer, current);
	
	/* First check if the message length field is contained in the data buffer so it is assured to be a valid length value 
	 * and then check if the full message is contained in the data remaining in the buffer */
	if(current+sizeofLong > offset+length  || msgLength > length) {
		throw new ArrayIndexOutOfBoundsException(LocalResourceBundle.ControlMessage_NOT_ENOUGH_BYTES_);
	}
	
	return readFromBuffer(buffer, offset);
}

private static boolean isDetailedAgentInfoCommand(CommandElement ce) {

	if(ce instanceof AgentDetailsCommand) {
		return true;
	}

	else if(ce instanceof AgentInactiveCommand) {
		return true;
	}
	
	else if(ce instanceof AgentActiveCommand) {
		return true;
	}
	
	return false;

}

private static boolean isSimpleAgentInfoCommand(CommandElement ce) {
	if(ce instanceof StopMonitorCommand) {
		return true;
	}

	else if(ce instanceof DetachFromAgentCommand) {
		return true;
	}
	
	else if(ce instanceof AttachToAgentCommand) {
		return true;
	}
	
	else if(ce instanceof RegisterAgentInterestCommand) {
		return true;
	}
	
	else if(ce instanceof AgentDetachedCommand) {
		return true;
	}
	
	else if(ce instanceof AgentAttachedCommand) {
		return true;
	}
	
	else if(ce instanceof AgentQueryStateCommand) {
		return true;		
	}

	else if(ce instanceof QueryAgentDetailsCommand) {
		return true;
	}
	

	return false;
}

public static String debugToStringCommandElement(CommandElement ce) {
	boolean commandHandled = false;
	String result = "";
	
	String simpleName;
	
	// Java 1.4 compliant version of getClass
	try {
		simpleName = ce.getClass().getName();
		simpleName = simpleName.substring(simpleName.lastIndexOf(".")+1);
	} catch(Exception e) {
		simpleName = ce.getClass().getName();
	}
	
	result += simpleName;
	
	if(isSimpleAgentInfoCommand(ce)) {
		SimpleAgentInfoCommand c = (SimpleAgentInfoCommand)ce;
		result += " name:["+c.getAgentName()+"] pid:["+c.getProcessId()+"]";

	
	} else 	if(isDetailedAgentInfoCommand(ce)) {
		DetailedAgentInfoCommand c = (DetailedAgentInfoCommand)ce;
		result += " agent name:("+c.getAgentName()+") ";
		result += "agent type:("+c.getAgentType()+") ";
		result += "agent uiid:("+c.getAgentUUID()+") ";
		result += "proc uuid:("+c.getProcessUUID()+")";

	} else if(ce instanceof AuthenticateCommand) {
		/** Name and password not available as getters of command */		

	} else if(ce instanceof GetPropertyListCommand) {
		GetPropertyListCommand c = (GetPropertyListCommand)ce;

		result += " name:("+c.getName()+") ";
		result += "type:("+c.getType()+") ";
		result += "agent uiid:("+c.getAgentUUID()+") ";
		
	} else if(ce instanceof BinaryCustomCommand) {
		BinaryCustomCommand bcc = (BinaryCustomCommand)ce;
		result += " data:("+bcc.getData()+") name:("+bcc.getAgentName()+") pid:("+bcc.getProcessId()+")";		
	}
	
	else if(ce instanceof KillProcessCommand) {
		KillProcessCommand c = (KillProcessCommand)ce;
		result += " pid: "+c.getProcessId();
	}
	
	else if(ce instanceof CustomCommand) {
		CustomCommand cc = (CustomCommand)ce;
		result += " data:("+cc.getData()+") name:("+cc.getAgentName()+") pid:("+cc.getProcessId()+")";
	}
	
	else if(ce instanceof SetNVPairCommand) {
		SetNVPairCommand c = (SetNVPairCommand)ce;

		result += " type:("+c.getType()+") ";
		result += "name:("+c.getName()+") ";
		result += "value:("+c.getValue()+")";

	}
	
	
	else if(ce instanceof StartMonitoringLocalAgentCommand) {
		StartMonitoringLocalAgentCommand c = (StartMonitoringLocalAgentCommand)ce;
		result += " file:["+c.getFile()+"] name:["+c.getAgentName()+"] pid:["+c.getProcessId()+"]";
		
	}
	
	else if(ce instanceof StartMonitoringRemoteAgentCommand) {
		StartMonitoringRemoteAgentCommand c = (StartMonitoringRemoteAgentCommand)ce;
		result += " ip:("+c.getIP()+") port:("+c.getPort()+") name:["+c.getAgentName()+"] pid:["+c.getProcessId()+"]";
	}
	else if(ce instanceof QueryAgentListCommand) {
		/** No state */
	}
		
	else if(ce instanceof QueryProcessListCommand ) {
		/** No state */
	}
	
	else if(ce instanceof LaunchProcessCommand) {
		LaunchProcessCommand c = (LaunchProcessCommand)ce;
		
		result +=  " exe:("+c.getExe()+") ";
		
		result += "args:("+c.getArgs()+") ";
		
		result += "location:("+c.getLocation()+") ";
		
		result += "consoleIP:("+c.getConsoleIP()+") ";
		result += "consolePort:("+c.getConsolePort()+") ";
		
		result += "env:(";
		for(Iterator i = c.getEnvironment().iterator(); i.hasNext();) {
			RAString s = (RAString) i.next();
			
			result += "["+s+"]";
		}
		result += ")";

		result += "agents:(";
		for(Iterator i = c.getAgents().iterator(); i.hasNext();) {
			RAString s = (RAString) i.next();
			
			result += "["+s.getData()+"]";
		}

	}
	
	else if(ce instanceof ServerSecurityInfoCommand) {
		ServerSecurityInfoCommand c = (ServerSecurityInfoCommand)ce;
		
		result += " isPwdProt:("+c.isPasswordProtected()+") isClientAuthReq:("+c.isClientAuthenticationRequired()+") Secure port:("+c.getSecurePort()+")";
		
	}
	
	else if(ce instanceof AuthenticationSuccessfulCommand) {
		AuthenticationSuccessfulCommand c = (AuthenticationSuccessfulCommand)ce;
		result += " size:("+c.getSize()+")";
	}
	
	else if(ce instanceof AuthenticationFailedCommand) {
		AuthenticationFailedCommand c = (AuthenticationFailedCommand)ce;
		result += " ticket:("+c.getTicket()+")";
	}
		
	else if(ce instanceof ResourceLocation) {
		ResourceLocation c = (ResourceLocation)ce;
		result += " jobkey:("+c.getJobKey()+") port:("+c.getPort()+") size:("+c.getSize()+")";
	}

	else if(ce instanceof MonitorPeerRequestPortCommand) {
		// Peer Monitoring
	}
	
	else if(ce instanceof AgentRequestMonitorPortCommand) {
		// Peer Monitoring
	}
	
	else if(ce instanceof PeerUnreachableCommand) {
		// Peer Monitoring
	}

	else if(ce instanceof MonitorPeerRequestCommand) {
		// Peer Monitoring
	}
	
	else if(ce instanceof AgentRequestMonitorCommand) {
		// Peer Monitoring
	}
	
	else if(ce instanceof AgentConfigurationCommand) {
		AgentConfigurationCommand c = (AgentConfigurationCommand)ce;
		result += " ";
		result += "process-id:("+c.getProcessId()+") ";
		result += "process-uuid:("+c.getProcessUUID()+") ";
		result += "agent-name:("+c.getAgentName()+") ";
		result += "agent-uuid:("+c.getAgentUUID()+") ";
		result += "agent-type:("+c.getAgentType()+") ";
		result += "node-uuid:("+c.getNodeUUID()+") ";
		
		result += "entries:(";
		for(int i = 0; i < c.getConfigurations().length; i++) {
			AgentConfigurationEntry entry = (AgentConfigurationEntry)c.getConfigurations()[i];
			result += " ["+entry.getType()+"|"+entry.getName()+"|"+entry.getValue()+"]";
		}
		result += ")";

	}
	
	else if(ce instanceof AgentScopingInformationCommand) {
		AgentScopingInformationCommand c = (AgentScopingInformationCommand)ce;

		result += " ";
		result += "process-id:("+c.getProcessId()+") ";
		result += "msg-proc-id:("+c.getMessageProcessId()+") ";
		result += "proc-uuid:("+c.getProcessUUID()+") ";
		result += "agent-name:("+c.getAgentName()+") ";
		result += "agent-uuid:("+c.getAgentUUID()+") ";
		result += "agent-type:("+c.getAgentType()+") ";
		result += "node-uuid:("+c.getNodeUUID()+")";
		
	}
	
	else if(ce instanceof PropertyListCommand) {
		PropertyListCommand c = (PropertyListCommand)ce;

		SetNVPairCommand[] r = c.getPropertyListValues();
		
		result += " entries(";
		for(int i = 0; i < r.length; i++) {
			SetNVPairCommand rc =  r[i];
			result += " ["+rc.getType();
			result += " "+rc.getName();
			result += " "+rc.getValue()+"]";
		}
		result += ")";
		
	}
	
	else if(ce instanceof ProcessExitedCommand) {
		ProcessExitedCommand c = (ProcessExitedCommand)ce;
		result += " pid:("+c.getProcessId()+")";
		
	}
	else if(ce instanceof ErrorCommand) {
		ErrorCommand e = (ErrorCommand)ce;

		result += " severity:("+ e.getSeverity()+")";
		
		result += " errorId:("+ e.getErrorId()+")";
		
		result += " errorString:("+ e.getErrorString()+")";		

	}
	
	else if(ce instanceof ActiveAgentListCommand) {
		ActiveAgentListCommand c = (ActiveAgentListCommand)ce;
		result += " agent list:(";
		for(int x = 0; x < c.getAgents().length; x++) {
			result += "["+c.getAgents()[x]+"]";
		}
		result += ")";
	}
	
	else if(ce instanceof RegisteredProcessListCommand) {
		RegisteredProcessListCommand c = (RegisteredProcessListCommand)ce;
		result += " proc list:(";
		for(int x = 0; x < c.getProcessList().length; x++) {
			result += c.getProcessList()[x];
			if(x +1 < c.getProcessList().length) {
				result += ", ";
			}
		}
		result += ")";
	}
	
	else if(ce instanceof ProcessLaunchedCommand) {
		ProcessLaunchedCommand c = (ProcessLaunchedCommand)ce;
		result += " exe:("+c.getExe()+") ";
		result += "args:("+c.getArgs()+") ";
		
		result += "env:(";
		for(int x = 0; x < c.getEnvironment().length; x++) {
			result += "["+c.getEnvironment()[x]+"] ";
		}
		result +=")";
		
		
	}
	
	else if(ce instanceof ManageFileCommand) {
		ManageFileCommand c = (ManageFileCommand)ce;
		
		result += " operation:("+c.getOperation()+") filename:("+c.getFilename()+")";
	}
	
	else if(ce instanceof GetPropertyListCommand) {
		GetPropertyListCommand c = (GetPropertyListCommand)ce;
		
		result += " name("+c.getName()+") ";
		result += "type("+c.getType()+") ";
		result += "agentUUID("+c.getAgentUUID()+")";

	} else {
		
		result += " -- Command not recognized.";
		
		commandHandled = true;
	}
		
	return result;
		
}

/**
 * readFromBuffer method comment.
 */
public int readFromBuffer(byte[] buffer, int offset) {
	int current=super.readFromBuffer(buffer, offset);

	_length=Message.readRALongFromBuffer(buffer, current);
	if(_length > buffer.length) {
		throw new ArrayIndexOutOfBoundsException(LocalResourceBundle.ControlMessage_NOT_ENOUGH_BYTES_);
	}

	current+=sizeofLong;
	/* authentication key */
	current=readRAStringFromBuffer(buffer, current, _key);

	/* How many commands are in this message */
	long count=Message.readRALongFromBuffer(buffer, current);
	current+=sizeofLong;
	
	_entries=new CommandElement[(int)count];

	/* Read in each command */
	for(int i=0; i<count; i++) {
		long tag=Message.readRALongFromBuffer(buffer, current);
		
		current+=sizeofLong;
		CommandElement command=null;
		switch((int)tag) {
		case (int)RA_AUTHENTICATE: // 0x01
			command = new AuthenticateCommand();
			current = ((AuthenticateCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AUTHENTICATION_FAILED: // 0x02
			command = new AuthenticationFailedCommand();
			current = ((AuthenticationFailedCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AUTHENTICATION_SUCCESSFUL: // 0x03
			command = new AuthenticationSuccessfulCommand();
			current = ((AuthenticationSuccessfulCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_SERVER_SECURITY_REQUIREMENTS: // 0x04
			command = new ServerSecurityInfoCommand();
			current = ((ServerSecurityInfoCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_LAUNCH_PROCESS: // 0x10
			command = new LaunchProcessCommand();
			current = ((LaunchProcessCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_QUERY_PROCESS_LIST: // 0x11
			command = new QueryProcessListCommand();
			current = ((QueryProcessListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_QUERY_AGENT_LIST: // 0x12
			command = new QueryAgentListCommand();
			current = ((QueryAgentListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_REGISTER_AGENT_NOTIFICATION: // 0x13
			command = new RegisterAgentInterestCommand();
			current = ((RegisterAgentInterestCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_ATTACH_TO_AGENT: // 0x14
			command = new AttachToAgentCommand();
			current = ((AttachToAgentCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_DETACH_FROM_AGENT: // 0x15
			command = new DetachFromAgentCommand();
			current = ((DetachFromAgentCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_START_MONITORING_AGENT_REMOTE: // 0x16
			command = new StartMonitoringRemoteAgentCommand();
			current = ((StartMonitoringRemoteAgentCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_START_MONITORING_AGENT_LOCAL: // 0x17
			command = new StartMonitoringLocalAgentCommand();
			current = ((StartMonitoringLocalAgentCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_STOP_MONITORING_AGENT: // 0x18
			command = new StopMonitorCommand();
			current = ((StopMonitorCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_SET_NAME_VALUE_PAIR: // 0x19
			command = new SetNVPairCommand();
			current = ((SetNVPairCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_CUSTOM_COMMAND: // 0x1A
			command = new CustomCommand();
			current = ((CustomCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_KILL_PROCESS: // 0x1B
			command = new KillProcessCommand();
			current = ((KillProcessCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_QUERY_AGENT_DETAILS: // 0x1C
			command = new QueryAgentDetailsCommand();
			current = ((QueryAgentDetailsCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_BINARY_CUSTOM_COMMAND: // 0x1D
            command = new BinaryCustomCommand();
            current = ((BinaryCustomCommand)command).readFromBuffer(buffer, current);
            break;
		case (int)RA_GET_PROPERTY_LIST: // 0x1E
			command = new GetPropertyListCommand();
			current = ((GetPropertyListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_MANAGE_FILE: // 0x1F
			command = new ManageFileCommand() ;
			current = ((ManageFileCommand)command).readFromBuffer(buffer,current) ;
			break;
		case (int)RA_PROCESS_LAUNCHED: // 0x20
			command = new ProcessLaunchedCommand();
			current = ((ProcessLaunchedCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_PROCESS_LIST: // 0x21
			command = new RegisteredProcessListCommand();
			current = ((RegisteredProcessListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_LIST: // 0x22
			command = new ActiveAgentListCommand();
			current = ((ActiveAgentListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_ACTIVE: // 0x23
			command = new AgentActiveCommand();
			current = ((AgentActiveCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_INACTIVE: // 0x24
			command = new AgentInactiveCommand();
			current = ((AgentInactiveCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_ERROR_STRING: // 0x25
			command = new ErrorCommand();
			current = ((ErrorCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_ATTACH_SUCCESSFUL: // 0x26
			break; // Not used
		case (int)RA_ATTACH_FAILED: // 0x27
			break; // Not used
		case (int)RA_AGENT_DETAILS: // 0x28
			command = new AgentDetailsCommand();
			current = ((AgentDetailsCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_PROCESS_EXITED: // 0x29
			command = new ProcessExitedCommand();
			current = ((ProcessExitedCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_PROPERTY_LIST: // 0x2A
			command = new PropertyListCommand();
			current = ((PropertyListCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_QUERY_STATE: // 0x2B
			command = new AgentQueryStateCommand();
			current = ((AgentQueryStateCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_ATTACHED: // 0x2C
			command = new AgentAttachedCommand();
			current = ((AgentAttachedCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_DETACHED: // 0x2D
			command = new AgentDetachedCommand();
			current = ((AgentDetachedCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_LOCAL_AGENT_ACTIVE: // 0x30
			break; // Not used
		case (int)RA_AGENT_SCOPING_INFORMATION: // 0x31
			command = new AgentScopingInformationCommand();
			current = ((AgentScopingInformationCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_CONFIGURATION: // 0x32
			command = new AgentConfigurationCommand();
			current = ((AgentConfigurationCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_AGENT_CONTROLLER_AVAILABLE: // 0x50
			break; // Not used
		case (int)RA_AGENT_CONTROLLER_UNAVAILABLE: // 0x51
			break; // Not used
		case (int)RA_AGENT_REQUEST_MONITOR: // 0x61
			command = new AgentRequestMonitorCommand();
			current = ((AgentRequestMonitorCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_CONTROLLER_REQUEST_MONITOR: // 0x62
			command = new MonitorPeerRequestCommand();
			current = ((MonitorPeerRequestCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_PEER_UNREACHABLE: // 0x63
			command = new PeerUnreachableCommand();
			current = ((PeerUnreachableCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_CONTROLLER_MONITOR_PEER: // 0x64
			break; // Not used
		case (int)RA_AGENT_REQUEST_MONITOR_PORT: // 0x65
			command = new AgentRequestMonitorPortCommand();
			current = ((AgentRequestMonitorPortCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_CONTROLLER_REQUEST_MONITOR_PORT: // 0x66
			command = new MonitorPeerRequestPortCommand();
			current = ((MonitorPeerRequestPortCommand)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_RESOURCE_LOCATION: // 0x70
			command = new ResourceLocation();
			current = ((ResourceLocation)command).readFromBuffer(buffer, current);
			break;
		case (int)RA_CONSOLE_INFO: // 0x80
			break; // Not serializable
		}
		/* Store the command in the array */
		_entries[i]=command;
		
	}
	return current;
}
/**
 * Insert the method's description here.
 * Creation date: (10/31/00 5:59:07 PM)
 * @param key java.lang.String
 */
public void setKey(RAString key) {
	_key=key;
}
/**
 * writeToBuffer method comment.
 */
public int writeToBuffer(byte[] buffer, int offset) {
	int current=super.writeToBuffer(buffer, offset);

	/* Insert the length */
	_length=getSize();
	current=Message.writeRALongToBuffer(buffer, current, _length);
	/* Insert the authentication key */
	current=writeRAStringToBuffer(buffer, current, _key);

	/* place the command count in the buffer */
	current=Message.writeRALongToBuffer(buffer, current, _entries.length);
	
	for(int i=0; i<_entries.length; i++) {
		current=_entries[i].writeToBuffer(buffer, current);
		
	}
	return current;
}
}
