/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MultiplexedDataServer.java,v 1.11 2008/03/20 18:49:50 dmorris Exp $
 *
 * Contributors: 
 * IBM - Initial API and implementation
 ***********************************************************************/

package org.eclipse.hyades.internal.execution.local.common;

import java.net.InetAddress;

import org.eclipse.hyades.execution.local.internal.resources.LocalResourceBundle;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;

/**
 * Please see TCPDataServer for implmentation details.
 */
public class MultiplexedDataServer implements AgentListener, Constants {
	private Agent _agent;
	private DataProcessor _processor;
	private long magicNumber = 0xFEEDC0DE;
	private long terminationCode = 0xDEADC0DE;
	private InetAddress inetaddr = null;

	/* Different types of data */
	public static final byte BINARY_DATA			= 0;
	public static final byte UTF8_STRING_DATA		= 1;
	public static final byte UNICODE_STRING_DATA	= 2;

	public void startServer(DataProcessor processor) throws Exception {
		_processor = processor;
	}

	public boolean isProcessing() {
		return true; // always processing
	}

	public void stopServer() {
	}

	public void resumeServer() {
	}

	public void resumeServer(DataProcessor processor) {
	}

	public void shutdownServer() {
	    this._processor = null;
	}

	public void incommingData(byte[] b, InetAddress peer) {
		_processor.incommingData(b, b.length, peer);
	}

	public void incommingData(char[] c, InetAddress peer) {
		_processor.incommingData(c, c.length, peer);
	}

	public void agentActive(Agent agent) {
		try {
			inetaddr = agent.getProcess().getNode().getInetAddress();
		} catch (InactiveProcessException e) {
		}
	}

	public void agentInactive(Agent agent) {
		this._agent = agent;
	}

	public void error(Agent agent, String errorId, String errorMessage) {
	}

	public void handleCommand(Agent agent, CommandElement command) {
		switch((int)command.getTag()) {
		case (int)Constants.RA_BINARY_CUSTOM_COMMAND:
			BinaryCustomCommand binaryCustomCommand = (BinaryCustomCommand)command;
			int binaryDataLength;
			byte[] binaryData;

			// The byte array data and length
			binaryData = binaryCustomCommand.getDataBinary();
			binaryDataLength = binaryData.length;
			
			if (binaryDataLength >= sizeofLong) {
				// Read the magic number
				long code = Message.readRALongFromBuffer(binaryData, 0);

				if (code == magicNumber) {
					// Read the actual multiplexed data
					byte b[] = new byte[binaryDataLength - 4];
					System.arraycopy(binaryData, 4, b, 0, binaryDataLength - 4);
					processData(b, inetaddr);
				} else if (code == terminationCode) {
					/* Notify that the flusher is exiting */
					if (_processor instanceof DataServerListener) {
						((DataServerListener) _processor).dataServerExited();
					}
					if (this._agent != null && this._agent.getProcess() != null && this._agent.getProcess() instanceof ProcessImpl){
						((ProcessImpl) this._agent.getProcess()).removeAgent(this._agent);
					}
				}
			}
			break;
		}
	}

	private void processData(byte[] data, InetAddress addr) {
		if(data.length > Constants.MESSAGE_HEADER_LENGTH) {
			if(isValidHeader(data)) {
				int length = (int)getMessageLength(data);
				byte type = getMessageType(data);

				if(length + Constants.MESSAGE_HEADER_LENGTH <= data.length) {
					if(type == BINARY_DATA || type == UTF8_STRING_DATA) {
						byte[] forwardBuffer = new byte[length];
						System.arraycopy(data, Constants.MESSAGE_HEADER_LENGTH, forwardBuffer, 0, length);
						_processor.incommingData(forwardBuffer, length, addr);
					}
					else if(type == UNICODE_STRING_DATA) { // double-byte
						int strlen = length / 2;
						char[] forwardBuffer = new char[strlen];
						for(int i = 0; i < strlen; i++) {
							forwardBuffer[i] = (char)((char)data[Constants.MESSAGE_HEADER_LENGTH + 2*i] | (char)(data[Constants.MESSAGE_HEADER_LENGTH + 2*i + 1] << 8));
						}
						_processor.incommingData(forwardBuffer, strlen, addr);
					}
					else {
						System.err.println(LocalResourceBundle.MultiplexedDataServer_CORRUPTED_DATA_);
					}
				}
				else {
					System.err.println(LocalResourceBundle.MultiplexedDataServer_NO_ENOUGH_BYTES_);
				}
			}
			else {
				System.err.println(LocalResourceBundle.MultiplexedDataServer_NO_VALID_HEADER_);
			}
		}
		else {
			System.err.println(LocalResourceBundle.MultiplexedDataServer_NO_ENOUGH_HEADER_BYTES_);
		}
	}

	/**
	 * Get the length of the current message
	 */
	private long getMessageLength(byte[] b) {
		return Message.readRALongFromBuffer(b, 5);
	}

	/**
	 * Get the message type.  There are currently three types
	    of messages.  BINARY_DATA, UTF8_STRING_DATA, UNICODE_STRING_DATA.
	 */
	private byte getMessageType(byte[] b) {
		return b[9];

	}

	/**
	 * Check the message magic number.  If the magic number is incorrect we need
	    to go into recovery mode
	 */
	private boolean isValidHeader(byte[] b) {
		boolean valid = false;

		if((b[0] == Constants.RA_MAGIC_0) &&
			(b[1] == Constants.RA_MAGIC_1) &&
			(b[2] == Constants.RA_MAGIC_2) &&
			(b[3] == Constants.RA_MAGIC_3)) {
			valid = true;
		}

		return valid;
	}
}
