/**********************************************************************
 * Copyright (c) 2005, 2011 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NodeImpl.java,v 1.25 2011/01/21 19:57:25 mreid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.execution.security.ISecureClientParameters;
import org.eclipse.hyades.internal.execution.local.common.ActiveAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.AgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.GetPropertyListCommand;
import org.eclipse.hyades.internal.execution.local.common.KillProcessCommand;
import org.eclipse.hyades.internal.execution.local.common.MonitorPeerRequestCommand;
import org.eclipse.hyades.internal.execution.local.common.PropertyListCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryProcessListCommand;
import org.eclipse.hyades.internal.execution.local.common.RegisteredProcessListCommand;
import org.eclipse.hyades.internal.execution.local.common.SetNVPairCommand;
import org.eclipse.hyades.internal.execution.local.common.SimpleProcessCommand;
import org.eclipse.hyades.internal.execution.security.LoginFailedException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;
import org.eclipse.hyades.internal.execution.security.UntrustedAgentControllerException;
import org.eclipse.hyades.internal.execution.security.User;
import org.eclipse.tptp.platform.iac.administrator.internal.startstop.AutoStartStop;
import org.osgi.framework.Bundle;

public class NodeImpl implements Node, Constants {
	private static int DEFAULT_TIMEOUT = 15000; // 15 sec

	private String _name;

	private InetAddress[] _addr;

	private Connection _connection;

	private Application _application;

	private Hashtable _processList = new Hashtable(256);

	private SetNVPairCommand[] props = null;

	// Local listeners and handlers
	private InnerProcessListener _processListener = new InnerProcessListener();

	private InnerConnectionListener _connectionListener = new InnerConnectionListener();

	private InnerCommandHandler _commandHandler = new InnerCommandHandler();

	// Locks
	private Object _processListLock = new Object();

	private Object _agentListLock = new Object();

	private Object _agentDetailsLock = new Object();

	private Object _propertyListLock = new Object();

	//
	// Security related stuff
	//
	private boolean _isAuthenticated = false;
	private int _authenticationTimeout = 10000; // 5 sec authentication timeout 
	private Object _authenticationLock = new Object();
	private User _user = null;
	private ISecureClientParameters _securityParms = null;
	
	private static final String PLATFORM = System.getProperty("os.name").toLowerCase();//$NON-NLS-1$
	private static final String NEWLINE = System.getProperty("line.separator");//$NON-NLS-1$

	public NodeImpl(String name, InetAddress addr) {
		super();
		_name = name;

		try {
			_addr = InetAddress.getAllByName(addr.getHostName());
		} catch (UnknownHostException e) {
		}
	}

	public NodeImpl(String name, InetAddress[] addr) {
		super();
		_name = name;
		_addr = addr;
	}

	/**
	 * @see Node#killProcess(Process)
	 */
	public void killProcess(Process process) throws InactiveProcessException, NotConnectedException {
		/* Determine if we have a connection */
		if (_connection == null) {
			throw new NotConnectedException();
		}

		/* Determine if the process is active */
		if (!process.isActive()) {
			throw new InactiveProcessException();
		}

		/* Kill the process */
		ControlMessage message = new ControlMessage();
		KillProcessCommand command = new KillProcessCommand();
		command.setProcessId(Long.parseLong(process.getProcessId()));
		message.appendCommand(command);
		try {
			_connection.sendMessage(message, _commandHandler);
		} catch (IOException e) {
			// TODO Auto-generated catch block
		}
	}

	/**
	 * @see Node#shutdown(long)
	 */
	public void shutdown(long delay) throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * @see Node#reboot(long)
	 */
	public void reboot(long delay) throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * @see Node#listMonitors()
	 */
	public Enumeration listMonitors() throws NotConnectedException {
		throw new NotImplementedException();
	}

	/**
	 * List the processes running in the node
	 * 
	 * This method has side effects of setting the _processList member 
	 * variable, populating the agents within each process in _processList 
	 * and retrieving the details of each agent in each process.
	 */
	public Enumeration listProcesses() throws NotConnectedException {
		ControlMessage message;

		if (_connection == null) {
			throw new NotConnectedException();
		} else {
			_connection.addConnectionListener(_connectionListener); // to avoid
			// deadlock

			// Query the process list
			message = new ControlMessage();
			QueryProcessListCommand qplCommand = new QueryProcessListCommand();
			message.appendCommand(qplCommand);
			synchronized (_processListLock) {
				try {
					_connection.sendMessage(message, _commandHandler);
					_processListLock.wait(DEFAULT_TIMEOUT);
				} catch (Exception e) {
					e.printStackTrace();
					return null;
				}
			}

			// Populate the process with all agents
			Enumeration ePIDs = _processList.keys();
			while (ePIDs.hasMoreElements()) {
				Long pid = (Long) (ePIDs.nextElement());
				Process p = (ProcessImpl) _processList.get(pid);

				// Query the agent list
				message = new ControlMessage();
				QueryAgentListCommand qalCommand = new QueryAgentListCommand();
				qalCommand.setProcessId(pid.longValue());
				message.appendCommand(qalCommand);
				synchronized (_agentListLock) {
					try {
						_connection.sendMessage(message, _commandHandler);
						_agentListLock.wait(DEFAULT_TIMEOUT);
					} catch (Exception e) {
						e.printStackTrace();
						return null;
					}
				}

				// Populate the agent details
				Enumeration eAgents = p.listAgents();
				while (eAgents.hasMoreElements()) {
					Agent ag = (Agent) eAgents.nextElement();
					String agentName = ag.getName();

					message = new ControlMessage();
					QueryAgentDetailsCommand qadCommand = new QueryAgentDetailsCommand();
					qadCommand.setProcessId(pid.longValue());
					qadCommand.setAgentName(agentName);
					message.appendCommand(qadCommand);
					synchronized (_agentDetailsLock) {
						try {
							_connection.sendMessage(message, _commandHandler);
							_agentDetailsLock.wait(DEFAULT_TIMEOUT);
						} catch (Exception e) {
							e.printStackTrace();
							return null;
						}
					}
				}
			}
		}

		return _processList.elements();
	}

	/**
	 * @see Node#getProcess()
	 */
	public synchronized Process getProcess(String processId) {
		return (Process) _processList.get(new Long(processId));
	}

	/**
	 * @see Node#getInetAddress()
	 */
	public InetAddress getInetAddress() {
		if (_addr != null) {
			return _addr[0];
		}

		return null;
	}

	/**
	 * @see Node#getAllInetAddresses()
	 */
	public InetAddress[] getAllInetAddresses() {
		return _addr;
	}

	/**
	 * @see Node#getName()
	 */
	public String getName() {
		return _name;
	}

	/**
	 * @see Node#isConnected()
	 */
	public boolean isConnected() {
		if (_connection != null) {
			return _connection.isActive();
		}
		return false;
	}
	
	/**
	 * @see Node#connect()
	 */
	public synchronized Connection connect(int port) throws AgentControllerUnavailableException, SecureConnectionRequiredException, UntrustedAgentControllerException, LoginFailedException  {
		Connection testConnection = null;
		
		/* NM: Bugzilla 173330: Start IAC */
		
		/* First check to see whether the iac.administrator plugin is available */
		Bundle iacBundle = Platform.getBundle("org.eclipse.tptp.platform.iac.administrator");//$NON-NLS-1$
		
		if ((iacBundle != null) && (isLocal()))			
			AutoStartStop.startIAC(); 
		
		/* End of bugzilla 173330 */
		
		if ((_connection == null || !_connection.isActive() || (port != _connection.getPort()))) { // Bug
			// 54320
			try {

				/*
				 * If the keystore is specified then we will try a secure
				 * connection
				 */
				if (_securityParms == null) {
					testConnection = ConnectionFactory.getConnection(this);
					testConnection.connect(this, port);
				} else {
					/* Look through our extension points for a security provider */
					testConnection = ConnectionFactory.getSecuredConnection(this);
					if (testConnection != null) {
						testConnection.connect(this, port);
					}
				}

				/*
				 * If there is no exception after calling connect() that means
				 * the connection with the specified port is valid. Since we
				 * assume a 1-to-1 association between Node and Connection, that
				 * means any previous connectio is invalid (otherwise the
				 * attempted connect() will fail).
				 */
				_connection = testConnection;

			} catch (IOException e) {
				_connection = null;
				throw new AgentControllerUnavailableException();
			}
		}

		return _connection;

	}

	/**
	 * @see Node#getConnection()
	 */
	public Connection getConnection() {
		if(_connection == null) {
			return null;
		}

		return _connection;
	}

	public ProcessListener getProcessListener() {
		return _processListener;
	}

	/**
	 * @see Node#setUser()
	 */
	public void setUser(User user) {
		_user = user;
	}

	/**
	 * @see Node#getUser()
	 */
	public User getUser() {
		return _user;
	}

	public void setApplication(Application app) {
		_application = app;
	}

	public Application getApplication() {
		return _application;
	}

	public void setSecurityParameters(ISecureClientParameters manager) {
		_securityParms = manager;
	}

	public ISecureClientParameters getSecurityParameters() {
		return _securityParms;
	}

	/**
	 * Bug 53938
	 * 
	 * Query the options specified in the Agent Controller's configuration file.
	 * 
	 * @param name
	 *            The name of the option
	 * @param type
	 *            The type of the option
	 * @return An array of SetNVPairCommand objects
	 */
	public SetNVPairCommand[] getPropertyValues(String name, String type) throws NotConnectedException {
		/* Determine if we have a connection */
		if (_connection == null) {
			throw new NotConnectedException();
		}

		/* Request the process list from the client engine */
		ControlMessage message = new ControlMessage();
		GetPropertyListCommand command = new GetPropertyListCommand();
		command.setName(name);
		command.setType(type);
		message.appendCommand(command);

		synchronized (_propertyListLock) {
			try {
				_connection.sendMessage(message, _commandHandler);
				_propertyListLock.wait(DEFAULT_TIMEOUT);
			} catch (Exception e) {
				return null;
			}
		}

		return props;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.local.control.Node
	 *      #waitForProcess(java.lang.String, int)
	 * 
	 * Waits for a particular process to be present or the timeout value to be
	 * reached; after the process list returns (the internal process list in
	 * this node instance is updated) the process is retrieved via the get
	 * process method. The timeout value is approximate since there is some time
	 * needed to execute the various operations within the loop and each wait is
	 * done timeout divided by 2000 times. For example, a 60000 (60 second)
	 * timeout would cause the loop to execute 30 times and wait for 2 seconds
	 * at the end of each process list query.
	 * 
	 * The listProcesses(...) method default wait time itself is up to 7 seconds
	 * to receive the process list.
	 * 
	 */
	public synchronized Process waitForProcess(String processIdentity, int timeout) throws NotConnectedException,
			InactiveProcessException {

		// Change this from a polling model into a notify model
		try {
			for (int i = 0, j = timeout / 2000; i < j; i++) {
				Enumeration processes = this.listProcesses();
				if (processes != null) {
					for (; processes.hasMoreElements();) {
						Process process = (Process) processes.nextElement();
						if (process.getProcessId().equals(processIdentity)) {
							return process;
						}
					}
				}
				this.wait(2000);
			}
		} catch (InterruptedException e) {
			// Do not need to handle this
		}

		return null; // no process was found in the allowed time
	}

	/**
	 * List the processes running in the node
	 * 
	 * This method has side effects of setting the _processList member 
	 * variable and populating the agents within each process in _processList 
	 * Unlike the listProcesses() method, it does *not* retrieve the details
	 * for each agent.
	 */
	public Enumeration fastListProcesses() throws NotConnectedException {
		ControlMessage message;

		if (_connection == null) {
			throw new NotConnectedException();
		} else {
			_connection.addConnectionListener(_connectionListener); // to avoid
			// deadlock

			// Query the process list
			message = new ControlMessage();
			QueryProcessListCommand qplCommand = new QueryProcessListCommand();
			message.appendCommand(qplCommand);
			synchronized (_processListLock) {
				try {
					_connection.sendMessage(message, _commandHandler);
					_processListLock.wait(DEFAULT_TIMEOUT);
				} catch (Exception e) {
					e.printStackTrace();
					return null;
				}
			}

			// Populate the process with all agents
			Enumeration ePIDs = _processList.keys();
			while (ePIDs.hasMoreElements()) {
				Long pid = (Long) (ePIDs.nextElement());

				// Query the agent list
				message = new ControlMessage();
				QueryAgentListCommand qalCommand = new QueryAgentListCommand();
				qalCommand.setProcessId(pid.longValue());
				message.appendCommand(qalCommand);
				synchronized (_agentListLock) {
					try {
						_connection.sendMessage(message, _commandHandler);
						_agentListLock.wait(DEFAULT_TIMEOUT);
					} catch (Exception e) {
						e.printStackTrace();
						return null;
					}
				}
			}
			
		}

		return _processList.elements();
	}

	public void removeProcess(long pid) {
		if (_processList.containsKey(new Long(pid))) {
			_processList.remove(new Long(pid));
		}
	}

	/**
	 * Handler for default context (0)
	 */
	public void incommingCommand(Node node, CommandElement command) {
		_commandHandler.incommingCommand(node, command);
	}

	private boolean isLocal() {
		boolean isLocal = false;

		try {
			InetAddress localAddr = InetAddress.getLocalHost();
			for (int i = 0; i < _addr.length; i++) {
				if (localAddr.equals(_addr[i])) {
					isLocal = true;
				}
			}
		} catch (UnknownHostException e) {
		}

		return isLocal;
	}

	public boolean authenticateUser() {
		if(_user != null) {
			synchronized(_authenticationLock) {
				try {
					_user.login(_connection, _commandHandler);
					_authenticationLock.wait(_authenticationTimeout);
				} catch (InterruptedException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		return _isAuthenticated;
	}

	public boolean isUserAuthenticated() {
		return _isAuthenticated;
	}

	class InnerConnectionListener implements ConnectionListener {
		// If connection is lost, release all object locks
		public void connectionClosed(Connection connection) {
			synchronized (_processListLock) {
				_processListLock.notifyAll();
			}
			synchronized (_propertyListLock) {
				_propertyListLock.notifyAll();
			}
			// Can't stay authenticated if connection has closed. Also notify any
			// threads waiting on authentication. This is unlikely to be needed, but
			// guards against unexpected failures.
			synchronized (_authenticationLock) {
				_isAuthenticated = false;
				_authenticationLock.notifyAll();
			}
		}
	}

	class InnerCommandHandler implements CommandHandler {
		public void incommingCommand(Node node, CommandElement command) {
			int tag = (int) command.getTag();
			switch (tag) {
			case (int)Constants.RA_AUTHENTICATION_SUCCESSFUL:
				_isAuthenticated = true;
				synchronized(_authenticationLock) {
					_authenticationLock.notifyAll();
				}
				break;
			case (int)Constants.RA_AUTHENTICATION_FAILED:
				_isAuthenticated = false;
				synchronized(_authenticationLock) {
					_authenticationLock.notifyAll();
				}
				break;
			case (int) Constants.RA_PROCESS_LIST: {
				RegisteredProcessListCommand rplCmd = (RegisteredProcessListCommand) command;
				long[] pids = rplCmd.getProcessList();

				// _processList.clear(); // Dangerous!!! Will remove all agents information within the process
				// Clear the process list if there is no process registered
				if((pids == null) || (pids.length == 0)) {
					_processList.clear();
				}
				else {
					// Compare the returned process list and remove any terminated process
					Enumeration e_tmp = _processList.keys();
					while(e_tmp.hasMoreElements()) {
						Long pidstr_tmp = (Long)e_tmp.nextElement();
						Process p_tmp = (Process)_processList.get(pidstr_tmp);
						try {
							long pid_tmp = Long.parseLong(p_tmp.getProcessId());
							boolean exist = false;
							// Check if the process is still in the returned process list
							for(int i = 0; i < pids.length; i++) {
								if(pids[i] == pid_tmp) {
									exist = true;
								}
							}
							if(!exist) {
								_processList.remove(pidstr_tmp);
							}
						} catch (Exception e) {
						}
					}

					// Populate the process list
					for (int i = 0; i < pids.length; i++) {
						if (!_processList.containsKey(new Long(pids[i]))) { // Only
							// create
							// new
							// process
							// if
							// not
							// already
							// exist
							Process p = ProcessFactory.createProcess(node, pids[i]);
							p.setActive(true);
							_processList.put(new Long(pids[i]), p);
						}
					}
				}

				synchronized (_processListLock) {
					_processListLock.notifyAll();
				}
				break;
			}
			case (int) Constants.RA_AGENT_LIST: {
				ActiveAgentListCommand aalCmd = (ActiveAgentListCommand) (command);

				String[] agentNames = aalCmd.getAgents();
				String processName = aalCmd.getProcessName();
				long processId = aalCmd.getProcessId();

				Process p = (Process) _processList.get(new Long(processId));
				p.setName(processName);

				if (agentNames != null) {
					for (int i = 0; i < agentNames.length; i++) {
						Agent newAgent = AgentFactory.createAgent(p, agentNames[i]);
						newAgent.setActive(true);
					}
				}

				synchronized (_agentListLock) {
					_agentListLock.notifyAll();
				}
				break;
			}
			case (int) Constants.RA_AGENT_DETAILS: {
				AgentDetailsCommand adCmd = (AgentDetailsCommand) command;

				long processId = adCmd.getProcessId();
				String processUUID = adCmd.getProcessUUID();
				String agentName = adCmd.getAgentName();
				String agentType = adCmd.getAgentType();
				String agentUUID = adCmd.getAgentUUID();

				Process p = (Process) _processList.get(new Long(processId));

				if (p != null) {
					p.setUUID(processUUID);
					Agent ag = p.getAgent(agentName);
					if (ag != null) {
						ag.setType(agentType);
						ag.setUUID(agentUUID);
						ag.addAgentListener((ProcessImpl) p);
					}
				}

				synchronized (_agentDetailsLock) {
					_agentDetailsLock.notifyAll();
				}
				break;
			}
			case (int) Constants.RA_PROCESS_LAUNCHED:
			case (int) Constants.RA_PROCESS_EXITED:
			case (int) Constants.RA_AGENT_ACTIVE:
			case (int) Constants.RA_AGENT_INACTIVE:
			case (int) Constants.RA_ERROR_STRING:
			case (int) Constants.RA_CONSOLE_INFO:
			case (int) Constants.RA_CUSTOM_COMMAND:
			case (int) Constants.RA_BINARY_CUSTOM_COMMAND:
			case (int) Constants.RA_SET_NAME_VALUE_PAIR: {
				SimpleProcessCommand cmd = (SimpleProcessCommand) command;
				long pid = cmd.getProcessId();

				Process p = (Process) _processList.get(new Long(pid));
				if ((p != null) && (p instanceof ProcessImpl)) {
					((ProcessImpl) p).handleCommand(command);
				}

				// Remote process if exited
				if (tag == Constants.RA_PROCESS_EXITED) {
					_processList.remove(new Long(pid));
				}

				break;
			}
			case (int) Constants.RA_PROPERTY_LIST: {
				PropertyListCommand qs = (PropertyListCommand) command;
				props = qs.getPropertyListValues();
				synchronized (_propertyListLock) {
					_propertyListLock.notify();
				}
				break;
			}
			case (int) Constants.RA_CONTROLLER_REQUEST_MONITOR:
			case (int) Constants.RA_CONTROLLER_REQUEST_MONITOR_PORT: {
				MonitorPeerRequestCommand cmd = (MonitorPeerRequestCommand) command;
				long pid = cmd.getProcessId();

				Process p = (Process) _processList.get(new Long(pid));
				if (p != null) {
					try {
						((ProcessImpl) p).handleCommand(command);
					} catch (ClassCastException e) {
						// TODO: Error casting Process
					}
				}
				break;
			}
			default: {
			}
			}
		}
	}

	/**
	 * Proess Listener
	 * 
	 * @author samwai
	 * 
	 */
	class InnerProcessListener implements ProcessListener {
		public synchronized void processLaunched(Process process) {
			try {
				Long key = new Long(process.getProcessId());
				if (!_processList.containsKey(key)) {
					_processList.put(key, process);
				}
			} catch (InactiveProcessException e) {
				/* We can ignore this */
			}
		}

		public synchronized void processExited(Process process) {
			try {
				Long key = new Long(process.getProcessId());
				_processList.remove(key);
			} catch (InactiveProcessException e) {
				/* We can ignore this */
			}
		}
	}

}
