/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProcessImpl.java,v 1.20 2010/03/02 02:11:33 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.ConsoleInfoCommand;
import org.eclipse.hyades.internal.execution.local.common.ConsoleNotStartedException;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.DetailedAgentInfoCommand;
import org.eclipse.hyades.internal.execution.local.common.ErrorCommand;
import org.eclipse.hyades.internal.execution.local.common.HyadesExecMessageDebug;
import org.eclipse.hyades.internal.execution.local.common.LaunchProcessCommand;
import org.eclipse.hyades.internal.execution.local.common.MonitorPeerRequestCommand;
import org.eclipse.hyades.internal.execution.local.common.MultiplexedDataServer;
import org.eclipse.hyades.internal.execution.local.common.ProcessExitedCommand;
import org.eclipse.hyades.internal.execution.local.common.ProcessLaunchedCommand;
import org.eclipse.hyades.internal.execution.local.common.SimpleAgentInfoCommand;

public class ProcessImpl implements Process, AgentListener, ConnectionListener {
	protected String _name = null;

	protected String _exe = null;

	protected String _params = null;

	protected String _location = null;

	protected String _processId = null;

	protected String _UUID = null;

	protected Node _node = null;

	protected Console _console = null;

	protected boolean _isActive = false;

	protected boolean _isComplete = false;

	protected Vector _agents = new Vector(10);

	protected Vector _variables = new Vector(20);

	protected Vector _listeners = new Vector(10);

	/* This is a lock for launching the process as this is synchronous */
	private Object _launcherLock = new Object();

	/*
	 * In eclipse, the model changes need to be done on the UI thread, this
	 * requires a bit of additional locking that, by rights, shouldn't be
	 * required
	 */
	private static Object _eclipseLock = new Object();

	/*
	 * Lock used for waitForAgent(...) method and agentActive(...) method
	 * notification
	 */
	private Object _waitForAgentLock = new Object();

	private boolean _noSuchAliasExceptionThrown = false;

	public ProcessImpl(Node node) {
		this(node, 0);
	}

	public ProcessImpl(Node node, long pid) {
		this(node, null, pid);
	}

	public ProcessImpl(Node node, String executable) {
		this(node, executable, null);
	}

	public ProcessImpl(Node node, String executable, String parameters) {
		_node = node;
		_exe = executable;
		_params = parameters;
		_console = getConsole();
	}

	public ProcessImpl(Node node, String executable, long pid) {
		_node = node;
		_exe = executable;
		_processId = new Long(pid).toString();
	}

	/**
	 * @see Process#launch()
	 */
	public void launch() throws ProcessActiveException, NotConnectedException,
			NoSuchApplicationException {
		/*
		 * Bug 112677 Here we use a static object lock to make sure each
		 * ProcessImpl will not cross each other when launching a process. Even
		 * though the ProcessImpl objects are isolated from each other, they are
		 * sharing the same Connection object which is used to send messages to
		 * the Agent Controller
		 */
		synchronized (_eclipseLock) { // Bug 112677 Multiple launches in quick
			// succession hangs

			/* Determine if the process is already launched or if it has exited */
			if (isActive() || _isComplete) {
				throw new ProcessActiveException();
			}

			/* Launch the process */
			ControlMessage message = new ControlMessage();
			LaunchProcessCommand command = new LaunchProcessCommand();
			command.setExe(_exe);
			command.setArgs(_params);
			command.setLocation(_location);
			
			if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_PROCESS_LAUNCH_DEBUG) {
				HyadesExecMessageDebug.writeDebugMessage("Attempting to launch process - exe:["+_exe+"] params:["+_params+"] thread-hash:["+Thread.currentThread().hashCode()+"] proc-hash:["+this.hashCode()+"]");
			}
			
			/* Setup the console */
			_console = getConsole();
			_console.start();
			try {
				command.setConsole(_console);
			} catch (ConsoleNotStartedException e) {
				/* We just started the console so this isn't going to happen */
			}

			Enumeration e1 = _variables.elements();
			while (e1.hasMoreElements()) {
				Variable var = (Variable) e1.nextElement();
				command.addEnvironmentVariable(var.getName(), var.getValue());
			}

			/*
			 * Enumerate all the agents and add the interest only if autoAttach
			 * is true
			 */
			Enumeration e2 = _agents.elements();
			while (e2.hasMoreElements()) {
				Agent agent = (Agent) e2.nextElement();
				command.addAgent(agent.getName());
			}

			message.appendCommand(command);
			Connection connection = _node.getConnection();
			try {
				/*
				 * We need to listen in case this connection gets dropped while
				 * waiting for the launch to happen
				 */
				connection.addConnectionListener(this);
				connection.sendMessage(message, new CommandHandler() {
					public void incommingCommand(Node node,
							CommandElement command) {
						handleCommand(command);
					}
				});
			} catch (IOException e) {
				throw new NotConnectedException();
			}

			long waitStart = System.currentTimeMillis();
			
			synchronized (_launcherLock) {
				try {
					/*
					 * Never wait more than 10 seconds in case something goes
					 * wrong
					 */
					_launcherLock.wait(10000);
				} catch (InterruptedException e) {
					/* We may need to handle this exception */
				}
			}
			
			if(HyadesExecMessageDebug.HYADES_PROCESS_LAUNCH_DEBUG && HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG) {
				HyadesExecMessageDebug.writeDebugMessage("Process launch wait time is "+(System.currentTimeMillis() - waitStart)+" msecs  - thread-hash:["+Thread.currentThread().hashCode()+"] proc-hash:["+this.hashCode()+"]");
			}
			
			if(_console != null) {
				_console.setDebugProcess(this);
			}
			
			if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_PROCESS_LAUNCH_DEBUG) {
				HyadesExecMessageDebug.writeDebugMessage("Process Launched - pid:["+_processId+"] exe:["+_exe+"] params:["+_params+"] thread-hash:["+Thread.currentThread().hashCode()+"] proc-hash:["+this.hashCode()+"]");
			}

			/* If we have a pending NoSuchApplication error we need to throw it. */
			if (_noSuchAliasExceptionThrown) {
				_noSuchAliasExceptionThrown = false;
				throw new NoSuchApplicationException();
			}

		} // _eclipseLock
	}

	/**
	 * @see Process#kill(long)
	 */
	public void kill(long delay) throws InactiveProcessException,
			NotConnectedException {
		if (_node == null) {
			throw new InactiveProcessException();
		}
		_node.killProcess(this);
	}

	/**
	 * @see Process#addAgent()
	 */
	public void addAgent(Agent agent) {
		if (!_agents.contains(agent)) {
			_agents.addElement(agent);
		}

	}

	public void removeAgent(Agent agent) {
		if (_agents.contains(agent)) {
			_agents.remove(agent);
		}
	}

	/**
	 * @see Process#listAgents()
	 */
	public Enumeration listAgents() {
		return ((Vector) _agents.clone()).elements();
	}

	/**
	 * Remove agents from the process if their name do not exist in the list
	 * provided
	 * 
	 * @param newList
	 *            List of agents that should not be removed
	 */
	// Bug 71586 begins
	public void removeInactiveAgents(String[] newList) {
		Enumeration agents = listAgents();
		while (agents.hasMoreElements()) {
			Agent a = (Agent) agents.nextElement();

			boolean active = false;
			// Check if the agent is still shown in the new list
			for (int i = 0; i < newList.length; i++) {
				if (a.getName().equals(newList[i])) { // Compare the agent's
					// name
					active = true;
				}
			}

			// If the agent name does not appear in the new list, remove it.
			if (!active) {
				_agents.remove(a);
			}
		}
	}

	// Bug 71586 ends

	/**
	 * @see Process#getAgentsByType()
	 */
	public Enumeration getAgentsByType(String type) {
		Vector results = new Vector();
		synchronized (_agents) {
			Enumeration agents = listAgents();
			while (agents.hasMoreElements()) {
				Agent agent = (Agent) agents.nextElement();
				String agentType = agent.getType();
				if (agentType != null && agentType.equals(type)) {
					results.addElement(agent);
				}
			}
		}
		return results.elements();
	}

	public Agent getAgent(String name) {
		synchronized (_agents) {
			Enumeration agents = listAgents();
			while (agents.hasMoreElements()) {
				Agent agent = (Agent) agents.nextElement();
				if (agent.getName().equals(name)) {
					return agent;
				}
			}
		}
		return null;
	}

	/**
	 * @see Process#getEnvironment()
	 */
	public Enumeration getEnvironment() {
		return _variables.elements();
	}

	/**
	 * @see Process#addProcessListener(ProcessListener)
	 */
	public void addProcessListener(ProcessListener listener) {
		synchronized (_listeners) {
			if (!_listeners.contains(listener)) {
				_listeners.add(listener);
			}
		}
	}

	/**
	 * @see Process#removeProcessListener(ProcessListener)
	 */
	public void removeProcessListener(ProcessListener listener) {
		synchronized (_listeners) {
			if (_listeners.contains(listener)) {
				_listeners.remove(listener);
			}
		}
	}

	/**
	 * @see Process#getNode()
	 */
	public Node getNode() {
		return _node;
	}

	/**
	 * @see Process#setName()
	 */
	public void setName(String name) {
		_name = name;
	}

	/**
	 * @see Process#getName()
	 */
	public String getName() {
		return _name;
	}

	/**
	 * @see Process#getUUID()
	 */
	public String getUUID() {
		// Bug 58583
		int retryCount = 30; // retry 30 times
		int delay = 1000; // 1 sec between retry, total 30 sec max
		int i = 0;

		while ((_UUID == null) && (i < retryCount)) {
			try {
				Thread.sleep(delay);
			} catch (InterruptedException e) {
			}
			i++;
		}

		return _UUID;
	}

	/**
	 * @see Process#isActive()
	 */
	public boolean isActive() {
		return _isActive;
	}

	public void setActive(boolean isActive) {
		_isActive = isActive;
	}

	/**
	 * @see Process#getConsole()
	 */
	public Console getConsole() {

		if (_console == null) {

			Connection conn = getNode().getConnection();
			this._console = new Console(this._node);
			
		}
		return _console;
	}


	/**
	 * @see Process#addEnvironmentVariable()
	 */
	public void addEnvironmentVariable(Variable variable)
			throws ProcessActiveException {
		if (_isActive) {
			throw new ProcessActiveException();
		}
		_variables.add(variable);

	}

	/**
	 * @see Process#getEnvironmentVariable
	 */
	public Variable getEnvironmentVariable(String name) {
		Enumeration e = getEnvironment();
		while (e.hasMoreElements()) {
			Variable var = (Variable) e.nextElement();
			if (var.getName().equals(name)) {
				return var;
			}
		}
		return null;
	}

	/**
	 * @see Process#removeEnvironmentVariable
	 */
	public void removeEnvironmentVariable(Variable variable)
			throws ProcessActiveException {
		if (_isActive) {
			throw new ProcessActiveException();
		}
		_variables.remove(variable);

	}

	/**
	 * @see Process#setExecutable()
	 */
	public void setExecutable(String exe) throws ProcessActiveException {
		if (_isActive) {
			throw new ProcessActiveException();
		}
		_exe = exe;
	}

	/**
	 * @see Process#getExecutable()
	 */
	public String getExecutable() {
		return _exe;
	}

	/**
	 * @see Process#setParameters()
	 */
	public void setParameters(String params) throws ProcessActiveException {
		if (_isActive) {
			throw new ProcessActiveException();
		}
		_params = params;
	}

	/**
	 * @see Process#getParameters()
	 */
	public String getParameters() {
		return _params;
	}

	/**
	 * @see Process#setLocation()
	 */
	public void setLocation(String location) throws ProcessActiveException {
		if (_isActive) {
			throw new ProcessActiveException();
		}
		_location = location;
	}

	/**
	 * @see Process#getProcessId()
	 */
	public String getProcessId() throws InactiveProcessException {
		// Bug 58583
		int retryCount = 30; // retry 30 times
		int delay = 1000; // 1 sec between retry, total 30 sec max
		int i = 0;

		while ((_processId == null) && (i < retryCount)) {
			try {
				Thread.sleep(delay);
			} catch (InterruptedException e) {
			}
			i++;
		}

		if (_processId == null) {
			throw new InactiveProcessException();
		}
		return _processId;
	}

	public void setProcessId(String pid) {
		_processId = pid;
	}

	/**
	 * @see AgentListener#agentActive
	 */
	public void agentActive(Agent agent) {
		if (!_agents.contains(agent)) {
			_agents.add(agent);
		}
		/*
		 * Notify any waiting thread in the waitForAgent(...) method to minimize
		 * time spent waiting for next poll point.
		 */
		synchronized (this._waitForAgentLock) {
			this._waitForAgentLock.notifyAll();
		}
	}

	/**
	 * @see AgentListener#agentInactive
	 */
	public void agentInactive(Agent agent) {
		/* Should we remove this from the agent list? */
		if (agent instanceof AgentImpl) {
			AgentImpl _agent = (AgentImpl) agent;
			if (_agent._listeners != null && _agent._listeners.size() > 0) {
				for (int i = 0; i < _agent._listeners.size(); i++) {
					Object listener = _agent._listeners.get(i);
					if (listener instanceof MultiplexedDataServer)
						return;
				}
			}
		}

		if (_agents.contains(agent)) {
			_agents.remove(agent);
		}
	}

	/**
	 * @see AgentListener#error
	 */
	public void error(Agent agent, String errorId, String errorMessage) {
		/* Ignore */
	}

	/**
	 * @see AgentListener#handleCommand
	 */
	public void handleCommand(Agent agent, CommandElement command) {
		/* Ignore */
	}

	/**
	 * @see ConnectionListener#connectionClosed
	 */
	public void connectionClosed(Connection connection) {
		/*
		 * If there is a pending launch lock we need to release it as our
		 * connection has been lost
		 */
		synchronized (_launcherLock) {
			_launcherLock.notify();
		}
	}

	public String getlocation() {
		return _location;
	}

	/**
	 * @see Process#getUUID()
	 */
	public void setUUID(String uuid) {
		_UUID = uuid;
	}

	protected void handleCommand(CommandElement command) {
		Enumeration agents = null;

		switch ((int) command.getTag()) {
		
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_LAUNCHED:
			/* Only process a ProcessLaunched command once */
			if (_isActive) {
				break;
			}
		

			/* Extract all the information for this process */
			ProcessLaunchedCommand plc = (ProcessLaunchedCommand) command;
			_processId = new Long(plc.getProcessId()).toString();
			_UUID = plc.getProcessUUID();

			if(HyadesExecMessageDebug.HYADES_EXEC_MSG_DEBUG && HyadesExecMessageDebug.HYADES_PROCESS_LAUNCH_DEBUG) {
				HyadesExecMessageDebug.writeDebugMessage("RA_PROCESS_LAUNCHED received - pid:["+_processId+"] exe:["+_exe+"] params:["+_params+"] proc-hash:["+this.hashCode()+"]");
			}
			
			/* Capture all the environment variables */
			_variables.removeAllElements();
			String[] vars = plc.getEnvironment();
			for (int i = 0; i < vars.length; i++) {
				int equalsOffset = vars[i].indexOf('=');
				if (equalsOffset > 0) {
					String name = vars[i].substring(0, equalsOffset);
					String value = vars[i].substring(equalsOffset + 1);
					_variables.addElement(new Variable(name, value));
				}
			}

			_isActive = true;

			try {
				/* Inform the listeners that the process is launched */
				synchronized (_listeners) {
					Enumeration e = _listeners.elements();
					while (e.hasMoreElements()) {
						((ProcessListener) e.nextElement())
								.processLaunched(this);
					}
				}

			} finally {
				/*
				 * Remove our connection listener and if the connection is hosed
				 * we need to throw a exception
				 */
				Connection connection = _node.getConnection();
				connection.removeConnectionListener(this);
				// if(!connection.isActive()) {
				// throw new NotConnectedException();
				// }

				/*
				 * If our process failed to launch properly then we need to stop
				 * the console thread
				 */
				if (!isActive() && _console != null) {
					_console.close();
				}

			}

			/* Notify that the process is launched so "launch()" can return */
			synchronized (_launcherLock) {
				_launcherLock.notify();
			}
			break;
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_ACTIVE:
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_AGENT_INACTIVE:
			/*
			 * When an agent becomes active/inactive we need to ensure it is a
			 * member of the agent list for this process. We also need to inform
			 * all the listeners for this agent that it is active/inactive. This
			 * is accomplished by delegating to the handleCommand() of the
			 * AgentImpl
			 */
			DetailedAgentInfoCommand aac = (DetailedAgentInfoCommand) command;
			synchronized (_agents) {
				Agent agent = null;
				agents = _agents.elements();
				while (agents.hasMoreElements()) {
					Agent current = (Agent) agents.nextElement();
					if (current.getName().equals(aac.getAgentName())) {
						agent = current;
						break;
					}
				}
				if (agent == null) {
					agent = AgentFactory.createAgent(this, aac.getAgentName(),
							aac.getAgentType()); // TODO: Should we do this
					// for RA_AGENT_INACTIVE?
					((AgentImpl) agent).setUUID(aac.getAgentUUID());
				}

				/*
				 * Forward this command to the agent so it can handle the
				 * details
				 */
				try {
					AgentImpl agentImpl = (AgentImpl) agent;
					if (agent.getUUID() == null) {
						agentImpl.setUUID(aac.getAgentUUID());
					}
					agentImpl.handleCommand(command);

				} catch (ClassCastException exc) {
					/*
					 * This is not an instance of AgentImpl, do not attempt to
					 * notify it is active
					 */
				}

			}
			break;
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_ERROR_STRING:
			/* When we are launching a process there could be an error */
			ErrorCommand ec = (ErrorCommand) command;
			if (ec.getErrorId().equals("RAC002")) {
				_noSuchAliasExceptionThrown = true;
				synchronized (_launcherLock) {
					_launcherLock.notify();
				}
			}

			agents = _agents.elements();
			while (agents.hasMoreElements()) {
				Agent agent = (Agent) agents.nextElement();
				if (agent.getName().equals(ec.getAgentName())) {
					try {
						((AgentImpl) agent).handleCommand(command);
					} catch (ClassCastException exc) {
						// TODO: Error casting agent
					}

				}
			}

			break;
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_PROCESS_EXITED:
			ProcessExitedCommand peCmd = (ProcessExitedCommand) command;
			if (_console != null) {
				_console.close(); /* Close the console port when process exits */
			}

			if (!_isComplete) {
				_isComplete = true;
				synchronized (_listeners) {
					Enumeration e = _listeners.elements();
					while (e.hasMoreElements()) {
						((ProcessListener) e.nextElement()).processExited(this);
					}
				}

				// Clean up process members as the process exits (avoid memory
				// leaks)
				this._listeners.clear();
				this._variables.clear();
				this._agents.clear();

			}

			/* Notify the agents */
			agents = _agents.elements();
			while (agents.hasMoreElements()) {
				Agent agent = (Agent) agents.nextElement(); // Notify all agents
				try {
					((AgentImpl) agent).handleCommand(command);
				} catch (ClassCastException exc) {
					// TODO: Error casting agent
				}
			}

			((NodeImpl) _node).removeProcess(peCmd.getProcessId());

			break;
		case (int) org.eclipse.hyades.internal.execution.local.common.Constants.RA_CONSOLE_INFO:
			ConsoleInfoCommand ciCmd = (ConsoleInfoCommand) command;
			break;
		case (int) Constants.RA_CONTROLLER_REQUEST_MONITOR:
		case (int) Constants.RA_CONTROLLER_REQUEST_MONITOR_PORT: {
			MonitorPeerRequestCommand cmd = (MonitorPeerRequestCommand) command;
			String agentName = cmd.getAgentName();

			agents = _agents.elements();
			while (agents.hasMoreElements()) {
				Agent agent = (Agent) agents.nextElement(); // Notify all agents
				if (agent.getName().equals(agentName)) {
					try {
						((AgentImpl) agent).handleCommand(command);
					} catch (ClassCastException exc) {
						// TODO: Error casting agent
					}
				}
			}

			break;
		}
		default:
			if (command instanceof SimpleAgentInfoCommand) {
				/*
				 * Custom commands can be generated by agents running on this
				 * process. Find the proper agent and forward the message on to
				 * it. This gets invoked only when a launch is done not attach,
				 * as these are routed directly to the agent.
				 */
				Agent agent = getAgent(((SimpleAgentInfoCommand) command)
						.getAgentName());
				try {
					if (agent != null) {
						((AgentImpl) agent).handleCommand(command);
					}
				} catch (ClassCastException e) {
					/* This isn;t our impl so we can ignore the command */
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.internal.execution.local.control.Process
	 *      #waitForAgent(java.lang.String, java.lang.String, int)
	 * 
	 * Waits for a particular agent to be present or the timeout value to be
	 * reached; after the agent list returns (the internal agent list in this
	 * process instance is updated) the agent is retrieved via the get agent
	 * method. The timeout value is approximate since there is some time needed
	 * to execute the various operations within the loop and each wait is done
	 * timeout divided by 2000 times. For example, a 60000 (60 second) timeout
	 * would cause the loop to execute 30 times and wait for 2 seconds at the
	 * end of each process list query.
	 */
	public Agent waitForAgent(String agentType, String agentName, int timeout)
			throws NotConnectedException, InactiveProcessException {

		synchronized (this._waitForAgentLock) {

			// Change this from a polling model into a notify model (this is
			// good enough for now)
			try {
				for (int i = 0, j = timeout / 2000; i < j; i++) {
					this.getNode().listProcesses(); // bugzilla_92619 (without
					// this, agents are never
					// refreshed)
					for (Enumeration agents = this.listAgents(); agents
							.hasMoreElements();) {
						Agent agent = (Agent) agents.nextElement();
						if (agent.getName().indexOf(agentName) != -1) { // checks
							// for
							// proper
							// substring
							return agent;
						}
					}
					this._waitForAgentLock.wait(2000);
				}
			} catch (InterruptedException e) {
				// Do not need to handle this
				e.printStackTrace();
			}

			return null; // no agent was found in the allowed time
		}

	}

}
