/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerExtendedTest.java,v 1.30 2009/09/28 19:17:46 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.test.execution.local;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;
import java.util.Random;

import org.eclipse.core.runtime.Assert;
import org.eclipse.hyades.execution.core.DaemonConnectException;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.internal.execution.core.file.ServerNotAvailableException;
import org.eclipse.hyades.internal.execution.file.FileServer;
import org.eclipse.hyades.internal.execution.file.FileServiceConstants;

/**
 * File transfer test to exercise the current file manager extended
 * implementation via the file manager extended class and interface. This class
 * is usable outside of Eclipse as a standalone test. Some test files are
 * generated first and then these test files are operated on using the new file
 * manager implementation. The test files are generated with exponentially
 * increasing lengths to give a good sample of various file lengths. File name
 * plurality, progress monitor support and various additional operational
 * semantic fine-tuning is tested with this class. Additional javadocs can be
 * found in the IFileManagerExtended interface definition. Timing of file
 * transfer operations will also be conducted to demonstrate that file transfer
 * performance has increased about 50 times with the minimum transfer time taken
 * no longer starting at approximately 3s. In a typical test, a typical Java
 * class takes 50ms to transfer to the remote host and a large 16MB file takes
 * about 3s. File transfer in the opposite direction, to the local host, is
 * comparable.
 * 
 * 
 * This test tests facets of the file transfer service:
 * <li>Absolute versus relative file identifiers supported
 * <li>Multiple and single file command operands
 * <li>Query server, get file, put file and delete file commands
 * <li>Client and server-side thread concurrency
 * <li>Tiny to very large files sizes (via FILE_COUNT)
 * <li>Loading down of ephemeral client ports and recovery
 * 
 * @author Scott E. Schneider
 */
public class FileManagerExtendedTest {

	/**
	 * Amount of time to wait between command bursts
	 */
	private static final int COMMAND_INTERVAL_SLEEP = 1000;

	/**
	 * Number of concurrent threads that will run an instance of this test, no
	 * duplicate filenames will be used
	 */
	private static final int CONCURRENT_CLIENT_THREADS = 1;

	/**
	 * The number or worker threads that loop over the test body (not creating
	 * new files with different names) for each concurrent client thread as
	 * defined by CONCURRENT_CLIENT_THREADS
	 */
	private static final int CONCURRENT_CLIENT_WORKER_THREADS = 1;

	/**
	 * Tests features on agent controllers of TPTP version 4.1 or greater
	 */
	private static final boolean ENABLE_EXTENDED_FEATURES_4_1 = true;

	/**
	 * The number of size-unique files to generate
	 */
	private static final int FILE_COUNT = 25;

	/**
	 * The number of samples for each size-unique file instance generated (for
	 * example, five samples indicates that five files will be used for each
	 * size plateau)
	 */
	private static final int FILE_SAMPLE_SIZE = 4;

	/**
	 * The file content header prefix to use for each generated file
	 */
	private static final String HEADER = "File transfer test with a file of size: ";//$NON-NLS-1$

	/**
	 * The host that is running the agent controller of interest
	 */
	private static final String HOST = "127.0.0.1";//$NON-NLS-1$

	/**
	 * The number of times each command is executed, increase this to get more
	 * accurate and over-time averages
	 */
	private static final int ITERATIONS = 10;

	/**
	 * The file size offset; each file must be at least this size
	 */
	private static final int OFFSET = 4096;

	/**
	 * Delay to use before a new instance of this test on a concurrent thread is
	 * started, used to stagger to operations a bit
	 */
	private static final int THREAD_START_DELAY = 1250;

	/**
	 * Sets up and executes extended file manager test
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		for (int i = 0; i < FileManagerExtendedTest.CONCURRENT_CLIENT_THREADS; i++) {
			new Thread() {
				public void run() {
					new FileManagerExtendedTest().execute();
				}
			}.start();
			synchronized (Thread.currentThread()) {
				try {
					Thread.currentThread().wait(
							FileManagerExtendedTest.THREAD_START_DELAY);
				} catch (Throwable t) {
					t.printStackTrace();
				}
			}
		}
	}

	/**
	 * Create absolute identifiers as destination identifiers given the source
	 * identifiers.
	 * 
	 * @param sourceIdentifiers
	 *            the source identifiers
	 * @return the absolute identifiers
	 */
	private FileIdentifierList createAbsoluteIdentifiers(
			FileIdentifierList sourceIdentifiers) {
		FileIdentifierList destinationIdentifiers = FileIdentifierList.create();
		for (Iterator iterator = sourceIdentifiers.iterator(); iterator
				.hasNext();) {
			String fileIdentifier = (String) iterator.next();
			int extension = fileIdentifier.lastIndexOf(".");
			destinationIdentifiers.add(fileIdentifier.substring(0, extension)
					.concat("-destination")//$NON-NLS-1$
					+ fileIdentifier.substring(extension));
		}
		return destinationIdentifiers;
	}

	/**
	 * Creates and populates files given the name, extension and size, files are
	 * marked to delete automatically on virtual machine exit
	 * 
	 * @param name
	 *            the name of the file to create and populate
	 * @param extension
	 *            the file extension for the name
	 * @param size
	 *            the size of the file to be generated
	 * @return the file populated
	 * @throws IOException
	 */
	private File createAndPopulateFile(String name, String extension, int size)
			throws IOException {
		File temp = File.createTempFile(name, extension);
		temp.deleteOnExit(); // clean up after tests exit cleanly
		RandomAccessFile file = new RandomAccessFile(temp, "rw");//$NON-NLS-1$
		FileChannel channel = file.getChannel();
		ByteBuffer byteBuffer = ByteBuffer.allocateDirect(size);
		byteBuffer.clear();
		String string = FileManagerExtendedTest.HEADER + size;
		byteBuffer.put(string.getBytes("UTF-8"));//$NON-NLS-1$
		byteBuffer.put(this.getFixedLengthRandomBytes(size - string.length()));
		byteBuffer.flip();
		channel.position(0);
		channel.write(byteBuffer);
		channel.close();
		file.close();
		return temp;
	}

	/**
	 * Create relative identifiers as destination identifiers given the source
	 * identifiers.
	 * 
	 * @param sourceIdentifiers
	 *            the source identifiers
	 * @return the relative identifiers
	 */
	private FileIdentifierList createRelativeIdentifiers(
			FileIdentifierList sourceIdentifiers) {
		FileIdentifierList destinationIdentifiers = FileIdentifierList.create();
		for (Iterator iterator = sourceIdentifiers.iterator(); iterator
				.hasNext();) {
			String fileIdentifier = (String) iterator.next();
			int extension = fileIdentifier.lastIndexOf(".");
			if (extension <= -1) {
				throw new IllegalArgumentException();
			}
			int separator = fileIdentifier.indexOf(File.separatorChar);
			if (separator <= -1) {
				throw new IllegalArgumentException();
			}
			destinationIdentifiers.add(fileIdentifier.substring(separator + 1,
					extension).concat("-destination2")//$NON-NLS-1$
					+ fileIdentifier.substring(extension));
		}
		return destinationIdentifiers;
	}

	/**
	 * Sets up, connects and executes the tests
	 */
	public void execute() {

		// Track if node was passed in
		boolean alreadyExisting = true;

		// Allocate session variable
		ISession session = null;

		// Allocate node variable
		NodeImpl node = null;

		// Establish node if not given
		if (node == null) {

			// Wasn't passed in to this method
			alreadyExisting = false;

			// Create a node pointing to local machine
			node = new NodeImpl(FileManagerExtendedTest.HOST);

			// Establish a session by connecting to the node
			try {
				session = node.connect("10002", null,
						FileManagerExtendedTest.ITERATIONS);//$NON-NLS-1$
			} catch (UnknownHostException e) {
				//
			} catch (UnknownDaemonException e) {
				//
			} catch (DaemonConnectException e) {
				//
			}

		}

		// Declare file manager variable
		IFileManager fileManager = null;

		// If the agent controller is not available, create test file server
		if (session == null && alreadyExisting == false) {

			// Create test file server on specified port
			FileServer fileServer = new FileServer(0);
			fileServer.run();

			// Get file server status blocks until it should be running
			if (fileServer.getFileServerStatus() != FileServiceConstants.RA_FS_STATUS_OK) {
				// Wait for file server to completely initialize
			}

			// Attempt to connect to standalone file transfer server
			try {

				// Obtain file manager extended to command server
				fileManager = FileManagerFactory.getInstance().createTimed(
						new InetSocketAddress("localhost", FileServer//$NON-NLS-1$
								.getServerPort()),
						FileManagerExtendedTest.ITERATIONS);

			} catch (ServerNotAvailableException e) {
				//
			}

		} else {

			// Retrieve the file manager implementation from the node
			fileManager = ((NodeImpl) node).getFileManager();
			if (fileManager == null) {
				try {
					fileManager = FileManagerFactory.getInstance().createTimed(
							node.node.getConnection(),
							FileManagerExtendedTest.ITERATIONS);
				} catch (ServerNotAvailableException e) {
					//
				}
			}

		}

		// Assert that the new file manager extended implementation is available
		if (fileManager == null
				|| !(fileManager instanceof IFileManagerExtended)) {
			throw new NullPointerException();
		}

		final IFileManagerExtended manager = (IFileManagerExtended) fileManager;

		try {

			// Generate test files used for the file manager operations
			final FileIdentifierList sourceFileIdentifiers = this
					.generateSourceTestFiles();
			final FileIdentifierList destinationFileIdentifiersA = this
					.createAbsoluteIdentifiers(sourceFileIdentifiers);
			final FileIdentifierList destinationFileIdentifiersB = this
					.createRelativeIdentifiers(sourceFileIdentifiers);
			final FileIdentifierList sourceFileIdentifier = FileIdentifierList
					.create((String) sourceFileIdentifiers.get(0));
			final FileIdentifierList destinationFileIdentifier = FileIdentifierList
					.create("single-file-test-destination.dummy");//$NON-NLS-1$

			// Use client worker threads to test overlapping file transfer
			for (int i = 0; i < FileManagerExtendedTest.CONCURRENT_CLIENT_WORKER_THREADS; i++) {
				new Thread() {
					public void run() {

						try {

							// Assert that the server is available
							if (!manager.isServerAvailable()) {
								throw new ServerNotAvailableException(
										new Throwable());
							}

							Thread
									.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

							// Upload files to server
							manager.putFile(sourceFileIdentifiers,
									destinationFileIdentifiersA);

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								// Upload files to server (relative paths)
								manager.putFile(sourceFileIdentifiers,
										destinationFileIdentifiersB);

								// Upload a single file (relative target path)
								manager.putFile(sourceFileIdentifier,
										destinationFileIdentifier);

							}

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								Thread
										.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

								// Assert directory existence on server
								boolean[] existent = manager
										.isDirectoryExistent(destinationFileIdentifiersA
												.getParentIdentifiers());
								Assert.isNotNull(existent);

							}

							Thread
									.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

							// Download files to client side (without options)
							manager.getFile(sourceFileIdentifiers,
									destinationFileIdentifiersA);

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								/*
								 * Download files to client side (absolute
								 * targets)
								 */
								manager.getFile(sourceFileIdentifiers,
										destinationFileIdentifiersB);

								// Download the single file (now absolute path)
								manager.getFile(sourceFileIdentifier,
										destinationFileIdentifier);

							}

							Thread
									.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

							// Delete files from the remote host
							manager.deleteFile(destinationFileIdentifiersA);

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								// Delete files from the remote host (absolute
								manager.deleteFile(destinationFileIdentifiersB);

								// Delete single file from remote host
								manager.deleteFile(destinationFileIdentifier);

							}

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								Thread
										.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

								// Delete directories from the remote host
								manager
										.deleteDirectory(destinationFileIdentifiersA
												.getParentIdentifiers());

							}

							Thread
									.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

							// If extended features are supported
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								Thread
										.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

								// Assert directory existence on server
								boolean[] existent = manager
										.isDirectoryExistent(destinationFileIdentifiersA
												.getParentIdentifiers());
								Assert.isNotNull(existent);

							}

							// Determine server reach on a few combinations
							if (FileManagerExtendedTest.ENABLE_EXTENDED_FEATURES_4_1) {

								Thread
										.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

								// Reach google on port 80, assumes web server
								Assert.isTrue(manager.determineServerReach(
										"www.google.com", 80));//$NON-NLS-1$

								// Reach bad network address on arbitrary port
								Assert.isTrue(!manager
										.determineServerReach(
												"this.should.not.exist", 80));//$NON-NLS-1$

								// Reach valid host and invalid port
								Assert.isTrue(!manager
										.determineServerReach("localhost",//$NON-NLS-1$
												32000));

							}

							Thread
									.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

						} catch (Throwable t) {
							t.printStackTrace();
						}

					}
				}.start();
			}

		} catch (IOException e) {

			// Dump stack trace if issues arise
			e.printStackTrace();

		}

		// Be a responsible TPTP-citizen and release the session
		if (session != null) {
			session.release();
		}

	}

	/**
	 * Generates source test files for the test, the files used as source
	 * arguments in file operations
	 * 
	 * @return the file identifier list of the generated test files
	 * @throws IOException
	 */
	private FileIdentifierList generateSourceTestFiles() throws IOException {

		// Initialize new file identifier list to represent the file identities
		FileIdentifierList generatedFiles = FileIdentifierList.create();

		// First create test files to use in file transfer testing
		File[] files = new File[FileManagerExtendedTest.FILE_COUNT];
		for (int i = 0; i < FileManagerExtendedTest.FILE_COUNT; i++) {
			int size = (int) Math.pow(2, i) + FileManagerExtendedTest.OFFSET;
			for (int j = 0; j < FileManagerExtendedTest.FILE_SAMPLE_SIZE; j++) {
				files[i] = this.createAndPopulateFile("file_" + j + "-"
						+ String.valueOf(size - FileManagerExtendedTest.OFFSET)
						+ "+" + String.valueOf(FileManagerExtendedTest.OFFSET)
						+ "__", ".dummy", size);//$NON-NLS-1//$NON-NLS-2$
				generatedFiles.add(files[i]);
			}
		}

		return generatedFiles;

	}

	/**
	 * Calculates a random byte array of the count specified number of elements
	 * 
	 * @param count
	 *            the length of the random byte array
	 * @return the random byte array
	 */
	private byte[] getFixedLengthRandomBytes(int count) {
		Random random = new Random(System.currentTimeMillis());
		byte[] bytes = new byte[count];
		random.nextBytes(bytes);
		return bytes;
	}

}