/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RemoteComponentSkeleton.java,v 1.16 2008/06/11 20:06:19 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.remote;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.hyades.execution.remote.internal.resources.RemoteResourceBundle;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;

public class RemoteComponentSkeleton {

	/*
	 * Whether the native impls are available; the value of this is fully known
	 * at class initialization time and is set by a static initializer in this
	 * class. Is immutable once set in the static initializer to avoid potential
	 * threading issues across instances of this class.
	 */
	private static final boolean nativesAvailable;
	
	/*
	 * Environment can override the directory from which the agent controller libraries are loaded.
	 */
	public static final String AC_LIBRARY_PATH = "ac.library.path";//$NON-NLS-1$

	/* Whether the agnet is being monitored currently */
	private boolean isLogging = false;

	/* Whether the agent is registered with the Agent Controller */
	private boolean isInitialized = false;

	/* Listeners for monitoring events */
	protected final Vector monitorListeners = new Vector();

	/* Listeners for agent controller events */
	protected final Vector serverListeners = new Vector();

	/* Listeners for custom commands */
	protected final Vector commandListeners = new Vector();

	/* Listeners for client events */
	protected final Vector clientListeners = new Vector();

	/* Whether the UUID's for the node and JVM are valid */
	private boolean uuidsValid = false;

	/* The UUID's usedby the Agent Controller */
	private String agentUUID;

	private String nodeUUID;

	private String jvmUUID;

	/* The following fields are used by the native code. DO NOT MODIFY */
	private String name;

	private String type;

	private boolean isRegistered = false;

	/*
	 * This is a instance storage mechanism that the native code uses to quickly
	 * locate the logging mechanism data that is stored in the native library.
	 * DO NOT MODIFY THIS VARIABLE
	 */
	private long mechanism = -1; /*
									 * bugzilla 65470 - changed initial value
									 * from 0 to -1 because 0 is a valid list
									 * index
									 */

	/*
	 * Whether the Agent Controller is running on the local box. A single flag
	 * allows us to streamline handshaking for multiple agent instances.
	 */
	private static boolean serviceAvailable = false;

	/* The number of agents */
	private static int agentCount = 0;

	/* Whether we are currently requesting a remote monitor through a peer agent */
	private boolean requestForMonitor = false;

	/* Bug 105188 begins */
	private Object _optionLock = new Object();
	/* Bug 105188 ends */

	/*
	 * Try and load the native library. If this fails we will ignore the error;
	 * first assign to local variable to determine if natives are loadable and
	 * then assign blank final natives available static variable.
	 * 
	 * if JVM argument -Dac.library.path=<PATH> is specified the DLL's will load from 
	 * this directory, otherwise DLL's in the system PATH will be loaded.
	 */
	static {
		boolean nativesLoadable = false;
		String osName = (String) System.getProperties().get("os.name");//$NON-NLS-1$
		String ext = (osName.toUpperCase().startsWith("WIN") ? ".dll" : ".so");//$NON-NLS-1$

		try {
			String acLibraryPath = System.getProperty(AC_LIBRARY_PATH, null);
			if (acLibraryPath != null && !acLibraryPath.endsWith(File.separator)) {
				acLibraryPath = acLibraryPath.concat(File.separator);
			}
		
			if (acLibraryPath != null) {
				// Bug 112749 begins
				System.load(acLibraryPath+"hcclco"+ext); //$NON-NLS-1$
				System.load(acLibraryPath+"hccls"+ext); //$NON-NLS-1$
				System.load(acLibraryPath+"hcbnd"+ext); //$NON-NLS-1$
				System.load(acLibraryPath+"hcclsm"+ext); //$NON-NLS-1$
				System.load(acLibraryPath+"hccldt"+ext); //$NON-NLS-1$
				// Bug 112749 ends
				System.load(acLibraryPath+"hcjbnd"+ext); //$NON-NLS-1$
				nativesLoadable = true;
			}
		} catch (Throwable e) {
			e.printStackTrace(System.err);
		}
		
		try {
			if (!nativesLoadable) {
				// Bug 112749 begins
				System.loadLibrary("hcclco");//$NON-NLS-1$
				System.loadLibrary("hccls");//$NON-NLS-1$
				System.loadLibrary("hcbnd");//$NON-NLS-1$
				System.loadLibrary("hcclsm");//$NON-NLS-1$
				System.loadLibrary("hccldt");//$NON-NLS-1$
				// Bug 112749 ends
				System.loadLibrary("hcjbnd");//$NON-NLS-1$
				nativesLoadable = true;
			}
		} catch (Throwable e) {
			//Ignore since the natives are not on the system path and the RemoteComponentSkeleton gracefully handles the case when the natives are not available. 
		}
		
		nativesAvailable = nativesLoadable;
	}

	/* Default configuration for this IRemoteEntity. */
	private AgentConfiguration _defaultConfiguration = null;

	/**
	 * Create aaAgent with the specified name and type<br>
	 * 
	 * @param name
	 *            java.lang.String - the remotely identifyable name of the agent
	 * @param type
	 *            java.lang.String - the remotely identifyable type of the agent
	 */
	public RemoteComponentSkeleton(String name, String type) {
		super();
		setName(name);
		setType(type);

		/*
		 * Create the default configuration for this IRemoteEntity using its
		 * name.
		 */
		_defaultConfiguration = new AgentConfiguration();
		_defaultConfiguration.setAgentName(name);
	}

	/**
	 * Add a listener for agent controller events
	 * 
	 * @param listener -
	 *            org.eclipse.hyades.internal.execution.remote.AgentControllerListener
	 */
	public void addAgentControllerListener(AgentControllerListener listener) {
		synchronized (serverListeners) {
			serverListeners.add(listener);
		}
	}

	/**
	 * Add a listener for monitor events.
	 * 
	 * @param listener
	 *            org.eclipse.hyades.internal.execution.remote.MonitorListener
	 */
	public void addMonitorListener(MonitorListener listener) {
		synchronized (monitorListeners) {
			monitorListeners.add(listener);
		}
	}

	/**
	 * Add a listener for client events.
	 * 
	 * @param listener -
	 *            org.eclipse.hyades.internal.execution.remote.RemoteClientListener
	 */
	public void addRemoteClientListener(RemoteClientListener listener) {
		synchronized (clientListeners) {
			clientListeners.add(listener);
		}

	}

	/**
	 * Add a listener for client events.
	 * 
	 * @param listener -
	 *            org.eclipse.hyades.internal.execution.remote.RemoteClientListener
	 */
	public void addCommandListener(CustomCommandHandler listener) {
		synchronized (commandListeners) {
			commandListeners.add(listener);
		}

	}

	/**
	 * Called by the native implementation to indicate that the agent is now
	 * registered with the Agent Controller.
	 */
	void agentControllerActive() {
		serviceAvailable = true;
		uuidsValid = true;

		synchronized (this.getClass()) {
			if (agentCount == 0) {
				this.getClass().notify();
			}
		}

		synchronized (serverListeners) {
			Enumeration e = serverListeners.elements();
			while (e.hasMoreElements()) {
				AgentControllerListener listener = (AgentControllerListener) e.nextElement();
				listener.agentControllerActive();
			}
		}
	}

	/**
	 * Called by the native implementation to indicate that the agent is no
	 * longer registered with the Agent Controller. At the same time the
	 * IRemoteExecutionComponent's default configuration is destroyed as it may
	 * be changed by the Agent Controller before the next time we re-register.
	 */
	void agentControllerInactive() {
		serviceAvailable = false;
		setLogging(false);
		_defaultConfiguration.clear();
		synchronized (serverListeners) {
			Enumeration e = serverListeners.elements();
			while (e.hasMoreElements()) {
				AgentControllerListener listener = (AgentControllerListener) e.nextElement();
				listener.agentControllerInactive();
			}
		}
	}

	/**
	 * Called by the native implementation to indicate that the agents remotely
	 * attached client has either detached from the agent or that the client has
	 * died.
	 */
	void remoteClientExited() {
		setLogging(false);
		synchronized (clientListeners) {
			Enumeration e = clientListeners.elements();
			while (e.hasMoreElements()) {
				RemoteClientListener listener = (RemoteClientListener) e.nextElement();
				listener.clientInactive();
			}
		}
	}

	/**
	 * Deregister from the Agent Controller.
	 */
	public synchronized void deregister() {
		if (this.isInitialized) {
			this.deregister0(this.name);
			this.isInitialized = false;
			/**
			 * Synchronize decrement of agent count
			 */
			synchronized (this.getClass()) {
				agentCount--;
			}
		}
	}

	/**
	 * Native delegate of IRemoteExecutionComponent.deregister()
	 */
	private native void deregister0(String name);

	/**
	 * If the instance is finalized it will be deregistered from the Agent
	 * Controller. Otherwise it is not deregistered until the JVM exits.
	 */
	public void finalize() {
		try {
			this.deregister0(name);
		} finally {
			try {
				super.finalize();
			} catch (Throwable t) {
			}
		}
	}

	/**
	 * Retrieve the name of the IRemoteExecutionComponent instance.
	 * 
	 * @return java.lang.String
	 */
	public String getName() {
		return name;
	}

	/**
	 * Retrieve the type of this IRemoteExecutionComponent instance.
	 * 
	 * @return java.lang.String
	 */
	public String getType() {
		return type;
	}

	// Bug 79816
	public long getPID() {
		if (nativesAvailable) {
			return getPID0();
		} else {
			return 0;
		}
	}

	private native long getPID0();

	/* Bug 77768 begins */
	/**
	 * Get the agent option value by name only
	 * 
	 * @param name
	 *            Name of the option
	 * @return Value of the option
	 */
	public String getOptionValue(String name) {
		return getOptionValue(name, null);
	}

	/**
	 * Get the agent option value by name and type
	 * 
	 * @param name
	 *            Name of the option
	 * @param type
	 *            Type of the option
	 * @return Value of the option
	 */
	public String getOptionValue(String name, String type) {
		AgentConfigurationEntry entry;
		Enumeration elements;
		Object obj;

		elements = _defaultConfiguration._entries.elements();

		/* Bug 105188 begins */
		if(!elements.hasMoreElements()) {
			synchronized(_optionLock) {
				try {
					_optionLock.wait(5000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			elements = _defaultConfiguration._entries.elements();
		}
		/* Bug 105188 ends */

		while (elements.hasMoreElements()) {
			obj = elements.nextElement();
			if (obj instanceof AgentConfigurationEntry) {
				entry = (AgentConfigurationEntry) obj;
				if ((name != null) && (type != null)) {
					if (entry.getName().equals(name) && entry.getType().equals(type)) {
						return entry.getValue();
					}
				} else if (name == null) { // ignore the name field
					if (entry.getType().equals(type)) {
						return entry.getValue();
					}
				} else if (type == null) { // ignore the type field
					if (entry.getName().equals(name)) {
						return entry.getValue();
					}
				}
			}
		}

		return null;
	}

	/* Bug 77768 ends */

	// Bug 79816
	public String generateUUID() {
		if (nativesAvailable) {
			return generateUUID0();
		} else {
			return null;
		}
	}

	private native String generateUUID0();

	/**
	 * incommingCommand method comment.
	 */
	/*
	 * public void handleMessage(IMessage message) {
	 * 
	 * 
	 * switch ((int) command.getTag()) { case (int) Constants.RA_CUSTOM_COMMAND :
	 * break; } if (handler != null) { handler.incommingCommand(null, command); } }
	 */

	/**
	 * Registers the IRemoteExecutionComponent with the Agent Controller. If the
	 * Agent Controller is not running it throws an
	 * AgentControllerUnavailableException but the registration is still pending
	 * and the AgentControllerListeners will be invoked when the Agent
	 * Controller becomes available.
	 */
	public synchronized void initialize() throws AgentControllerUnavailableException {
		/* If we could not load the .dll we need to throw an exception */
		if (!nativesAvailable) {
			throw new AgentControllerUnavailableException();
		}

		if (!isInitialized) {
			this.initializeEngine0(name, type);

			/*
			 * We will wait up to ten seconds before continuing if this is the
			 * first instance.
			 */
			synchronized (this.getClass()) {
				if (agentCount == 0) {
					try {
						this.getClass().wait(10000); // Bug 74367
					} catch (InterruptedException e) {
						/* Allow to continue */
					}
				}
				agentCount++;

			}
			isInitialized = true;
		}
		if (!serviceAvailable) {
			throw new AgentControllerUnavailableException();
		}
	}

	public synchronized void initializeFast() throws AgentControllerUnavailableException {
		/* If we could not load the .dll we need to throw an exception */
		if (!nativesAvailable) {
			throw new AgentControllerUnavailableException();
		}

		if (!isInitialized) {
			this.initializeEngine0(name, type);

			synchronized (this.getClass()) {
				agentCount++;
			}
			isInitialized = true;
		}
		if (!serviceAvailable) {
			throw new AgentControllerUnavailableException();
		}
	}

	/**
	 * Native implementation of IRemoteExecutionComponent.initializeEngine().
	 */
	private native void initializeEngine0(String name, String uuid) throws AgentControllerUnavailableException;

	/**
	 * Returns whether the agent controller is currently active.
	 */
	public boolean isAgentControllerActive() {
		return RemoteComponentSkeleton.serviceAvailable;
	}

	/**
	 * Returns whether or not this instance of the agent is currently being
	 * monitored.<br>
	 * ie. Whether the log messages are being collected.<br>
	 */
	public boolean isLogging() {
		return isLogging;
	}

	/**
	 * Log the specified data.
	 * 
	 * @param msg
	 * @param offset
	 * @param len
	 * @return The number of bytes successfully sent
	 */
	// Bug 74367
	public synchronized long logMessageReturn(byte[] msg, int offset, int len) {
		long byteSent = 0;

		if (isLogging) {
			byteSent = this.logMessage0(name, msg, offset, len);
		}

		return byteSent;
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param msg
	 * @param offset
	 * @param The
	 *            number of bytes successfully sent
	 */
	// Bug 74367
	public synchronized void logMessage(byte[] msg, int offset, int len) {
		logMessageReturn(msg, offset, len);
	}

	/**
	 * Log the specified message.
	 * 
	 * @param msg
	 * @return The number of bytes successfully sent.
	 */
	// Bug 74367
	public synchronized long logMessageReturn(String msg) {
		long byteSent = 0;

		if (isLogging) {
			byteSent = this.logMessage0(name, msg);
		}
		return byteSent;
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param msg
	 * @return
	 */
	// Bug 74367
	public synchronized void logMessage(String msg) {
		logMessageReturn(msg);
	}

	/**
	 * Native implementation of IRemoteExecutionComponent.logMessage()
	 * 
	 * @param msg
	 *            byte[]
	 * @param len
	 *            int
	 */
	// Bug 74367
	private native long logMessage0(String agentName, byte[] msg, int offset, int len);

	/**
	 * Native implemenation of IRemoteExecutionComponent.logMeggase()
	 * 
	 * @param msg
	 *            java.lang.String
	 */
	// Bug 74367
	private native long logMessage0(String agentName, String msg);

	/**
	 * The following method is used to only log error messages to an active
	 * agent.
	 * 
	 * @param message
	 * @param messageId
	 * @param severity
	 * @return Number of byte successfully sent
	 */
	// Bug 74367
	public synchronized long logErrMsgReturn(String message, String messageId, int severity) {
		return logErrorMessage(message, messageId, severity);
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param message
	 * @param messageId
	 * @param severity
	 */
	// Bug 74367
	public synchronized void logErrMsg(String message, String messageId, int severity) {
		logErrMsgReturn(message, messageId, severity);
	}

	/**
	 * Native implementation of logErrorMessage
	 */
	// Bug 74367
	private native long logErrorMessage(String message, String messageId, int severity);

	/**
	 * Log the UTF8 encoded message.
	 * 
	 * @param msg
	 * @return Number of bytes successfully sent.
	 */
	// Bug 74367
	public synchronized long logMessageUTF8Return(String msg) {
		long byteSent = 0;

		if (isLogging) {
			/*
			 * We will try and log to the correct API for handling UTF-8
			 * strings. With old .dll's this will not be resolvable and we
			 * should use the old version. The old version will only handle
			 * ASCII strings
			 */
			try {
				byteSent = this.logMessageUTF81(name, msg.getBytes("UTF-8"));//$NON-NLS-1$
			} catch (Throwable e) {
				byteSent = logMessageUTF80(name, msg);
			}
		}

		return byteSent;
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param msg
	 */
	// Bug 74367
	public synchronized void logMessageUTF8(String msg) {
		logMessageUTF8Return(msg);
	}

	/**
	 * @param msg
	 *            java.lang.String
	 */
	// Bug 74367
	private native long logMessageUTF81(String agentName, byte[] msg);

	/**
	 * This does not support UTF-8 encoding properly.
	 * 
	 * @deprecated
	 */
	// Bug 74367
	private native long logMessageUTF80(String agentName, String msg);

	/*
	 * public void setMessageHandler(IMessageHandler handler) { this.handler =
	 * handler; }
	 */

	/**
	 * Sends a message to the remotely attached client if it exists.
	 * 
	 * @param message
	 * @param contextId
	 */
	// Bug 74367
	public long sendMessageToAttachedClientReturn(String message, long contextId) {
		long byteSent = 0;

		if (nativesAvailable) {
			try {
				byte[] bytes = message.getBytes("UTF-8");//$NON-NLS-1$
				byteSent = sendMessageBinary0(bytes, 0, bytes.length, contextId);
			} catch (UnsupportedEncodingException ex) {
			}
		}

		return byteSent;
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param message
	 * @param contextId
	 */
	// Bug 74367
	public void sendMessageToAttachedClient(String message, long contextId) {
		sendMessageToAttachedClientReturn(message, contextId);
	}

	/**
	 * The native implementation of sendMessageToAttachedCleint.
	 */
	// Bug 74367
	native long sendMessage0(String message, long contextId);

	/**
	 * Sends a message to the remotely attached client if it exists.
	 * 
	 * @param message
	 * @param offset
	 * @param length
	 * @param contextId
	 * @return Number of bytes successfully sent.
	 */
	// Bug 74367
	public long sendMessageToAttachedClientReturn(byte[] message, int offset, int length, long contextId) {
		long byteSent = 0;

		if (nativesAvailable) {
			byteSent = sendMessageBinary0(message, offset, length, contextId);
		}

		return byteSent;
	}

	/**
	 * This is to support the original method which does not return the bytes
	 * sent.
	 * 
	 * @param message
	 * @param offset
	 * @param length
	 * @param contextId
	 */
	// Bug 74367
	public void sendMessageToAttachedClient(byte[] message, int offset, int length, long contextId) {
		sendMessageToAttachedClientReturn(message, offset, length, contextId);
	}

	/**
	 * The native implementation of sendMessageToAttachedCleint.
	 */
	// Bug 74367
	native long sendMessage0(byte[] message, int offset, int length, long contextId);

	/**
	 * The native implementation of sendMessage that sends the message as binary
	 * data.
	 */
	// Bug 74367
	native long sendMessageBinary0(byte[] message, int offset, int length, long contextId);

	/**
	 * This is called by the native code to turn logging "on" or "off". When
	 * turned "off" there is no stepping through the JNI boundary.
	 */
	private synchronized void setLogging(boolean log) {
		isLogging = log;
		synchronized (monitorListeners) {
			Enumeration e = monitorListeners.elements();
			while (e.hasMoreElements()) {
				MonitorListener listener = (MonitorListener) e.nextElement();
				if (isLogging) {
					listener.monitorActive();
				} else {
					listener.monitorInactive();
				}
			}
		}
		if (isLogging && requestForMonitor) {
			synchronized (this) {
				this.notify();
			}
		}

	}

	/**
	 * Invoked by the native layer to set the agent name to the default name if
	 * this instance was created with the default constructor.
	 * 
	 * @param name
	 *            java.lang.String
	 */
	private void setName(String name) {
		this.name = new String(name);
	}

	/**
	 * Invoked by the native layer to set the type to the default type if this
	 * instance was created with the default condtructor.
	 * 
	 * @param uuid
	 *            java.lang.String
	 */
	private void setType(String type) {
		this.type = new String(type);
	}

	/**
	 * Invoked by the native layer to provide the agent UUID.
	 */
	void setAgentUUID(String type) {
		agentUUID = type;
	}

	/**
	 * retrieve the agent's UUID as used by the Agent Controller.
	 */
	public String getAgentUUID() {
		return agentUUID;
	}

	/**
	 * Invoked by the native layer to provide the node UUID.
	 * 
	 */
	void setNodeUUID(String type) {
		nodeUUID = type;
	}

	/**
	 * Retrieve the node's UUID as used by the Agent Controller. Throws
	 * AgentNotRegisteredException if the IRemoteExecutionComponent has not
	 * registered with the Agent Controller yet.
	 */
	public String getNodeUUID() throws AgentNotRegisteredException {
		if (!uuidsValid) {
			throw new AgentNotRegisteredException();
		}
		return nodeUUID;
	}

	/**
	 * Invoked by the native layer to set the JVM's UUID.
	 */
	void setJVMUUID(String type) {
		jvmUUID = type;
	}

	/**
	 * Retrieve the JVM's UUID as used by the Agent Controller. Throws
	 * AgentNotRegisteredException if the IRemoteExecutionComponent has not
	 * registered with the Agent Controller yet.
	 */
	public String getJVMUUID() throws AgentNotRegisteredException {
		if (!uuidsValid) {
			throw new AgentNotRegisteredException();
		}
		return jvmUUID;
	}

	/**
	 * Request for this agent to be monitored by the same client that is
	 * monitoring another agent.
	 * 
	 * @param addr -
	 *            java.net.InetAddress
	 * @param agentUUID -
	 * @param timeout -
	 *            how long to wait for the client to start monitoring.
	 *            Milliseconds
	 */
	public void requestMonitorThroughPeer_p(InetAddress addr, long port, String agentUUID, long timeout) {
		if (!nativesAvailable) {
			return;
		}

		synchronized (this) {
			requestForMonitor = true;
			requestMonitorThroughPeer0p(addr.getAddress(), port, agentUUID);
			try {
				this.wait(timeout);
			} catch (InterruptedException e) {
			} finally {
				requestForMonitor = false;
			}
		}
	}

	/**
	 * Bug 77768 Request for this agent to be monitored by the same client that
	 * is monitoring another agent. Using the default port.
	 * 
	 * @param addr
	 * @param agentUUID
	 * @param timeout
	 */
	public void requestMonitorThroughPeer(InetAddress addr, String agentUUID, long timeout) {
		requestMonitorThroughPeer_p(addr, 10002, agentUUID, timeout);
	}

	/**
	 * @deprecated Bug 77768
	 * @param addr
	 * @param agentUUID
	 */
	native void requestMonitorThroughPeer0(byte[] addr, String agentUUID);

	/**
	 * Bug 77768
	 * 
	 * @param addr
	 * @param agentUUID
	 */
	private native void requestMonitorThroughPeer0p(byte[] addr, long port, String agentUUID);

	/**
	 * Request for this agent to be monitored by the same client that is
	 * monitoring another agent.
	 * 
	 * @param addr -
	 *            java.net.InetAddress
	 * @param timeout -
	 *            how long to wait for the client to start monitoring.
	 *            Milliseconds
	 */
	public void requestMonitorThroughPeer_p(InetAddress addr, long port, long pid, String agentName, long timeout) {
		if (!nativesAvailable) {
			return;
		}

		synchronized (this) {
			requestForMonitor = true;
			requestMonitorThroughPeer0p(addr.getAddress(), port, pid, agentName);
			try {
				this.wait(timeout);
			} catch (InterruptedException e) {
			} finally {
				requestForMonitor = false;
			}
		}
	}

	/**
	 * Bug 77768 Request for this agent to be monitored by the same client that
	 * is monitoring another agent. Using the default port.
	 * 
	 * @param addr
	 * @param pid
	 * @param agentName
	 * @param timeout
	 */
	public void requestMonitorThroughPeer(InetAddress addr, long pid, String agentName, long timeout) {
		requestMonitorThroughPeer_p(addr, 10002, pid, agentName, timeout);
	}

	/**
	 * @deprecated Bug 77768
	 * @param addr
	 * @param pid
	 * @param agentName
	 */
	native void requestMonitorThroughPeer0(byte[] addr, long pid, String agentName);

	/**
	 * Bug 77768
	 * 
	 * @param addr
	 * @param pid
	 * @param agentName
	 */
	native void requestMonitorThroughPeer0p(byte[] addr, long port, long pid, String agentName);

	/**
	 * Invoked by the native layer to indicate that the agent requested throug a
	 * requestMonitorThroughPeer was not able to be contacted. This will allow
	 * the invoking thread of requestMonitorThroughPeer to continue without
	 * waiting the entire timeout period.
	 */
	void peerUnreachable() {
		synchronized (this) {
			if (requestForMonitor) {
				this.notifyAll();
			}
		}
	}

	/**
	 * Return the default configuration for this IRemoteExecutionComponent.
	 * 
	 * @return org.eclipse.hyades.internal.execution.remote.AgentConfiguration
	 *         reference to the default configuration.
	 */
	public AgentConfiguration getDefaultConfiguration() {
		return _defaultConfiguration;
	}

	/**
	 * This method is called by the native code to add an entry to the default
	 * configuration for this IRemoteExecutionComponent.
	 * 
	 * @param entry
	 *            the new
	 *            org.eclipse.hyades.internal.execution.remote.AgentConfigurationEntry
	 *            to be added to the default configuration.
	 */
	void addEntryToDefaultConfiguration(AgentConfigurationEntry entry) {
		_defaultConfiguration.addEntry(entry);

		// Bug 141060 NPE thrown if tptp.debug property is not set in RemoteComponentSkeleton
		if((System.getProperty("tptp.debug") != null) && (System.getProperty("tptp.debug").equalsIgnoreCase("true"))) {//$NON-NLS-1$
			System.out.println(RemoteResourceBundle.RemoteComponentSkeleton_ADDING_ + entry.getName() + "/" + entry.getType() + "/" + entry.getValue());
		}

		/* Bug 105188 begins */
		synchronized(_optionLock) {
			_optionLock.notifyAll();
		}
		/* Bug 105188 ends */
	}

	public void broadcastMessage(String message, long contextId) {
		sendMessageToAttachedClient(message, contextId);
	}

	public void broadcastMessage(byte[] message, int offset, int length, long contextId) {
		sendMessageToAttachedClient(message, offset, length, contextId);
	}

	Object createObjectInContextOfClassLoader(String objectType) {
		Object result = null;
		try {
			Class clazz = Class.forName(objectType);
			result = clazz.newInstance();
		} catch (Throwable e) {
		}
		return result;
	}

	void incomingCommand(CommandElement command) {
		if (command instanceof CustomCommand) {
			synchronized (commandListeners) {
				Enumeration e = commandListeners.elements();
				while (e.hasMoreElements()) {
					CustomCommandHandler listener = (CustomCommandHandler) e.nextElement();
					listener.handleCommand((CustomCommand) command);
				}
			}
		}
	}

	/**
	 * Returns whether the agent registered successfully with Agent Controller
	 * 
	 * @return Returns isRegistered.
	 */
	public boolean isRegistered() {
		return isRegistered;
	}
}
