/********************************************************************** 
 * Copyright (c) 2006, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationClientAdapter.java,v 1.35 2010/01/18 14:44:43 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.adapters.ant;

import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Vector;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.Property;
import org.apache.tools.ant.taskdefs.condition.Condition;
import org.apache.tools.ant.types.EnumeratedAttribute;
import org.apache.tools.ant.types.FileList;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.FileList.FileName;
import org.apache.tools.ant.util.FileUtils;
import org.eclipse.hyades.automation.client.internal.resources.AutomationClientResourceBundle;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;

/**
 * <p>These are the two included ant task definitions, the first one, named
 * "Automation" allows for the automation of any automatable published service
 * and the second, named "Test" allows for the automation of test-specific
 * services, such as test execution (for example, executing a JUnit test suite
 * defined by TPTP).</p>
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul E. Slauenwhite
 * @version January 18, 2010
 * @since   October 5, 2005
 */
public class AutomationClientAdapter {

	/**
	 * This ant task is a general TPTP ant task used to request the execution of
	 * published automatable TPTP services. A TPTP automatable service is
	 * published via the implementation of specific automation-related plug-in
	 * extension points and must implement automation-related Java interfaces as
	 * well.
	 * 
	 * @author Scott E. Schneider
	 * 
	 */
	public static class Automation extends Task {

		/**
		 * Stores the Java automation client adapter used for access the
		 * automation client
		 */
		private org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter adapter;

		/**
		 * The automation command to execute
		 */
		private String command;

		/**
		 * The number of iterations that this automation service should execute
		 */
		private int iterations;

		/**
		 * The service's state to be transmitted to the server
		 */
		private Properties properties;

		/**
		 * Quiet property; used to suppress excessive chatter
		 */
		private boolean quiet;

		/**
		 * The automation service service, the name of the service to execute
		 */
		private String service;

		/**
		 * The synchronicity directive for this task instance, if determines if
		 * the executed service will block until the service is complete or
		 * return immediately after the service has been instructed to execute
		 * by the client adapters.
		 */
		AutomationClientAdapter.Synchronicity synchronicity;

		/**
		 * VM arguments for the instance of the Java VM used as the service host
		 */
		private String vmargs;

		/**
		 * Add the configured property to the properties
		 * 
		 * @param property
		 *            the property to add
		 */
		public void addConfiguredProperty(Property property) {
			this.properties
					.setProperty(property.getName(), property.getValue());
		}

		/**
		 * Add property to the list with the given name and value
		 * 
		 * @param name
		 *            name of the property
		 * @param value
		 *            value of the property
		 */
		public void addProperty(String name, boolean value) {
			this.addProperty(name, Boolean.toString(value));
		}

		/**
		 * Add the property with the given name and associated file value
		 * 
		 * @param name
		 *            the name of the property
		 * @param file
		 *            the value of the file, will be rendered to string
		 */
		public void addProperty(String name, File file) {
			if (file != null) {
				this.addProperty(name, file.toString());
			}
		}

		/**
		 * Add the given named value with the value as a path
		 * 
		 * @param name
		 *            the name of the value
		 * @param path
		 *            the value as a path, to be used for things like system
		 *            paths or classpaths, standard ant path
		 */
		public void addProperty(String name, Path path) {
			if (path != null) {
				this.addProperty(name, path.toString());
			}
		}

		/**
		 * Add the given named value property to the properties
		 * 
		 * @param name
		 *            the name of the value
		 * @param value
		 *            the value
		 */
		public void addProperty(String name, String value) {
			if (name != null && value != null) {
				Property property = new Property();
				property.setName(name);
				property.setValue(value);
				this.addConfiguredProperty(property);
			}
		}
		
		/**
		 * Add the given object to the properties hashmap
		 */
		public void putToProperties(String name, Object value) {
			properties.put(name, value);
		}

		/**
		 * Add the given object to the properties hashmap
		 */
		public Object getFromProperties(String name) {
			return properties.get(name);
		}
		
		
		/**
		 * @param suiteList
		 * @param filesets2
		 */
		public void appendFilesets(List resultList, Vector filesets) {
			appendFilesets(resultList, filesets, false);
		}

		/**
		 * @param suiteList
		 * @param filelists2
		 */
		public void appendFilelists(List resultList, Vector filelists) {
			appendFilelists(resultList, filelists, false);
		}

		/**
		 * @param absolutize 
		 * @param suiteList
		 * @param filesets2
		 */
		public void appendFilesets(List resultList, Vector filesets, boolean absolutize) {
			if (filesets != null) {
				Iterator it = filesets.iterator();
				while (it.hasNext()) {
		            FileSet fileset = (FileSet) it.next();
		            appendFileset(resultList, fileset, absolutize);
				}
			}
		}

		/**
		 * @param resultList
		 * @param fileset
		 * @param absolutize 
		 */
		private void appendFileset(List resultList, FileSet fileset, boolean absolutize) {
            DirectoryScanner scanner = fileset.getDirectoryScanner(getProject());
            File rootDir = fileset.getDir(getProject());
            String[] files = scanner.getIncludedFiles();
            appendFileArray(resultList, rootDir, files, absolutize);
		}

		/**
		 * @param absolutize 
		 * @param suiteList
		 * @param filelists2
		 */
		public void appendFilelists(List resultList, Vector filelists, boolean absolutize) {
			if (filelists != null) {
				Iterator it = filelists.iterator();
				while (it.hasNext()) {
		            FileList filelist = (FileList) it.next();
		            appendFilelist(resultList, filelist, absolutize);
				}
			}
		}

		/**
		 * @param resultList
		 * @param filelist
		 * @param absolutize 
		 */
		private void appendFilelist(List resultList, FileList filelist, boolean absolutize) {
			File rootDir = filelist.getDir(getProject());
			String[] files = filelist.getFiles(getProject());
			appendFileArray(resultList, rootDir, files, absolutize);			
		}

		/**
		 * @param resultList
		 * @param rootDir
		 * @param files
		 * @param absolutize 
		 */
		private void appendFileArray(List resultList, File rootDir, String[] files, boolean absolutize) {
			for (int i = 0; i < files.length; i++) {
				File theFile = FileUtils.getFileUtils()
						.resolveFile(rootDir, files[i]);
				if (theFile.exists()) {
					String file = null;
					if (absolutize)
						file = theFile.getAbsolutePath();
					else
						file = files[i];
					
					if (!quiet) {
						System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_ADDING_FILE_", theFile.getAbsolutePath()));  //$NON-NLS-1$
					}					
					resultList.add(file);
				}
				else
					System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_FILE_NOT_EXIST_", theFile.getAbsolutePath())); //$NON-NLS-1$
			}
		}
		
		
		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {
			try {
				if (this.command.equalsIgnoreCase("execute")) {//$NON-NLS-1$
					this.addProperty("vmargs", this.vmargs);//$NON-NLS-1$
					this.addProperty("quiet", this.quiet);//$NON-NLS-1$
					for (int i = 0; i < this.iterations; i++) {
						if (this.synchronicity == null) {

							// Let base client code determine default
							adapter.execute(this.service, this.properties);
						} else {

							// Use specified synchronicity
							adapter.execute(this.service, this.properties,
									this.synchronicity
											.retrieveServiceEquivalent());

						}
					}
				}
			} catch (Throwable t) {
				throw new BuildException(t);
			}
		}

		/**
		 * Returns project property for the given key name
		 * 
		 * @param name
		 *            the property name to retrieve the value of
		 * @return the value of the project property
		 */
		String getProjectProperty(String name) {
			return this.getProject().getProperty(name);
		}

		/**
		 * Retrieve the specified property's value
		 * 
		 * @param name
		 *            the name of the property to return the value of
		 * @return the property value
		 */
		public String getProperty(String name) {
			return this.properties.getProperty(name);
		}

		/**
		 * Returns the root value as defined by eclipse.home currently
		 * 
		 * @return
		 */
		private String getRoot() {
			String root = this.getProject().getProperty(
					AutomationClientAdapter.ROOT_VARIABLE);
			return (root != null ? root : ".");//$NON-NLS-1$
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			this.service = this.service == null ? this
					.getProjectProperty("tptp.automation.service")//$NON-NLS-1$
					: this.service;
			if (this.command == null) {
				String defaultCommand = this
						.getProjectProperty("tptp.automation.command");//$NON-NLS-1$
				this.command = defaultCommand != null ? defaultCommand
						: "execute"; //$NON-NLS-1$
			}
			if (this.iterations == 0) {
				String defaultIterations = this
						.getProjectProperty("tptp.automation.iterations");//$NON-NLS-1$
				this.iterations = defaultIterations != null ? Integer.valueOf(
						defaultIterations).intValue() : 1;
			}
			this.vmargs = this.vmargs == null ? this
					.getProjectProperty("tptp.automation.vmargs") : this.vmargs;//$NON-NLS-1$
			if (this.synchronicity == null) {
				String defaultSynchronicity = this
						.getProjectProperty("tptp.test.synchronicity");//$NON-NLS-1$
				this.synchronicity = defaultSynchronicity != null ? new Synchronicity(
						defaultSynchronicity)
						: null;
			}
			this.quiet = !this.quiet ? Boolean.valueOf(
					this.getProjectProperty("tptp.automation.quiet"))//$NON-NLS-1$
					.booleanValue() : this.quiet;
			if (properties == null) {
				this.properties = new Properties();
			}
			this.adapter = new org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter(
					this.getRoot());
		}

		/**
		 * Determine if file is absolute versus relative to project and
		 * normalize accordingly
		 * 
		 * @param file
		 *            the file as calculated by ant
		 * @return the normalized string to be passed into the service
		 */
		String normalizePath(File file) {

			// Retrieve path of suite
			String path = file.getPath();

			// If the suite is a valid file, use it
			if (!file.isFile()) {
				/*
				 * If not, assume it is project relative and strip off base dir
				 * prefix, this prefix is added automatically when type is File
				 * by ant.
				 */
				int prefix = this.getProject().getBaseDir().getPath().length();
				path = path.substring(prefix);
			}

			// Return the normalized path
			return path;

		}

		/**
		 * Set the automation command to use
		 * 
		 * @param command
		 *            the automation command, currently only execute is
		 *            supported
		 */
		public void setCommand(String command) {
			this.command = command;
		}

		/**
		 * The number of iterations to execute this command
		 * 
		 * @param iterations
		 *            the number of iterations from 1 to many
		 */
		public void setIterations(int iterations) {
			this.iterations = iterations;
		}

		/**
		 * Sets quiet mode on or off, quiet reduces chatter to the absolute
		 * minimum as defined by a particular service
		 * 
		 * @param quiet
		 *            used to turn quiet mode on or off (true or false)
		 */
		public void setQuiet(boolean quiet) {
			this.quiet = quiet;
		}

		/**
		 * Identifies the service to invoke
		 * 
		 * @param service
		 *            the service identifier, which matches the extension point
		 *            id in the implementor within Eclipse
		 */
		public void setService(String service) {
			this.service = service;
		}

		/**
		 * Set the synchronicity setting for this task instance
		 * 
		 * @param synchronicity
		 *            indicating blocking or non-blocking execution
		 */
		public void setSynchronicity(
				AutomationClientAdapter.Synchronicity synchronicity) {
			this.synchronicity = synchronicity;
		}

		/**
		 * Set the VM arguments for the instance of the VM that will host the
		 * service
		 * 
		 * @param vmArgs
		 *            VM arguments both standard and specific to the given VM in
		 *            use
		 */
		public void setVmArgs(String vmargs) {
			this.vmargs = vmargs;
		}

	}

	/**
	 * There are more specific TPTP ant tasks that provide domain-specific
	 * interfaces to the more general TPTP automatable services (for example
	 * this is the TPTP test ant task that enables a more strongly-typed
	 * execution of tests).
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Execution extends AutomationClientAdapter.Automation {

		/**
	     * Filesets describing the locations of the tests to be executed
	     */
	    private Vector filesets = new Vector();
	    
	    /**
	     * Fileslists specifying the locations of the tests to be executed
	     */
	    private Vector filelists = new Vector();

		/**
		 * The test execution classpath to use
		 */
		private Path classpath;

		/**
		 * Launch configuration to direct the execution
		 */
		private String configuration;

		/**
		 * Agent controller connection string
		 */
		private String connection;

		/**
		 * The deployment file to use that defines the deployment to use for
		 * this test execution
		 */
		private String deployment;

		/**
		 * Whether or not the results should be overwritten or not
		 */
		private boolean overwrite;

		/**
		 * The Eclipse project that the test definition is stored in
		 */
		private String project;

		/**
		 * The file to hold the results in
		 */
		private String results;

		/**
		 * The results property specifies the property to set with the return
		 * value of the results, results in a property used as both and in and
		 * out property in this service
		 */
		private String resultsProperty;

		/**
		 * The file that defines the test suite itself
		 */
		private String suite;

		/**
		 * Path to where the test workspace is stored
		 */
		private File workspace;

		private String resultsrefid;

		/**
		 * Creates a classpath
		 * 
		 * @return the path of the classpath
		 */
		public Path createClasspath() {
			Path classpath = new Path(this.getProject());
			this.classpath = classpath;
			return classpath;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {

			// Configure service properties
			this.addProperty("workspace", this.workspace);//$NON-NLS-1$
			this.addProperty("project", this.project);//$NON-NLS-1$
			this.addProperty("suite", this.suite);//$NON-NLS-1$
			this.addProperty("deployment", this.deployment);//$NON-NLS-1$
			this.addProperty("results", this.results);//$NON-NLS-1$
			this.addProperty("overwrite", this.overwrite);//$NON-NLS-1$
			this.addProperty("classpath", this.classpath.toString());//$NON-NLS-1$
			this.addProperty("configuration", this.configuration);//$NON-NLS-1$
			this.addProperty("connection", this.connection);//$NON-NLS-1$
			this.addProperty("resultsrefid", this.resultsrefid);//$NON-NLS-1$
			this.setCommand("execute");//$NON-NLS-1$
			
			// If filesets or flelists have been specified, copy the
			// files into the input array
			List suiteList = new ArrayList();
			this.appendFilelists(suiteList, filelists, false);
			this.appendFilesets(suiteList, filesets, false);
			if (!suiteList.isEmpty()) {
				if (suite != null && suite.length() > 0) {
					suiteList.add(suite);
				}
				putToProperties("suite", suiteList);//$NON-NLS-1$
			}
			
			if (this.getFromProperties("suite") == null && this.getFromProperties("configuration") == null) {//$NON-NLS-1$ //$NON-NLS-2$
				System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_NO_TEST_SPECIFIED_")); //$NON-NLS-1$
				return;
			}

			// Execute service (via automation base class)
			super.execute();

			List resultList = null;
			Object results = getFromProperties("results");//$NON-NLS-1$
			
			if (results instanceof List){
				resultList = ((List)(results));
			}
			else{
				
				resultList = new ArrayList();
				resultList.add(results);
			} 

			if ((resultsrefid != null) && (resultsrefid.length() > 0)) {

				String resultProjectDir = ((String)(getFromProperties("resultProjectDir")));//$NON-NLS-1$

				if (resultProjectDir == null) {
					resultProjectDir = getRootDirFromInputs();
				}

				getProject().addReference(resultsrefid, createFileListFromList(resultList, resultProjectDir));
			} 
			else if(resultList.size() > 1){
				System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_NO_PARAM_SPECIFIED_")); //$NON-NLS-1$
			}

			if((this.resultsProperty != null) && (resultList.size() == 1)){

				//Set the first result file path to the results property:
				this.getProject().setProperty(this.resultsProperty, ((String)(resultList.get(0))));
			}
		}

		private String getRootDirFromInputs() {
			String rootDir = null;
			if (!filesets.isEmpty()) {
				FileSet set = (FileSet) filesets.firstElement();
				rootDir = set.getDir(getProject()).getAbsolutePath();
			}
			else if (!filelists.isEmpty()) {
				FileList list = (FileList) filelists.firstElement();
				rootDir = list.getDir(getProject()).getAbsolutePath();
			}
			return rootDir;
		}

		/**
		 * @param list a list of files in the specified project 
		 * @param projectDir the directory of a project
		 * @return a file list rooted in the specified directory, containing 
		 * the files in the specified list 
		 */
		private FileList createFileListFromList(List list, String projectDir) {
			File projectFile = new File(projectDir);
			FileList newFilelist = new FileList();
			newFilelist.setDir(projectFile);
			
			Iterator it = list.iterator();
			while (it.hasNext()) {
				String path = (String) it.next();
				FileName name = new FileName();
				name.setName(path);
				newFilelist.addConfiguredFile(name);
			}
			return newFilelist;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			super.init();
			String defaultWorkspace = this
					.getProjectProperty("tptp.test.workspace");//$NON-NLS-1$
			this.workspace = defaultWorkspace != null ? this.getProject()
					.resolveFile(defaultWorkspace) : null;
			this.project = this.getProjectProperty("tptp.test.project");//$NON-NLS-1$
			this.suite = this.getProjectProperty("tptp.test.suite");//$NON-NLS-1$
			this.deployment = this.getProjectProperty("tptp.test.deployment");//$NON-NLS-1$
			this.results = this.getProjectProperty("tptp.test.results");//$NON-NLS-1$
			this.overwrite = Boolean.valueOf(
					this.getProjectProperty("tptp.test.overwrite"))//$NON-NLS-1$
					.booleanValue();
			this.classpath = new Path(this.getProject(), this.getProject()
					.getProperty("tptp.test.classpath"));//$NON-NLS-1$
			this.configuration = this
					.getProjectProperty("tptp.test.configuration");//$NON-NLS-1$
			this.connection = this.getProjectProperty("tptp.test.connection");//$NON-NLS-1$
			this.resultsProperty = this
					.getProjectProperty("tptp.test.results-property");//$NON-NLS-1$
			String defaultService = this
					.getProjectProperty("tptp.test.service");//$NON-NLS-1$
			this.setService(defaultService != null ? defaultService
					: "org.eclipse.hyades.test.tools.core.execute");//$NON-NLS-1$
		}
		
	    /**
	     * Check that this task has been configured properly.
	     * @throws BuildException if configuration errors are detected.
	     */
	    protected synchronized void checkConfiguration() throws BuildException {
	        if (suite == null && filesets.size() + filelists.size() == 0) {
	            throw new BuildException(AutomationClientResourceBundle.getString("AutomationClientAdapter_SPEICIFY_SOURCE_OF_TESTS_")); //$NON-NLS-1$
	        }
	    }		

	    /**
	     * Add a set of files to report on.
	     * @param set the <code>Fileset</code> to add.
	     */
	    public void addFileset(FileSet set) {
	        filesets.addElement(set);
	    }

	    /**
	     * Add a filelist to report on.
	     * @param list the <code>Filelist</code> to add.
	     */
	    public void addFilelist(FileList list) {
	        filelists.addElement(list);
	    }
		

		/**
		 * Path to the launch configuration that specifies this test launch
		 * 
		 * @param configuration
		 */
		public void setConfiguration(File configuration) {
			if (configuration != null) {
				this.configuration = this.normalizePath(configuration);
			}
		}

		/**
		 * Sets the agent controller connection string to use
		 * 
		 * @param connection
		 *            will be parsed for agent controller to use
		 */
		public void setConnection(String connection) {
			if (connection != null) {
				this.connection = connection;
			}
		}

		/**
		 * Provides the ability to set the deployment
		 * 
		 * @param deployment
		 *            the deployment file to use
		 */
		public void setDeployment(File deployment) {
			if (deployment != null) {
				this.deployment = this.normalizePath(deployment);
			}
		}

		/**
		 * Provides the ability to set the overwrite value for the execution
		 * results
		 * 
		 * @param overwrite
		 *            whether of not the execution results will be overwritten
		 */
		public void setOverwrite(boolean overwrite) {
			this.overwrite = overwrite;
		}

		/**
		 * Provides the ability to set the project associated with the test
		 * suite
		 * 
		 * @param project
		 *            the Eclipse project
		 */
		public void setProject(String project) {
			this.project = project;
		}

		/**
		 * Provides the ability to set the execution results file
		 * 
		 * @param results
		 *            the results of the execution
		 */
		public void setResults(String results) {
			this.results = results;
		}

		/**
		 * Provides the ability to set resultst property name, the results URI
		 * will be populated as a return value for consumption within the
		 * caller's Ant script
		 * 
		 * @param resultsProperty
		 *            The name of the results URI property to set upon service
		 *            completion
		 */
		public void setResultsProperty(String resultsProperty) {
			this.resultsProperty = resultsProperty;
		}

		/**
		 * Provides the ability to set the test suite
		 * 
		 * @param suite
		 *            the test suite definition file service, can be absolute or
		 *            relative to the basedir, if this isn't valid, then it is
		 *            assumed it is project relative
		 */
		public void setSuite(File suite) {
			if (suite != null) {
				this.suite = this.normalizePath(suite);
			}
		}

		/**
		 * Provides the ability to set the workspace value
		 * 
		 * @param workspace
		 *            the workspace to use, a fully-qualified path
		 */
		public void setWorkspace(File workspace) {
			this.workspace = workspace;
		}
		

		/**
		 * @param resultsrefid the resultsrefid to set
		 */
		public void setResultsrefid(String resultsrefid) {
			this.resultsrefid = resultsrefid;
		}

	}

	/**
	 * There are more specific TPTP ant tasks that provide domain-specific
	 * interfaces to the more general TPTP automatable services (for example
	 * this is the TPTP test ant task that enables a more strongly-typed URL
	 * content extraction operation)
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Extraction extends AutomationClientAdapter.Automation {

		/**
		 * The match regular expression used to retrieve data
		 */
		private String match;

		/**
		 * The source to perform extraction on
		 */
		private String uri;

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {
			this.addProperty("uri", this.uri);//$NON-NLS-1$
			this.addProperty("match", this.match);//$NON-NLS-1$
			this.setCommand("execute");//$NON-NLS-1$
			super.execute();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			super.init();
			this.uri = this.getProjectProperty("tptp.extraction.uri");//$NON-NLS-1$
			this.match = this.getProjectProperty("tptp.extraction.match");//$NON-NLS-1$
			String defaultService = this
					.getProjectProperty("tptp.extraction.service");//$NON-NLS-1$
			this.setService(defaultService != null ? defaultService
					: "org.eclipse.hyades.test.tools.core.extract");//$NON-NLS-1$
		}

		/**
		 * Sets the match expression to use for this extraction
		 * 
		 * @param match
		 *            regular expression that will extract the data
		 */
		public void setMatch(String match) {
			if (match != null) {
				this.match = match;
			}
		}

		/**
		 * Sets the URI to use as the extraction source
		 * 
		 * @param uri
		 *            the URI pointing to the input to use for extraction
		 */
		public void setURI(String uri) {
			if (uri != null) {
				this.uri = uri;
			}
		}

	}

	/**
	 * There are more specific TPTP ant tasks that provide domain-specific
	 * interfaces to the more general TPTP automatable services (for example
	 * this is the TPTP test ant task that enables a more strongly-typed
	 * interrogation of tests).
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Interrogation extends
			AutomationClientAdapter.Automation implements Condition {

	    /**
	     * Filesets describing the location of the results to be interrogated
	     */
	    private Vector filesets = new Vector();
	    
	    /**
	     * Fileslists specifying lists of result files for interrogated 
	     */
	    private Vector filelists = new Vector();
		
		/**
		 * The Eclipse project that the test definition is stored in
		 */
		private String project;

		/**
		 * The file to hold the results in
		 */
		private String results;

		/**
		 * Verdict property, set to verdict of interrogated test result
		 */
		private String verdictProperty;

		/**
		 * Path to where the test workspace is stored
		 */
		private File workspace;

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.taskdefs.condition.Condition#eval()
		 */
		public boolean eval() throws BuildException {

			/*
			 * The init method has to handle the case where the property has
			 * already been set, since condition eval does not call init for us
			 */
			this.init();
			this.execute();

			// Return true if the verdict is 'pass'
			return this.getProperty("verdict").equalsIgnoreCase("pass");//$NON-NLS-1$ //$NON-NLS-2$

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {

			// Set the properties for the service
			this.addProperty("workspace", this.workspace);//$NON-NLS-1$
			this.addProperty("project", this.project);//$NON-NLS-1$
			this.addProperty("results", this.results);//$NON-NLS-1$
			this.addProperty("verdict", "fail");//$NON-NLS-1$ //$NON-NLS-2$

			// If filesets or filelists have been specified, copy the
			// files into the input array
			List resultList = new ArrayList();
			this.appendFilelists(resultList, filelists, true);
			this.appendFilesets(resultList, filesets, true);
			if (!resultList.isEmpty()) {
				if (results != null && results.length() > 0) {
					resultList.add(results);
				}
				putToProperties("results", resultList);//$NON-NLS-1$
			}
			
			// Execute service (via automation base class)
			super.execute();

			// Set verdict property specified to verdict return value
			if (this.verdictProperty != null) {
				this.getProject().setProperty(this.verdictProperty,
						this.getProperty("verdict"));//$NON-NLS-1$
			}

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			super.init();
			if (this.workspace == null) {
				String defaultWorkspace = this
						.getProjectProperty("tptp.test.workspace");//$NON-NLS-1$
				this.workspace = defaultWorkspace != null ? this.getProject()
						.resolveFile(defaultWorkspace) : null;
			}
			this.project = this.project == null ? this
					.getProjectProperty("tptp.test.project") : this.project;//$NON-NLS-1$
			this.results = this.results == null ? this
					.getProjectProperty("tptp.test.results") : this.results;//$NON-NLS-1$
			this.verdictProperty = this.verdictProperty == null ? this
					.getProjectProperty("tptp.test.verdict-property")//$NON-NLS-1$
					: this.verdictProperty;
			String defaultService = this
					.getProjectProperty("tptp.interrogation.service");//$NON-NLS-1$
			this.setService(defaultService != null ? defaultService
					: "org.eclipse.hyades.test.tools.core.interrogate");//$NON-NLS-1$
		}

	    /**
	     * Add a set of files to report on.
	     * @param set the <code>Fileset</code> to add.
	     */
	    public void addFileset(FileSet set) {
	        filesets.addElement(set);
	    }

	    /**
	     * Add a filelist to report on.
	     * @param list the <code>Filelist</code> to add.
	     */
	    public void addFilelist(FileList list) {
	        filelists.addElement(list);
	    }
		

		
		
		/**
		 * Provides the ability to set the project associated with the test
		 * suite
		 * 
		 * @param project
		 *            the Eclipse project
		 */
		public void setProject(String project) {
			this.project = project;
		}

		/**
		 * Provides the ability to set the execution results file
		 * 
		 * @param results
		 *            the results of the execution
		 */
		public void setResults(String results) {
			this.results = results;
		}

		/**
		 * Provides the ability to set verdict property name, the verdict will
		 * be populated as a return value for consumption within the caller's
		 * Ant script
		 * 
		 * @param verdictProperty
		 *            The name of the verdict property to set upon service
		 *            completion
		 */
		public void setVerdictProperty(String verdictProperty) {
			this.verdictProperty = verdictProperty;
		}

		/**
		 * Provides the ability to set the workspace value
		 * 
		 * @param workspace
		 *            the workspace to use, a fully-qualified path
		 */
		public void setWorkspace(File workspace) {
			this.workspace = workspace;
		}

	}

	/**
	 * There are more specific TPTP ant tasks that provide domain-specific
	 * interfaces to the more general TPTP automatable services (for example
	 * this is the TPTP test ant task that enables a more strongly-typed
	 * publication of test results).
	 * 
	 * @author Joe Toomey
	 */
	public static class Publication extends AutomationClientAdapter.Automation {

	    /**
	     * Filesets describing the location of the results to be reported
	     */
	    private Vector filesets = new Vector();
	    
	    /**
	     * Fileslists specifying lists of result files for reporting 
	     */
	    private Vector filelists = new Vector();
		
		/**
		 * The Eclipse project that the test definition is stored in
		 */
		private String project;

		/**
		 * Report file name to use to store the published reports in
		 */
		private String report;
		
		/**
		 * Optional BIRT report template
		 */
		private String reportTemplateLocation;

		/**
		 * Results path for reporting publication
		 */
		private String results;

		/**
		 * Path to where the test workspace is stored
		 */
		private File workspace;
		
		/**
		 * Optional start date of report range
		 */
		private String startDateTime;
		
		/**
		 * Optional end date of report range
		 */
		private String endDateTime;
		
		/**
		 * Optional SimpleDateFormat pattern for startDateTime and endDateTime
		 */
		private String dateTimePattern;
		
		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {
			checkConfiguration();
			addProperty("workspace", this.workspace);//$NON-NLS-1$
			addProperty("project", this.project);//$NON-NLS-1$
			addProperty("report", this.report);//$NON-NLS-1$
			if (startDateTime != null)
				addProperty("startDate", parseDateTime(startDateTime));//$NON-NLS-1$
			
			if (endDateTime != null)
				addProperty("endDate", parseDateTime(endDateTime));//$NON-NLS-1$
			
			if (reportTemplateLocation != null)
				addProperty("reportTemplateLocation", reportTemplateLocation);//$NON-NLS-1$

			// If filesets or flelists have been specified, copy the
			// files into the input array
			List resultList = new ArrayList();
			appendFilelists(resultList, filelists, false);
			appendFilesets(resultList, filesets, false);
			putToProperties("results", resultList);//$NON-NLS-1$
			
			setCommand("execute");//$NON-NLS-1$
			super.execute();
		}

		/**
		 * @param startDateTime2
		 * @return
		 */
		private String parseDateTime(String dateTime) {

			DateFormat format;
			long time = -1;
			
			if (dateTimePattern == null)
				format = DateFormat.getDateTimeInstance(DateFormat.SHORT,
						DateFormat.SHORT, Locale.US);
			else
				format = new SimpleDateFormat(dateTimePattern);
			
			try {
				time = format.parse(dateTime).getTime();
			} catch (ParseException e) {
				System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_UNABLE_PARSE_DATETIME_", dateTime)); //$NON-NLS-1$
			}
			return Long.toString(time);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			
			super.init();
			
			String defaultWorkspace = this.getProjectProperty("tptp.test.workspace");//$NON-NLS-1$
			
			if(defaultWorkspace != null){
				this.workspace = this.getProject().resolveFile(defaultWorkspace);
			}
			else{
				this.workspace = null;
			}

			this.project = this.getProjectProperty("tptp.test.project");//$NON-NLS-1$
			this.report = this.getProjectProperty("tptp.test.report");//$NON-NLS-1$

			this.setService(this.getProjectProperty("tptp.publication.service"));//$NON-NLS-1$
		}

	    /**
	     * Check that this task has been configured properly.
	     * @throws BuildException if configuration errors are detected.
	     */
	    protected synchronized void checkConfiguration() throws BuildException {
	        if (results == null && filesets.size() + filelists.size() == 0) {
	            throw new BuildException(AutomationClientResourceBundle.getString("AutomationClientAdapter_SPEICIFY_SOURCE_OF_RESULTS_")); //$NON-NLS-1$
	        }
//	        if (results != null && filesets.size() + filelists.size() > 0) {
//	            throw new BuildException("For workspace result files, specify"
//                        + " only the <results> property (along with <workspace>"
//                        + " and <project>.  For filesystem"
//                        + " result files, use filelist and/or fileset.");
//	        }
	    }		
	    /**
	     * Add a set of files to report on.
	     * @param set the <code>Fileset</code> to add.
	     */
	    public void addFileset(FileSet set) {
	        filesets.addElement(set);
	    }

	    /**
	     * Add a filelist to report on.
	     * @param list the <code>Filelist</code> to add.
	     */
	    public void addFilelist(FileList list) {
	        filelists.addElement(list);
	    }
		
		/**
		 * Provides the ability to set the project associated with the test
		 * suite
		 * 
		 * @param project
		 *            the Eclipse project
		 */
		public void setProject(String project) {
			this.project = project;
		}

		/**
		 * Set the report file name to use for the published results
		 * 
		 * @param report
		 *            the report file name to use
		 */
		public void setReport(String report) {
			this.report = report;
		}

		/**
		 * Provides the ability to set the execution results file
		 * 
		 * @param results
		 *            the results of the execution
		 */
		public void setResults(String results) {
			this.results = results;
		}

		/**
		 * Provides the ability to set the workspace value
		 * 
		 * @param workspace
		 *            the workspace to use, a fully-qualified path
		 */
		public void setWorkspace(File workspace) {
			this.workspace = workspace;
		}

		/**
		 * @param endDate the endDate to set
		 */
		public void setEndDateTime(String endDate) {
			this.endDateTime = endDate;
		}

		/**
		 * @param filelists the filelists to set
		 */
		public void setFilelists(Vector filelists) {
			this.filelists = filelists;
		}

		/**
		 * @param filesets the filesets to set
		 */
		public void setFilesets(Vector filesets) {
			this.filesets = filesets;
		}

		/**
		 * @param startDate the startDate to set
		 */
		public void setStartDateTime(String startDate) {
			this.startDateTime = startDate;
		}

		/**
		 * @param dateTimePattern the dateTimePattern to set
		 */
		public void setDateTimePattern(String dateTimePattern) {
			this.dateTimePattern = dateTimePattern;
		}

		/**
		 * @param reportTemplateLocation the reportTemplateLocation to set
		 */
		public void setReportTemplateLocation(String reportTemplateLocation) {
			this.reportTemplateLocation = reportTemplateLocation;
		}

	}

	/**
	 * Enumerated attribute with the values "SYNCHRONOUS" and "ASYNCHRONOUS"
	 */
	public static class Synchronicity extends EnumeratedAttribute {

		/**
		 * Asynchronous value
		 */
		private static final String ASYNCHRONOUS = "asynchronous";//$NON-NLS-1$

		/**
		 * Synchronous value
		 */
		private static final String SYNCHRONOUS = "synchronous";//$NON-NLS-1$

		/**
		 * Public constructor, will be instantiated from ant
		 */
		public Synchronicity() {
		}

		/**
		 * Create a new synchronicity enumeration value with the given string
		 * representation
		 * 
		 * @param value
		 *            the string representing the enumeration value
		 */
		public Synchronicity(String value) {
			this.setValue(value);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.types.EnumeratedAttribute#getValues()
		 */
		public String[] getValues() {
			return new String[] { Synchronicity.SYNCHRONOUS,
					Synchronicity.ASYNCHRONOUS };
		}

		/**
		 * Indicates if value is set to asynchronous
		 * 
		 * @return true if asynchronous
		 */
		boolean isAsynchronous() {
			return this.isValue(Synchronicity.ASYNCHRONOUS);
		}

		/**
		 * Indicates if value is set to synchronous
		 * 
		 * @return true if synchronous
		 */
		boolean isSynchronous() {
			return this.isValue(Synchronicity.SYNCHRONOUS);
		}

		/**
		 * Helper that determines if the selected value is set to the specified
		 * value
		 * 
		 * @param value
		 *            the value to test against
		 * @return true if the value is not null and matches ignoring case
		 */
		private boolean isValue(String value) {
			return this.getValue() != null
					&& this.getValue().equalsIgnoreCase(value);
		}

		/**
		 * Returns back the service compatible synchronicity value given the ant
		 * side one
		 * 
		 * @return the equivalent standard synchronicity enumeration value
		 */
		private ProgressiveTask.Synchronicity retrieveServiceEquivalent() {
			return this.isSynchronous() ? ProgressiveTask.Synchronicity.SYNCHRONOUS
					: ProgressiveTask.Synchronicity.ASYNCHRONOUS;
		}

	}

	/**
	 * There are more specific TPTP ant tasks that provide domain-specific
	 * interfaces to the more general TPTP automatable services (for example
	 * this is the TPTP test ant task that enables a more strongly-typed
	 * verification of tests).
	 * 
	 * @author Scott E. Schneider
	 */
	public static class Verification extends AutomationClientAdapter.Automation
			implements Condition {

		/**
		 * Agent controller connection string
		 */
		private String connection;

		/**
		 * Verified property, set to status of agent controller (verified true
		 * means the agent controller is verified to be up and running)
		 */
		private String verifiedProperty;

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.taskdefs.condition.Condition#eval()
		 */
		public boolean eval() throws BuildException {

			/*
			 * The init method has to handle the case where the property has
			 * already been set, since condition eval does not call init for us
			 */
			this.init();
			this.execute();

			// Return true if verified is set to true
			return this.getProperty("verified").equalsIgnoreCase("true");//$NON-NLS-1$ //$NON-NLS-2$
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#execute()
		 */
		public void execute() throws BuildException {

			// Set properties and command for this execution
			this.addProperty("connection", this.connection);//$NON-NLS-1$
			this.addProperty("verified", "false");//$NON-NLS-1$ //$NON-NLS-2$
			this.setCommand("execute");//$NON-NLS-1$

			// Execute service (via automation base class)
			super.execute();

			// Set verdict property specified to verdict return value
			if (this.verifiedProperty != null) {
				this.getProject().setProperty(this.verifiedProperty,
						this.getProperty("verified"));//$NON-NLS-1$
			}

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.apache.tools.ant.Task#init()
		 */
		public void init() {
			super.init();
			this.connection = this.connection == null ? this
					.getProjectProperty("tptp.test.connection")//$NON-NLS-1$
					: this.connection;
			this.verifiedProperty = this.verifiedProperty == null ? this
					.getProjectProperty("tptp.test.verified-property")//$NON-NLS-1$
					: this.verifiedProperty;
			String defaultService = this
					.getProjectProperty("tptp.verification.service");//$NON-NLS-1$
			this.setService(defaultService != null ? defaultService
					: "org.eclipse.hyades.test.tools.core.verify");//$NON-NLS-1$
		}

		/**
		 * Sets the agent controller connection string to use
		 * 
		 * @param connection
		 *            will be parsed for agent controller to use
		 */
		public void setConnection(String connection) {
			if (connection != null) {
				this.connection = connection;
			}
		}

		/**
		 * Provides the ability to set verified property name, the verified
		 * status will be populated as a return value for consumption within the
		 * caller's Ant script
		 * 
		 * @param verifiedProperty
		 *            The name of the verified property to set upon service
		 *            completion
		 */
		public void setVerifiedProperty(String verifiedProperty) {
			this.verifiedProperty = verifiedProperty;
		}

	}
	
	/**
	 * @provisional New functionality added in 4.4.1 release, 
	 * still considered provisional API
	 * @generated
	 */
	public static class ImportExistingProjects extends AutomationClientAdapter.Automation {

		/**
		 * Path to the workspace 
		 * @generated
		 */
		private File workspace;

		/**
		 * Filesets for the directories collection
		 * @generated
		 */
		private Vector filesets = new Vector();
		
		/**
		 * Filelists for the directories collection
		 * @generated
		 */
		private Vector filelists = new Vector();

	  /**
	   * @generated
	   */
	  private String projectFile;
	  
		/**
		 * @generated NOT
		 */
		public void execute() throws BuildException {
	    addProperty("workspace", this.workspace);//$NON-NLS-1$
	  
	    // If filesets or filelists have been specified, copy the
	    // files into the input array
	    List projectFiles = new ArrayList();
	    appendFilelists(projectFiles, filelists, true);
	    appendFilesets(projectFiles, filesets, true);
	    putToProperties("projectFiles", projectFiles);//$NON-NLS-1$
	    
	    addProperty("projectFile", projectFile);//$NON-NLS-1$

	    setCommand("execute");//$NON-NLS-1$
	    super.execute();
	  }

		/**
		 * @generated
		 */
		public void init() {
	    super.init();
	    String defaultWorkspace = getProject().getProperty("tptp.test.workspace");//$NON-NLS-1$
	    this.workspace = defaultWorkspace != null ? this.getProject()
	        .resolveFile(defaultWorkspace) : null;
	    
	    String defaultService = getProject().getProperty("tptp.test.service");//$NON-NLS-1$
	    this.setService(defaultService != null ? defaultService
	        : "org.eclipse.hyades.test.tools.core.importExistingProjects");//$NON-NLS-1$
	  }

		/**
		 * @generated
		 */
		public ImportExistingProjects() {
	  }

		/**
		 * Add a set of files for the directories collection
		 * @param set the <code>Fileset</code> to add.
		 * @generated
		 */
		public void addFileset(FileSet set) {
	    filesets.addElement(set);
	  }

		/**
		 * Add a filelist for the directories collection
		 * @param list the <code>Filelist</code> to add.
		 * @generated
		 */
		public void addFilelist(FileList list) {
	    filelists.addElement(list);
	  }
	  
	  /**
	   * @generated
	   */
	  public String getProjectFile() {
	    return projectFile;
	  }

	  /**
	   * @generated
	   */
	  public void setProjectFile(String projectFile) {
	    this.projectFile = projectFile;
	  }
	  
	}
	

	/**
	 * The name of the variable that holds the eclipse home directory or root
	 * directory
	 */
	private static final String ROOT_VARIABLE = "tptp.automation.eclipse";//$NON-NLS-1$

}