/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: CommandLineArgumentParser.java,v 1.8 2008/04/28 15:45:40 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.adapters.shell;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Properties;

import org.eclipse.hyades.automation.client.internal.resources.AutomationClientResourceBundle;

/**
 * The command line parser class is used to parse the specified program
 * arguments via the command line interface. The parsed parameter values are set
 * in the properties state.
 * 
 * @author Duwayne J. Morris
 * @author Scott E. Schneider
 */
class CommandLineArgumentParser {

	/**
	 * Indicates if the config file is being processed in order to input
	 * additional arguments to the execution
	 */
	private boolean processingConfigFile;

	/**
	 * The path to the specified configuration file
	 */
	private String configfilepath;

	/**
	 * The message that accumulates and is then displayed to the user at an
	 * appropriate point
	 */
	private StringBuffer message;

	/**
	 * Parameter set booleans
	 */
	private boolean[] parameterSet;

	/**
	 * Configuration parameter set booleans
	 */
	private boolean[] configurationParameterSet;

	/**
	 * Platform line separator is stored
	 */
	private String lineSeparator;

	/**
	 * Constructs a command line argument parser for use by the automation
	 * client adapter
	 */
	public CommandLineArgumentParser() {
		super();
		this.parameterSet = new boolean[ICommandLineParameters.NUMPARMS];
		this.configurationParameterSet = new boolean[ICommandLineParameters.NUMPARMS];
		this.message = new StringBuffer();
		this.lineSeparator = System.getProperty("line.separator"); //$NON-NLS-1$
	}

	/**
	 * Check parameter for match and assign state to properties object.
	 * 
	 * @param properties
	 *            the properties to store this operation's state
	 * @param count
	 *            the number of parameters
	 * @param str
	 *            the input arguments
	 * @param strCompare
	 *            name of the potential parameter
	 * @param parmNumber
	 *            position of the potential parameter
	 * @return indicates if a match is found
	 */
	private boolean checkParameterForMatch(Properties properties, int count, String[] str, String strCompare,
			int parmNumber) {

		boolean bRet = false;

		if ((str[count].indexOf(strCompare) == 1) && (str[count].startsWith(ICommandLineParameters.CMD_DASH))
				&& (str[count].length() == strCompare.length() + ICommandLineParameters.CMD_DASH.length())) {
			bRet = true;
			if (parameterSet[parmNumber] == true && !processingConfigFile) {
				Object[] obj = new Object[1];
				obj[0] = str[count];
				message.append(AutomationClientResourceBundle.getString("ParseCmdLineArgs.1", obj));//$NON-NLS-1$
			} else if (configurationParameterSet[parmNumber] == true && processingConfigFile) {
				Object[] obj = new Object[1];
				obj[0] = str[count];
				message.append(AutomationClientResourceBundle.getString("ParseCmdLineArgs.3", obj));//$NON-NLS-1$
			} else if (!parameterSet[parmNumber]) {
				if (strCompare.compareTo(ICommandLineParameters.CMD_QUIET) == 0) {
					properties.setProperty(strCompare, "");//$NON-NLS-1$
				} else {
					if (count + 1 < str.length) {
						if (parmNumber == ICommandLineParameters.PARM_SUITE) {
							if (str[count + 1].indexOf(ICommandLineParameters.CMD_SUITE_EXT) < 0) {
								str[count + 1] = str[count + 1].concat(ICommandLineParameters.CMD_SUITE_EXT);
							}

						}
						properties.setProperty(strCompare, str[count + 1]);
					}
				}
			}
		}
		if (bRet == true) {
			parameterSet[parmNumber] = true;
			if (processingConfigFile)
				configurationParameterSet[parmNumber] = true;
		}
		return bRet;

	}

	/**
	 * Returns the last error, the state of the accumulated message instance
	 * variable
	 * 
	 * @return returns the last error as a string buffer
	 */
	public StringBuffer getLastError() {
		return message;
	}

	/**
	 * Top-level entry method to parse the given arguments into the named value
	 * pairs stored in the properties object.
	 * 
	 * @param properties
	 *            the state for this operation, the results of parsing the
	 *            arguments
	 * @param args
	 *            the arguments input via the command line interface
	 * @return indicates the parse was successful
	 */
	public boolean parse(Properties properties, String[] args) {

		int iCount = 0;
		if (args[0].compareTo(ICommandLineParameters.COMMENT) == 0) {
			iCount = 2;
		}
		if (args.length > 1 && args[1].compareTo(ICommandLineParameters.COMMENT) == 0) {
			iCount = 3;
		}

		for (; iCount < args.length; iCount++) {

			if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_ECLIPSEHOME,
					ICommandLineParameters.PARM_ECLIPSEHOME) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_WORKSPACE,
					ICommandLineParameters.PARM_WORKSPACE) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_PROJECT,
					ICommandLineParameters.PARM_PROJECT) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_SUITE,
					ICommandLineParameters.PARM_SUITE) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_RESULTS,
					ICommandLineParameters.PARM_RESULTS) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_OVERWRITE,
					ICommandLineParameters.PARM_OVERWRITE) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_CONFIGFILE,
					ICommandLineParameters.PARM_CONFIGFILE) == true)) {
				iCount++;
				if (iCount < args.length)
					configfilepath = args[iCount];
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_VMARGS,
					ICommandLineParameters.PARM_VMARGS) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_QUIET,
					ICommandLineParameters.PARM_QUIET) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_SERVICENAME,
					ICommandLineParameters.PARM_SERVICENAME) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_SERVICEARGS,
					ICommandLineParameters.PARM_SERVICEARGS) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_CONNECTION,
					ICommandLineParameters.PARM_CONNECTION) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_DEPLOYMENT,
					ICommandLineParameters.PARM_DEPLOYMENT) == true)) {
				iCount++;
			} else if ((checkParameterForMatch(properties, iCount, args, ICommandLineParameters.CMD_CONFIGURATION,
					ICommandLineParameters.PARM_CONFIGURATION) == true)) {
				iCount++;
			} else {
				properties.setProperty(args[iCount], args[iCount + 1]);
				iCount++;
			}
		}

		return true;

	}

	/**
	 * Parses the configuration file specified by a file path into the
	 * properties object specified.
	 * 
	 * @param properties
	 *            the properties to stored the parsed configuration file
	 * @return indicates the parse was successful
	 */
	public boolean parseConfigFile(Properties properties) {

		if (configfilepath != null) {
			File file;
			FileReader fileIn;
			LineNumberReader lineReader;
			ArrayList list = new ArrayList();
			processingConfigFile = true;
			try {
				file = new File(configfilepath);
				if (file.exists()) {
					String str;
					fileIn = new FileReader(file);
					lineReader = new LineNumberReader(fileIn);
					int iEnd;
					do {
						str = lineReader.readLine();

						if (str != null) {
							iEnd = str.indexOf('=');
							if (iEnd > 0) {
								String cmd = str.substring(0, iEnd);
								cmd = ICommandLineParameters.CMD_DASH + cmd.trim();
								String value = str.substring(iEnd + 1);
								value = value.trim();
								list.add(cmd);
								list.add(value);
							}
						}

					} while (str != null);
					String[] strArray = new String[list.size()];
					list.toArray(strArray);
					list.isEmpty();
					parse(properties, strArray);
				} else {
					Object[] obj = new Object[1];
					obj[0] = configfilepath;
					message
							.append(AutomationClientResourceBundle.getString("ParseCmdLineArgs.5", obj) + lineSeparator);//$NON-NLS-1$
				}
			} catch (IOException e) {

			}
		}
		processingConfigFile = false;
		return true;

	}

}
