/**********************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: 
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.automation.client.internal.resources;

import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.osgi.util.NLS;

/**
 * Automation client resource bundle.
 * <p/>
 * Message formatting is handled by either OSGi, ICU or Java, based on the following steps:
 * <p/>
 * <ol>
 * <li>If OSGi is available at run-time, message formatting is handled by {@link NLS#bind(String, Object[])}.</li>
 * <li>Otherwise, if ICU is available at run-time, message formatting is handled by {@link com.ibm.icu.text.MessageFormat#format(String, Object[])}.</li>
 * <li>Otherwise, message formatting is handled by {@link MessageFormat#format(String, Object[])}.</li>
 * </ol>
 * </p>
 * 
 * 
 * @author      Tony Wang
 * @author      Paul E. Slauenwhite
 * @version     January 20, 2009
 * @since       March 12, 2008
 */
public final class AutomationClientResourceBundle {
    
	/**
     * Static flag for quickly determining if the
     * ICU classes are available on the class path.
     * <p/>
     * By default, the ICU classes are assumed to 
     * be available on the class path.
     */
    private static boolean isICUAvailable = true;

	/**
     * Static flag for quickly determining if the
     * OSGi classes are available on the class path.
     * <p/>
     * By default, the OSGi classes are assumed to 
     * be available on the class path.
     */
    private static boolean isOSGiAvailable = true;

	/**
     * The resource bundle.
     */
    protected static ResourceBundle resourceBundle = null;

	/**
     * The name of the resource bundle.
     */
	private static final String BUNDLE_NAME = "org.eclipse.hyades.automation.client.internal.resources.messages";//$NON-NLS-1$

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * from the resource bundle.
     * <p/>
     * If the <code>key</code> does not exist in the resource bundle, the <code>key</code> 
     * is returned.
     * <p/>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p/>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */
    public static String getString(String key) {
            
    	if (resourceBundle == null ) {                

    		//Resolve the resource bundle:
    		try {
    			resourceBundle = ResourceBundle.getBundle(BUNDLE_NAME);
    		}
    		catch (MissingResourceException m) {
    			return key;
    		}
    	}       

    	try{
    		return (resourceBundle.getString(key.trim()).trim());    
    	}
    	catch(Exception e){
    		//Ignore and return the key.
    	}

        return key;
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameter from the resource bundle.
     * <p/>
	 * Message formatting is handled by either OSGi, ICU or Java, based on the following steps:
     * <p/>
	 * <ol>
	 * <li>If OSGi is available at run-time, message formatting is handled by {@link NLS#bind(String, Object[])}.</li>
	 * <li>Otherwise, if ICU is available at run-time, message formatting is handled by {@link com.ibm.icu.text.MessageFormat#format(String, Object[])}.</li>
	 * <li>Otherwise, message formatting is handled by {@link MessageFormat#format(String, Object[])}.</li>
	 * </ol>
     * <p/>
     * If the <code>key</code> does not exist in the resource bundle, the <code>key</code> is
     * returned.
     * <p/>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */
    public static String getString(String key, Object argument) {
        return (getString(key, new Object[] { argument}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the resource bundle.
     * <p/>
	 * Message formatting is handled by either OSGi, ICU or Java, based on the following steps:
     * <p/>
	 * <ol>
	 * <li>If OSGi is available at run-time, message formatting is handled by {@link NLS#bind(String, Object[])}.</li>
	 * <li>Otherwise, if ICU is available at run-time, message formatting is handled by {@link com.ibm.icu.text.MessageFormat#format(String, Object[])}.</li>
	 * <li>Otherwise, message formatting is handled by {@link MessageFormat#format(String, Object[])}.</li>
	 * </ol>
     * <p/>
     * If the <code>key</code> does not exist in the resource bundle, the <code>key</code> is
     * returned.
     * <p/>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */
    public static String getString(String key, Object argumentA, Object argumentB) {
        return (getString(key, new Object[] { argumentA, argumentB}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the resource bundle.
     * <p/>
	 * Message formatting is handled by either OSGi, ICU or Java, based on the following steps:
     * <p/>
	 * <ol>
	 * <li>If OSGi is available at run-time, message formatting is handled by {@link NLS#bind(String, Object[])}.</li>
	 * <li>Otherwise, if ICU is available at run-time, message formatting is handled by {@link com.ibm.icu.text.MessageFormat#format(String, Object[])}.</li>
	 * <li>Otherwise, message formatting is handled by {@link MessageFormat#format(String, Object[])}.</li>
	 * </ol>
     * <p/>
     * If the <code>key</code> does not exist in the resource bundle, the <code>key</code> is
     * returned.
     * <p/>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */    
    public static String getString(String key, Object[] arguments) {

    	try {
	    	
    		if(isOSGiAvailable){
		    	
	    		try {
		    		return (NLS.bind(getString(key), arguments));
		    	} 
		    	catch (NoClassDefFoundError n) {
		    		
		    		//OSGi is not available at run-time, so try ICU.
		    		isOSGiAvailable = false;
		    	}
	    	}
	
	    	if(isICUAvailable){
		    	
	    		try {
	        		return (com.ibm.icu.text.MessageFormat.format(getString(key), arguments));
	        	} 
	        	catch (NoClassDefFoundError n) {
	        		
	        		//OSGi is not available at run-time, so default to Java.
	        		isICUAvailable = false;
	        	}
	    	}
	
        	return (MessageFormat.format(getString(key), arguments));
		} 
		catch (Exception e) {
			return key;
		}
    }
}
