/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractService.java,v 1.20 2008/04/28 15:45:42 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.core;

import java.util.Properties;

/**
 * The abstract service class that all published services should extend from,
 * classes must extend from this class in order to hydrate the dehydrated
 * service state that is passed in to the automation server
 * 
 * @author Scott E. Schneider
 */

public abstract class AbstractService implements Service {

	/**
	 * The service memento class, used to save and then restore the service
	 * state from outside of Eclipse to within Eclipse
	 * 
	 * @author Scott E. Schneider
	 */
	private static class Memento implements Service.Memento {

		/**
		 * For serialization purposes
		 */
		private static final long serialVersionUID = 8724225538795702532L;

		/**
		 * The service identifier to command
		 */
		private String identifier;

		/**
		 * The service state stored and accessed as name value properties
		 */
		private Properties properties;

		/**
		 * Constructs a memento for this service's state given a service
		 * identifier
		 * 
		 * @param identifier
		 *            the automatable service to command
		 */
		private Memento(String identifier) {
			this.identifier = identifier;
			this.properties = new Properties();
		}

		/**
		 * Constructs a memento, only the abstract service class can create
		 * these, protects encapsulation
		 */
		private Memento(String identifier, Properties properties) {
			this.identifier = identifier;
			this.properties = properties;
		}

		/**
		 * Replace state with the memento given, using the same reference
		 * objects where appropriate and possible
		 * 
		 * @param memento
		 *            the source state that will replace the current memento
		 *            internal state
		 */
		private void replace(Service.Memento memento) {
			if (memento instanceof AbstractService.Memento) {
				AbstractService.Memento state = (AbstractService.Memento) memento;
				this.identifier = state.identifier;
				this.properties.clear();
				this.properties.putAll(state.properties);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			return this.identifier + "\r\n" + this.properties;//$NON-NLS-1$
		}

	}

	/**
	 * The service's persisting state
	 */
	private transient AbstractService.Memento state;

	/**
	 * The service is constructed (for use by the Eclipse platform extension
	 * registry's executable extension method)
	 */
	protected AbstractService() {
		// Constructed via executable extension on server-side
	}

	/**
	 * Constructs an abstract service given the specified state
	 * 
	 * @param state
	 *            the service's state
	 */
	protected AbstractService(Service.Memento state) {
		if (state instanceof AbstractService.Memento) {
			this.state = (AbstractService.Memento) state;
		}
	}

	/**
	 * The abstract service is constructed with the given service identifier
	 * 
	 * @param identifier
	 *            the service identifier
	 */
	protected AbstractService(String identifier) {
		this.state = new Memento(identifier);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#configure(java.util.Properties)
	 */
	public void configure(Properties properties) {
		this.state.properties = properties;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#configure(org.eclipse.hyades.automation.core.Service.Memento)
	 */
	public void configure(Service.Memento memento) {
		this.state.replace(memento);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#createMemento()
	 */
	public Service.Memento createMemento() {
		return new Memento(this.state.identifier, this.state.properties);
	}

	/**
	 * Returns the identifier for this service
	 * 
	 * @return the service identifier (typically the extension point id of the
	 *         implementor, identifies the specific service to be run on the
	 *         server)
	 */
	public String getIdentifier() {
		return this.state.identifier;
	}

	/**
	 * Retrives the service's persistent properties
	 * 
	 * @return the properties associated with this service
	 */
	protected Properties getProperties() {
		return this.state.properties;
	}
	
	/**
	 * Retrives the service's persistent properties
	 * needed access to properties in order to fix
	 * bugzilla 140638 without breaking anyon's code
	 * @return the properties associated with this service
	 */
	public Properties getPublicProperties() {
		return getProperties();
	}
	

	/**
	 * Helper method to retrieve property from state's properties
	 * 
	 * @param name
	 *            the name of the property to retrieve
	 * @return the property retrieved
	 */
	protected String getProperty(String name) {
		return this.state.properties.getProperty(name);
	}

	/**
	 * Helper method to retrieve property from state's properties
	 * 
	 * @param name
	 *            the name of the property to retrieve
	 * @defaultValue if name is not specified, use default value on return
	 * @return the property retrieved or default value if none
	 */
	protected String getProperty(String name, String defaultValue) {
		return this.state.properties.getProperty(name, defaultValue);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#getRoot()
	 */
	public String getRoot() {
		return ""; //$NON-NLS-1$
	}

	/**
	 * Indicates if this service instance is in quiet mode or not, quiet mode
	 * suppresses output/chatter to the console out.
	 * 
	 * @return false is output is not suppressed
	 */
	protected boolean isQuiet() {
		return Boolean.valueOf(this.getProperty("quiet", "false"))//$NON-NLS-1$ //$NON-NLS-2$
				.booleanValue();
	}

	/**
	 * Quiet conditional print implementation
	 * 
	 * @param text
	 *            the text to print to the standard out
	 */
	protected void print(String text) {
		if (!this.isQuiet()) {
			System.out.print(text);
		}
	}

	/**
	 * Quiet conditional println implementation
	 */
	protected void println() {
		if (!this.isQuiet()) {
			System.out.println();
		}
	}

	/**
	 * Quiet conditional println implementation
	 * 
	 * @param text
	 *            the text to println to the standard out
	 */
	protected void println(String text) {
		if (!this.isQuiet()) {
			System.out.println(text);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#setMemento(org.eclipse.hyades.automation.core.Service.Memento)
	 */
	public void setMemento(Service.Memento state) {
		if (state instanceof AbstractService.Memento) {
			this.state = (AbstractService.Memento) state;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return this.state != null ? this.state.toString() : super.toString();
	}

}