/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProgressiveTask.java,v 1.2 2008/03/20 18:49:59 dmorris Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.automation.core.utils;

import org.eclipse.core.runtime.IProgressMonitor;

/**
 * A progress task is a task that has progress monitor support. The runnable to
 * be executed by the task is executed on a separate thread and the task
 * execution can be done synchronously or asynchronously based on the task
 * invoker's preference. Progress tasks use the task thread pool and task
 * reusable thread classes; so any short-lived tasks can minimize any overhead
 * incurred by creating many short-lived threads. A tasks abstracts some of the
 * overhead involved with managing thread direction; a progressive task further
 * adds the coordination of progress monitor support.
 * 
 * The cancel poll inteval can be specified or the default can be used. At each
 * poll interval time, the cancel status of the progress monitor is checked and
 * if cancel has been triggered then the task will be exited and the reusable
 * thread used in the task will be interrupted.
 * 
 * Refer to the Test class in this package for a concrete example of usage, the
 * main steps are to first construct a progressive task and then to invoke the
 * execute method. The execute method will block in the synchronous case and
 * return without blocking in the asynchronous case.
 * 
 * @author Scott E. Schneider
 */
public class ProgressiveTask implements Runnable {

	/**
	 * A type-safe enumeration used to specify the synchronicity of the task
	 * when executed. For the execute method to block use SYNCHRONOUS, for the
	 * execute method to not block use ASYNCHRONOUS.
	 * 
	 * @author Scott E. Schneider
	 */
	public static final class Synchronicity {

		/**
		 * The task will be executed without blocking at the execute method
		 */
		public static final Synchronicity ASYNCHRONOUS = new Synchronicity("asynchronous");//$NON-NLS-1$

		/**
		 * The calling thread will be blocked when invoking the execute method
		 */
		public static final Synchronicity SYNCHRONOUS = new Synchronicity("synchronous");//$NON-NLS-1$

		/**
		 * Name of the enumeration value
		 */
		private final String name;

		/**
		 * Prevent outsider instantiation of this class using the type-safe
		 * enumeration idiom.
		 */
		private Synchronicity(String name) {
			this.name = name;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		public boolean equals(Object object) {
			return this.toString().equalsIgnoreCase(object.toString());
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#hashCode()
		 */
		public int hashCode() {
			return this.name.hashCode();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			return this.name;
		}

	}

	/**
	 * The default cancel poll interval is defined to be 3 seconds, which should
	 * be perceived as responsive enough.
	 */
	private static final int DEFAULT_CANCEL_POLL_INTERVAL = 3000;

	/**
	 * The default work units exhausted per cancel poll interval is 10 units.
	 * The progress monitor's work method will be called after each cancel poll
	 * interval.
	 */
	private static final int DEFAULT_WORK_UNITS_PER_POLL = 10;

	/**
	 * Establish a thread pool to use for all tasks, one thread pool is shared
	 * between all task instances.
	 */
	private static final ReusableThreadPool threadPool = new ReusableThreadPool();

	/**
	 * The cancel poll interval to use for this task. This is either the default
	 * cancel poll interval or specified as non-default when constructing the
	 * task.
	 */
	private final int cancelPollInterval;

	/**
	 * The name of the task, used when naming subservient threads owned by this
	 * task and potentially used for other purposes as well.
	 */
	private final String name;

	/**
	 * The progress monitor associated with the progressive task; used for
	 * checking the status of cancel and for reporting work completed by task.
	 */
	private final IProgressMonitor progressMonitor;

	/**
	 * The runnable that represents the real "work" to be performed and wrapped
	 * by the task notion.
	 */
	private final Runnable runnable;

	/**
	 * The task thread is used for asynchronous tasks that require the execute
	 * method's calling thread to be returned without participating in the
	 * actual work.
	 */
	private ReusableThread taskThread;

	/**
	 * The worker thread is the thread that actually runs the runnable contained
	 * within this task. It is a reusable thread and therefore can potentially
	 * be reused according to the strategy employed by the reusable thread
	 * factory.
	 */
	private ReusableThread workerThread;

	/**
	 * Constructs a progressive task given a task name, a runnable to be
	 * executed within the task and a progress monitor to use for cancel
	 * tracking and work reporting.
	 * 
	 * @param name
	 *            the name to be used for this task, used for subservient thread
	 *            naming
	 * @param runnable
	 *            the actual work to be performed within this task's execution
	 * @param monitor
	 *            the progress monitor to associate with this task
	 */
	public ProgressiveTask(String name, Runnable runnable, IProgressMonitor monitor) {
		this(name, runnable, monitor, ProgressiveTask.DEFAULT_CANCEL_POLL_INTERVAL);
	}

	/**
	 * Constructs a progressive task given a name, a runnable to be executed
	 * within the task, a progress monitor to use for cancel tracking and work
	 * reporting and a non-default cancel poll interval to use (overrides the
	 * default cancel poll interval)
	 * 
	 * @param name
	 * @param runnable
	 * @param monitor
	 * @param cancelPollInterval
	 */
	public ProgressiveTask(String name, Runnable runnable, IProgressMonitor monitor, int cancelPollInterval) {
		this.name = name;
		this.runnable = runnable;
		this.progressMonitor = monitor;
		this.cancelPollInterval = cancelPollInterval;
	}

	/**
	 * Executes the task using the synchronicity specified. This method returns
	 * when the task is complete if SYNCHRONOUS is used or returns immediately
	 * after initiating the task execution if ASYNCHRONOUS is specified.
	 * 
	 * @param synchronicity
	 *            the type-safe enumeration used to specify execution means
	 */
	public void execute(ProgressiveTask.Synchronicity synchronicity) {
		if (synchronicity == ProgressiveTask.Synchronicity.ASYNCHRONOUS) {
			this.taskThread = ProgressiveTask.threadPool.acquireThread();
			this.taskThread.attach(this);
			this.taskThread.start();
		} else {
			this.run();
		}
	}

	/**
	 * The name of this task, as specified in the construction of the task.
	 * 
	 * @return the name of the task
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Indicates whether this task has been canceled by the progress monitor
	 * cancel capability.
	 * 
	 * @return indicates if the task is canceled or not
	 */
	public boolean isCanceled() {
		return this.progressMonitor.isCanceled();
	}

	/**
	 * Indicates if the task is complete, meaning that the worker thread has
	 * completed executing the runnable wrapped by this task.
	 * 
	 * @return true if the task has completed
	 */
	public boolean isComplete() {
		return this.workerThread.isComplete();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		this.workerThread = ProgressiveTask.threadPool.acquireThread();
		this.workerThread.attach(this.runnable);
		this.workerThread.start();
		while (!this.isCanceled() && !this.workerThread.isComplete()) {
			try {
				synchronized (this.workerThread) {
					this.workerThread.wait(this.cancelPollInterval);
				}
				this.progressMonitor.worked(ProgressiveTask.DEFAULT_WORK_UNITS_PER_POLL);
			} catch (InterruptedException e) {
			}
		}
		ProgressiveTask.threadPool.releaseThread(this.workerThread);
		if (this.taskThread != null) {
			ProgressiveTask.threadPool.releaseThread(this.taskThread);
		}
	}

}