/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationClientAdapter.java,v 1.10 2008/04/28 15:45:41 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.adapters.java;

import java.util.List;
import java.util.Properties;

import org.eclipse.hyades.automation.client.AutomationClient;
import org.eclipse.hyades.automation.client.internal.resources.AutomationClientResourceBundle;
import org.eclipse.hyades.automation.client.strategies.AbstractExecutionStrategy;
import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;

/**
 * The Java automation client adapter provides a front-end to external Java code
 * that requires the automation of published services. This Java automation
 * client adapter is mainly a pass-through adapeter to the underlying launcher
 * lightweight automation component since the launcher is already exposed in
 * Java for client adapter use. It is still good practice to use this adapter
 * from the Java-side instead of using the automation launcher class directly.
 * 
 * @author Scott E. Schneider
 */
public class AutomationClientAdapter {

	/**
	 * Keep alive type-safe enumeration when keep alive value is expected
	 * 
	 * @author Scott E. Schneider
	 */
	public final static class KeepAlive {

		/**
		 * No keep alive is used and therefore all communication will go through
		 * other means besides such as file state transfer and Eclipse launch
		 * program arguments
		 */
		public static final KeepAlive DISABLED = new KeepAlive();

		/**
		 * Keep alive is used on-demand and expires after a reasonable timeout
		 * value has been exceeded (the timeout is measured from the exit of the
		 * last service invocation to the start of the next -- keep alive should
		 * increase performance since subsequent service invocations will not
		 * have to launch another Eclipse instance (does not make sense with
		 * strategies that do not launch an Eclipse instance, such as the
		 * in-process strategy)
		 */
		public static final KeepAlive ENABLED = new KeepAlive();

		/**
		 * Restrict instantiation per the type-safe enumeration idiom
		 */
		private KeepAlive() {
		};

	}

	/**
	 * The automation client to use to execute the automation command
	 */
	private AutomationClient client;

	/**
	 * Creates a Java automation client adapter for use by an already running
	 * Eclipse instance that prefers the service be serviced by the already
	 * running eclipse instance, used by plug-in code that prefers for the
	 * strategy to be in-process
	 */
	public AutomationClientAdapter() {
		this.client = AutomationClient.Factory.getInstance()
				.createAutomationClient(
						AbstractExecutionStrategy.Factory.getInstance()
								.createInProcessStrategy());
	}

	/**
	 * Constructs a Java automation client adapter for use by Java code to
	 * execute automation commands, uses the out of process strategy, use the
	 * default constructor for in process
	 * 
	 * @param root
	 *            the root of the eclipse to use (the eclipse home directory)
	 */
	public AutomationClientAdapter(String root) {
		this(root, AutomationClientAdapter.KeepAlive.DISABLED);
	}

	/**
	 * Constructs a Java automation client adapter for use by Java code to
	 * execute automation commands, uses the out of process strategy, use the
	 * default constructor for in process
	 * 
	 * @param root
	 *            the root of the eclipse to use (the eclipse home directory)
	 * @param keepAlive
	 *            setting keep-alive to true will use a variant out of process
	 *            strategy that keeps Eclipse alive between calls (within a
	 *            given timeout value) and thus saves on start up and state
	 *            transfer time (it uses remote method invocations instead of
	 *            writing to files to transfer state across process boundaries)
	 */
	public AutomationClientAdapter(String root,
			AutomationClientAdapter.KeepAlive keepAlive) {
		this.client = (keepAlive == AutomationClientAdapter.KeepAlive.ENABLED ? AutomationClient.Factory
				.getInstance().createAutomationClient(
						root,
						AbstractExecutionStrategy.Factory.getInstance()
								.createKeepAliveOutOfProcessStrategy())
				: AutomationClient.Factory.getInstance()
						.createAutomationClient(root));
	}

	/**
	 * Discover the published services ready for command by a client
	 * 
	 * @return the list of service identifiers available for command
	 */
	public List discover() {
		return client.discover();
	}

	/**
	 * Executes the service specified by the identifier with the given
	 * properties (executes synchronously, to run with a different synchronicity
	 * setting refer to the overloaded method that accepts a synchronicity
	 * enumeration value.
	 * 
	 * @param identifier
	 *            the service identifier
	 * @param properties
	 *            the properties that comprise the service state
	 * @return the return value from the service
	 * 
	 * @see #execute(String, Properties, ProgressiveTask.Synchronicity)
	 */
	public Object execute(String identifier, Properties properties) {
		if (!checkProjectProperty(properties))
			return null;
		return this.execute(identifier, properties,
				ProgressiveTask.Synchronicity.SYNCHRONOUS);
	}

	/**
	 * Executes the service specified by the identifier with the given
	 * properties, to execute services in parallel use the asynchonous
	 * synchronicity setting
	 * 
	 * @param identifier
	 *            the service identifier
	 * @param properties
	 *            the properties that comprise the service state
	 * @param synchronicity
	 *            the synchronicity setting to use when executing this service
	 * @return the return value from the service
	 */
	public Object execute(String identifier, Properties properties,
			ProgressiveTask.Synchronicity synchronicity) {
		
		Service service = client.request(identifier);
		service.configure(properties);
		Object value = client.execute(service, synchronicity);
		return value;
	}
	
	// in bugzilla 218246, the user has called in to execute a 
	// project with the path to the project rather than the project
	// name itself.  The eclipse API's called by Service.java are throwing
	// a modal dialog box, such as:
	
	//java.lang.RuntimeException: java.lang.IllegalArgumentException: Path for
	//project must have only one segment.
	// at
	//org.eclipse.hyades.automation.server.ServiceProxy.execute(ServiceProxy.java:172)
	
	//to fix this, look for any multiple segments in the project and return an 
	// error if a path is set as the project name
	
	private boolean checkProjectProperty(Properties properties){
		String proj = properties.getProperty("project"); //$NON-NLS-1$
		if (proj != null){
			if (proj.indexOf('\\') >= 0 || proj.indexOf('/') >= 0){
				System.out.println(AutomationClientResourceBundle.getString("AutomationClientAdapter_INVALID_PROJECT_", proj));  //$NON-NLS-1$
				return false;
			}
			
		}
		return true;
	}

}