/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractExecutionStrategy.java,v 1.11 2008/04/28 15:45:40 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client.strategies;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;

import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;

/**
 * The client execution strategy to use, this strategy specifies how the command
 * will be specifically transmitted to the automation server, examples of the
 * various strategies include in-process and out-of-process, out-of-process is
 * the default.
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractExecutionStrategy implements Service.Executable {

	/**
	 * The execution strategy factory, creates the requested strategy as well as
	 * providing a default
	 * 
	 * @author Scott E. Schneider
	 */
	public final static class Factory {

		/**
		 * The singleton factory to store
		 */
		private static final Factory factory;

		static {
			factory = new Factory();
		}

		/**
		 * Singleton static access point
		 * 
		 * @return the singleton instance of the factory
		 */
		public static final Factory getInstance() {
			return Factory.factory;
		}

		/**
		 * Restrict construction of singleton
		 */
		private Factory() {
		}

		/**
		 * Create the default strategy, this strategy will definitely work
		 * whereas other strategies might be unsupported
		 * 
		 * @return the default strategy
		 */
		public Service.Executable createDefaultStrategy() {
			return this.createOutOfProcessStrategy();
		}

		/**
		 * Create an in-process strategy, to launch the automation command
		 * within the same process as the requestor
		 * 
		 * @return the in-process strategy is returned
		 */
		public Service.Executable createInProcessStrategy() {
			return new InProcessStrategy();
		}

		/**
		 * Create an out-of-process keep-alive strategy, one that ensures an
		 * Eclipse instance is first started ready for remote invocations and
		 * then communicates through this channel until the connection times out
		 * 
		 * @return the out-of-process keep-alive strategy is returned
		 */
		public Service.Executable createKeepAliveOutOfProcessStrategy() {
			return new OutOfProcessStrategy.KeepAlive();
		}

		/**
		 * Create an out-of-process strategy, to launch the automation command
		 * within its own process
		 * 
		 * @return the out-of-process strategy is returned
		 */
		public Service.Executable createOutOfProcessStrategy() {
			return new OutOfProcessStrategy();
		}

		/**
		 * Creates a void strategy, basically does nothing but implements the
		 * service executable interface
		 * 
		 * @return the void strategy is returned, good for testing
		 */
		public Service.Executable createVoidStrategy() {
			return new VoidStrategy();
		}

	}

	/**
	 * The automation command identifier to be used as the command name on the
	 * command-line
	 */
	static final String AUTOMATION_COMMAND_IDENTIFIER = "tptp.automation.command";//$NON-NLS-1$

	/**
	 * The automation data identifier to be used as the command name on the
	 * command-line
	 */
	static final String AUTOMATION_DATA_IDENTIFIER = "tptp.automation.data";//$NON-NLS-1$

	/**
	 * The automation server identifier, specifies the command to run in Eclipse
	 * via the command-line
	 */
	static final String AUTOMATION_SERVER_IDENTIFIER = "org.eclipse.hyades.execution.server";//$NON-NLS-1$

	/**
	 * Used for transmitting the synchronicity setting for this particular
	 * service invocation
	 */
	static final String AUTOMATION_SYNCHRONICITY_DIRECTIVE = "tptp.automation.synchronicity";//$NON-NLS-1$

	/**
	 * Takes the process output and duplicates it to a local print stream, if
	 * this is not done, console out and error out will not be displayed and
	 * also the process might hang due to the streams not being directed
	 * anywhere.
	 * 
	 */
	static void redirectConsole(Process process, final InputStream inputStream,
			final PrintStream printStream) {
		new Thread("Console redirection from " + inputStream + " to " + printStream) //$NON-NLS-1$
		{
			public void run() {
				try {
					BufferedReader reader = new BufferedReader(
							new InputStreamReader(inputStream));
					String line;
					while ((line = reader.readLine()) != null) {
						printStream.println(line);
					}
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}.start();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service)
	 */
	public Object execute(Service service) {
		return this.execute(service, ProgressiveTask.Synchronicity.SYNCHRONOUS);
	}

}